// Screen.cpp: implementation of the CScreen class.
//
//////////////////////////////////////////////////////////////////////

#include "arch/frame/stdafx.h"
#include "arch/frame/Screen.h"
#include "arch/frame/mainfrm.h"
#include "arch/directx/dikeyboard.h"
#include "arch/directx/dimouse.h"

#include "fontdata.h"
#include "appleclock.h"
#include "memory.h"
#include "aipcdefs.h"

#include <windows.h>
#include <windowsx.h>

extern CDIKeyboard g_cDIKeyboard;
extern CDIMouse g_cDIMouse;

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

#define SAFE_RELEASE(a)	if (a){ delete a; a = NULL; }
#define BRIGHT24(c,b)	( (((BYTE)(((c>>16)&0xFF)*b))<<16)|(((BYTE)(((c>>8)&0xFF)*b)&0xFF)<<8)|((BYTE)((c&0xFF)*b)) )
#define RGB16(r,g,b)	( (((r)>>3)<<11)|(((g)>>2)<<5)|((b)>>3) )
#define RGB24(r,g,b)	( ( (r)<<16 ) | ( (g)<<8 ) | (b) )
#define RGB24_16(c24)	( (((c24)>>8)&0xF800)|(((c24)>>5)&0x7E0)|(((c24)>>3)&0x1F) )

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
BEGIN_MESSAGE_MAP(CScreen, CWnd)
//{{AFX_MSG_MAP(CScreen)
	ON_WM_CREATE()
	ON_WM_TIMER()
	ON_WM_ERASEBKGND()
	ON_WM_PAINT()
	ON_WM_LBUTTONUP()
	ON_WM_MOVE()
	//}}AFX_MSG_MAP
	ON_MESSAGE(WM_DISPLAYCHANGE, OnDisplayChange)
	ON_MESSAGE(UM_DISPLAY_CHANGE, OnDisplayChanged)
END_MESSAGE_MAP()

CScreen::CScreen()
{
	m_bPowerOn = FALSE;
	m_iBlinkCount = 0;
	m_nVideoMode = SM_COLOR;
	m_iFrameCount = 0;

	int i, j;
	
	for(i=0; i < 192; i++)
	{
		for(j=0; j < WIN_WIDTH; j++){
			m_pixelInfo[0][i][j]=0;
			m_pixelInfo[1][i][j]=0;
		}
	}
	m_pDisplay = new CDisplay();
//	m_pDisplay = NULL;
	m_pSurfaceDisk = NULL;
	m_pSurfaceMsg = NULL;

	m_nMsgVisiable = 0;
	m_bWindowed = TRUE;
//	m_bInitializing = FALSE;
	m_iScrMode = SS_TEXT;
	m_szMessage = "";
	m_hFont = ::CreateFont( 15, 0, 0, 0, 0, 0, 0, 0, ANSI_CHARSET,
							OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS,
							ANTIALIASED_QUALITY, FF_DONTCARE, "Arial Narrow" );
}

CScreen::~CScreen()
{
	SAFE_RELEASE(m_pDisplay);
	SAFE_RELEASE(m_pSurfaceDisk);
	SAFE_RELEASE(m_pSurfaceMsg);
}


void CScreen::Redraw()
{
	if( !m_bPowerOn )
		return;
	
	DDSURFACEDESC2 ddsd;
    RECT rect={0, 0, WIN_WIDTH, WIN_HEIGHT};

    ddsd.dwSize=sizeof(ddsd);
	BOOL bScreenDirty = FALSE;
	BOOL bFillLine = TRUE;
	int colorDepth;

	int y;
	
	HRESULT result;
	LPDIRECTDRAWSURFACE7 lpddsBack;

	CLockMgr<CCSWrapper> guard( m_Lock, TRUE );

	lpddsBack = m_pDisplay->GetBackBuffer();
	if ( lpddsBack == NULL )
		return;

    if FAILED( result=lpddsBack->Lock(&rect,&ddsd,DDLOCK_NOSYSLOCK|DDLOCK_WAIT,NULL) )
	{
        if( result == DDERR_SURFACELOST )
        {
			m_pDisplay->GetFrontBuffer()->Restore();
			lpddsBack->Restore();
			result=lpddsBack->Lock(&rect,&ddsd,DDLOCK_NOSYSLOCK|DDLOCK_WAIT,NULL);
			if FAILED( result )
				return;
			RedrawAll();		//  ٽ ׸ .
        }
	}


    char* pSurface = (char*)ddsd.lpSurface;
	BOOL bInvert = FALSE;
	colorDepth = m_iColorDepth >> 3;
	
	BYTE byteBlinkMask;
	
	if(((++m_iBlinkCount)&0x0F)==0){
		bInvert = TRUE;
	}
	
	if(m_iBlinkCount&0x10){
		byteBlinkMask = 0x40;
	}
	else
		byteBlinkMask = 0xC0;
	
	int mode = m_iScrMode & 0x07;
	
	if(!(m_iScrMode&SS_HIRES))
		mode |= SS_TEXT;

	BOOL b80COL = ( m_iScrMode & SS_80COL );
	BOOL bDHIRES = b80COL && ( m_iScrMode & SS_DHIRES );

	if ( m_iScrMode & SS_80STORE )		// display page 2 only if 80STORE is off
		mode &= ~SS_PAGE2;
	int page = mode & SS_PAGE2 ? 1 : 0;

	// for assembly
	_int64	bitmask64=0x0000000100000001,	// bitmask for two bytes
		zero64=0;                       // fill blank
	BYTE *pixelInfo;
	BYTE *addr;

	_int64* _fontData;
	BOOL bAltChar = m_iScrMode & SS_ALTCHAR;
	if ( bAltChar )
		_fontData = fontData2;
	else
		_fontData = fontData;

	unsigned long* appleColor;
	int lPitch = ddsd.lPitch;
	//	_int64 color;
	int color;
	
	//------------------
	if( m_nVideoMode == SM_COLOR )
	{
		color = 0;
		appleColor = m_crAppleColor;
	}
	else if ( m_nVideoMode == SM_COLOR2 )
	{
		color = 1;
		appleColor = m_crAppleColor2;
	}
	else if ( m_nVideoMode==SM_GREEN )
	{
		color = (int)m_iGreenColor;
		bFillLine = FALSE;
	}
	else
		color = (int)m_iWhiteColor;

	for( y = 0; y < 192; y++ )
	{
		int nextY=y;
		int* scanLine = m_aScanLine+y;
		
		if(m_aScanLine[y]!=1 &&
			!(m_aScanLine[y]==2 && (m_iScrMode&SS_TEXT||(m_iScrMode&SS_MIXED&&y>159)) && bInvert)){
			if(m_iScrMode&SS_TEXT || !(m_iScrMode&SS_HIRES) || (m_iScrMode&SS_MIXED && y>159)){
				y+=7;
			}
			continue;
		}
		bScreenDirty = TRUE;
		m_aScanLine[y] = 0;

		if ( b80COL )
			int b = 0;
		if(m_iScrMode&SS_TEXT || (m_iScrMode&SS_MIXED && y>159)){
			if(!color)
				color = (int)m_iWhiteColor;
			
			addr=m_apbScanAT[mode][y];
			pixelInfo = m_pixelInfo[page][y] + 2;
			//------------------------
			// draw Text - one line
			//------------------------
			// ebx - addr	: location of text memory
			// ecx - counter
			// dx  - color mask
			// mm0 - bitmask64
			// mm1 - fontData value
			// mm2 - zero64
			// mm3 - temporary for bitmask, and calcurate
			// esi - fontData offset
			// edi - pixelInfo which is target place
			//-------------------------
			if ( !b80COL )
			{
			__asm{
					emms					// MMX Code Start
					mov		dl, 0x11		// color info;
					mov		esi, _fontData

					mov		edi, pixelInfo		// address of pixel Info

					mov		ecx, 8
_BlankChar:
					push	ecx
					mov		ecx, 7
					xor		al, al
					rep stosb
					add		edi, WIN_WIDTH-7
					pop		ecx
					loop	_BlankChar
					sub		edi, WIN_WIDTH*5-7

					mov		ebx, addr		// memory location
					
					movq	mm2, zero64		// mm2 <- 0
					
					mov		ecx, 40			// 40 chars per one line
NextChar:
					movq	mm0, bitmask64     // mm0 <- bitmask64
					push	ecx
					mov		al, [ebx]		// text value
					mov		ah, al
					and		ah, 0xC0		// is blink character?
					cmp		ah, 0x40
					jnz		NotBlink
					cmp		bAltChar, 0
					jnz		NotBlink
					xor		al, byteBlinkMask
					push	eax
					mov		eax, scanLine
					mov		[eax], 2
					pop		eax
NotBlink:
					and		eax, 0xFF		// remove which is not al data
					shl		ax, 3			// font data offset
					movq	mm1, [esi+eax]	// mm1 <- fontData
					mov		ecx, 0x04		// 8/2 = 4 line
NextLine:
					push	ecx
					mov		ecx, 0x07		// Bit Shift Count
LineLoop:
					movq	mm3,mm0			// Bitmask Backup
					pand	mm0,mm1			// Check Bitmask
					pcmpeqd	mm0,mm3			// Packed Compare for Equal
					movd	eax, mm0		// if bit is high, ax is 0x1111 else 0x0000
					and		al, dl			// get color(0, 1, 2, 4, 8)
					mov		[edi+WIN_WIDTH*4], al	// write color info to pixel info array(lower 4 line) : WIN_WIDTH * 4
					rol		al, 1
					mov		[edi+WIN_WIDTH*4+1], al	// second pixel
					psrlq	mm0,32			// if Enhanced 3DNow! available, change to pswapd mm0,mm0
					movd	eax, mm0
					and		al, dl
					mov		[edi], al		// write color info to pixel info array(higher 4 line)
					rol		al, 1
					mov		[edi+1], al		// second pixel
					rol		dl, 2
					add		edi, 2			// jump 2 pixel 
					movq	mm0, mm3		// restore bit mask
					psllq	mm0, 1			// Shift Bitmask
					loop	LineLoop
					//LineLoop end
					pop		ecx
					psllq	mm0, 1			// Eat MSB
					ror		dl, 2			// return to start color
					sub		edi, WIN_WIDTH+14		// one line up : WIN_WIDTH + 14
					loop	NextLine
					//NextLine end
					pop		ecx
					rol		dl, 2			// continue to next color
					add		edi, WIN_WIDTH*4+14		// move one character from before start location : WIN_WIDTH * 4 + 14
					inc		ebx
					dec		ecx
					jz		EndNextChar
					jmp		NextChar
EndNextChar:
					//NextChar end
					emms
			}
			}
			else		// 80 column
			{
			__asm{
					emms					// MMX Code Start
					mov		dl, 0x22		// color info;
					
					mov		esi, _fontData
					mov		edi, pixelInfo		// address of pixel Info

					add		edi, WIN_WIDTH*3;

					mov		ecx, 8


					mov		ebx, addr		// memory location
					
					movq	mm2, zero64		// mm2 <- 0
					
					mov		ecx, 40			// 40 chars per one line
NextChar80:
					movq	mm0, bitmask64     // mm0 <- bitmask64
					push	ecx

					mov		al, [ebx+0x10000]		// text value
					mov		ah, al
					and		ah, 0xC0		// is blink character?
					cmp		ah, 0x40
					jnz		NotBlink80
					cmp		bAltChar, 0
					jnz		NotBlink80
					xor		al, byteBlinkMask
					push	eax
					mov		eax, scanLine
					mov		[eax], 2
					pop		eax
NotBlink80:
					and		eax, 0xFF		// remove which is not al data
					shl		ax, 3			// font data offset
					movq	mm1, [esi+eax]	// mm1 <- fontData
					mov		ecx, 0x04		// 8/2 = 4 line
NextLine80:
					push	ecx
					mov		ecx, 0x07		// Bit Shift Count
LineLoop80:
					movq	mm3,mm0			// Bitmask Backup
					pand	mm0,mm1			// Check Bitmask
					pcmpeqd	mm0,mm3			// Packed Compare for Equal
					movd	eax, mm0		// if bit is high, ax is 0x1111 else 0x0000
					and		al, dl			// get color(0, 1, 2, 4, 8)
					mov		[edi+WIN_WIDTH*4], al	// write color info to pixel info array(lower 4 line) : WIN_WIDTH * 4
					psrlq	mm0,32			// if Enhanced 3DNow! available, change to pswapd mm0,mm0
					movd	eax, mm0
					and		al, dl
					mov		[edi], al		// write color info to pixel info array(higher 4 line)
					rol		dl, 1
					add		edi, 1			// jump to next pixel
					movq	mm0, mm3		// restore bit mask
					psllq	mm0, 1			// Shift Bitmask
					loop	LineLoop80
					//LineLoop end
					pop		ecx
					psllq	mm0, 1			// Eat MSB
					rol		dl, 1			// return to start color
					sub		edi, WIN_WIDTH+7		// one line up : WIN_WIDTH + 7
					loop	NextLine80
					//NextLine end
					ror		dl, 1			// continue to next color
					add		edi, WIN_WIDTH*4+7		// move one character from before start location : WIN_WIDTH * 4 + 14


					// Next Character in Main Memory

					movq	mm0, bitmask64     // mm0 <- bitmask64

					mov		al, [ebx]		// text value
					mov		ah, al
					and		ah, 0xC0		// is blink character?
					cmp		ah, 0x40
					jnz		NotBlink80_M
					cmp		bAltChar, 0
					jnz		NotBlink80_M
					xor		al, byteBlinkMask
					push	eax
					mov		eax, scanLine
					mov		[eax], 2
					pop		eax
NotBlink80_M:
					and		eax, 0xFF		// remove which is not al data
					shl		ax, 3			// font data offset
					movq	mm1, [esi+eax]	// mm1 <- fontData
					mov		ecx, 0x04		// 8/2 = 4 line
NextLine80_M:
					push	ecx
					mov		ecx, 0x07		// Bit Shift Count
LineLoop80_M:
					movq	mm3,mm0			// Bitmask Backup
					pand	mm0,mm1			// Check Bitmask
					pcmpeqd	mm0,mm3			// Packed Compare for Equal
					movd	eax, mm0		// if bit is high, ax is 0x1111 else 0x0000
					and		al, dl			// get color(0, 1, 2, 4, 8)
					mov		[edi+WIN_WIDTH*4], al	// write color info to pixel info array(lower 4 line) : WIN_WIDTH * 4
					psrlq	mm0,32			// if Enhanced 3DNow! available, change to pswapd mm0,mm0
					movd	eax, mm0
					and		al, dl
					mov		[edi], al		// write color info to pixel info array(higher 4 line)
					rol		dl, 1
					add		edi, 1			// jump to next pixel
					movq	mm0, mm3		// restore bit mask
					psllq	mm0, 1			// Shift Bitmask
					loop	LineLoop80_M
					//LineLoop end
					pop		ecx
					psllq	mm0, 1			// Eat MSB
					rol		dl, 1			// return to start color
					sub		edi, WIN_WIDTH+7		// one line up : WIN_WIDTH + 7
					loop	NextLine80_M
					//NextLine end
					ror		dl, 1			// continue to next color
					add		edi, WIN_WIDTH*4+7		// move one character from before start location : WIN_WIDTH * 4 + 14


					inc		ebx

					pop		ecx
					dec		ecx
					jz		EndNextChar80
					jmp		NextChar80
EndNextChar80:
					sub		edi, WIN_WIDTH*3
					mov		ecx, 8
_BlankChar80:
					push	ecx
					mov		ecx, 7
					xor		al, al
					rep stosb
					add		edi, WIN_WIDTH-7
					pop		ecx
					loop	_BlankChar80
					//NextChar end
					emms
			}
			}
			nextY += 7;
		}
		// GR mode
		else if(!(m_iScrMode&SS_HIRES)){
			addr=m_apbScanAT[mode][y];
			pixelInfo = m_pixelInfo[page][y] + 2;
			
			//------------------------
			// draw GR mode - two line
			//------------------------
			//
			//-------------------------
			
			__asm{
					mov		edx, 0x1111		// color info;
					
					mov		edi, pixelInfo		// address of pixel Info

					mov		ecx, 8
_BlankChar_GR:
					push	ecx
					mov		ecx, 7
					xor		al, al
					rep stosb
					add		edi, WIN_WIDTH-7
					pop		ecx
					loop	_BlankChar_GR
					sub		edi, WIN_WIDTH*8-7

					mov		esi, addr		// memory location
					
					mov		ecx, 40			// 40 chars per one line
NextChar_GR:
					push	ecx
					mov		ecx, 14
					mov		ah, [esi]
NextPixel_GR:
					mov		al, ah
					and		al, dl
					mov		[edi], al
					mov		[edi+WIN_WIDTH], al
					mov		[edi+WIN_WIDTH*2], al
					mov		[edi+WIN_WIDTH*3], al
					shr		al, 4
					mov		[edi+WIN_WIDTH*4], al
					mov		[edi+WIN_WIDTH*5], al
					mov		[edi+WIN_WIDTH*6], al
					mov		[edi+WIN_WIDTH*7], al
					rol		dx, 1
					inc		edi
					loop	NextPixel_GR
					inc		esi
					pop		ecx
					loop	NextChar_GR
			}
			nextY += 7;
		}
		// hi-res mode
		else{
			if ( !bDHIRES )
			{
				addr = m_apbScanAT[mode][y];
				pixelInfo = m_pixelInfo[page][y] + 2;
				__asm{
					mov		edi, pixelInfo
					mov		ecx, 7
					xor		al, al
					rep stosb
					mov		esi, addr
					mov		dx, 0x1101			// high - color mask, low - bit test mask
					mov		ah, 0				// last pixel is zero
					mov		ecx, 40
NextByte_HIRES:
					mov		al, [esi]
					test	al, 0x80
					jz		NotHighColor_HIRES
					mov		[edi], ah			// if last pixel is exist, draw last pixel
					inc		edi					// increase pixelInfo address
					rol		dh, 1				// shift left color mask
NotHighColor_HIRES:
					mov		ah, al				// prepare pixel byte
					push	ecx
					mov		ecx, 6
					mov		al, ah
NextPixel_HIRES:
					test	al, dl
					setz	ah
					dec		ah					// if 'test al,dl' is not zero, ah is FF else 00
					and		ah, dh				// dh is color mask
					mov		[edi], ah
					inc		edi
					rol		ah, 1
					mov		[edi], ah
					inc		edi
					rol		dl, 1				// shift left test mask
					rol		dh, 2				// shift left color mask twice
					loop	NextPixel_HIRES
					// end of NextPixel_HIRES
					pop		ecx
					test	al, dl
					setz	ah
					dec		ah					// if 'test al,dl' is not zero, ah is FF else 00
					and		ah, dh				// dh is color mask
					mov		[edi], ah
					inc		edi
					rol		dl, 2				// shift left test mask, eat MSB
					rol		dh, 1				// shift left color mask
					rol		ah, 1				// if next byte is high color,
												// it will draw this last pixel
					test	al, 0x80
					jnz		HighColor_HIRES		// below is when it is not high color
					mov		[edi], ah			// if not high color, draw one more pixel
												// at first pixel
					inc		edi
					rol		ah, 1				// if next byte is high color,
					rol		dh, 1				// shift color mask
HighColor_HIRES:
					inc		esi
					loop	NextByte_HIRES
				}
			}
			else				// dhires
			{
				addr = m_apbScanAT[mode][y];

				pixelInfo = m_pixelInfo[page][y] + 2;
				__asm{
					mov		edi, pixelInfo
					mov		dx, 0x2201			// high - color mask, low - bit test mask
					mov		ah, 0				// last pixel is zero
					mov		ecx, 40
					mov		esi, addr
NextByte_DHIRES:
					push	ecx
					mov		ah, [esi+0x10000]			// aux memory
					mov		ecx, 7
NextPixel_DHIRES:
					test	ah, dl
					setz	al
					dec		al					// if 'test al,dl' is not zero, ah is FF else 00
					and		al, dh				// dh is color mask
					stosb
					rol		dh, 1
					rol		dl, 1				// shift left test mask
					loop	NextPixel_DHIRES
					rol		dl, 1				// eat MSB

					mov		ah, [esi]			// main memory
					mov		ecx, 7
NextPixel_DHIRES2:
					test	ah, dl
					setz	al
					dec		al					// if 'test al,dl' is not zero, ah is FF else 00
					and		al, dh				// dh is color mask
					stosb
					rol		dh, 1
					rol		dl, 1				// shift left test mask
					loop	NextPixel_DHIRES2
					rol		dl, 1				// eat MSB
					inc		esi
					// end of NextPixel_HIRE
					pop		ecx
					loop	NextByte_DHIRES
					mov		ecx, 7
					xor		al, al
					rep stosb
				}

			}
		}

		// draw pixelInfo to surface
		char* surface;
		if ( m_bWindowed )
			surface = pSurface+(y*2+4)*lPitch;
		else
			surface = pSurface+(y*2+(int)(FULL_HEIGHT-WIN_HEIGHT)/2)*lPitch
			+ (int)((FULL_WIDTH-WIN_WIDTH)*FULL_BPP/16 );
		pixelInfo = m_pixelInfo[page][y];
		
		__asm{
				mov		ebx, pixelInfo
				mov		esi, appleColor
				mov		edi, surface
				mov		ecx, y
NextLine1:
				push	ecx
				mov		ecx, 0
NextPixel1:
				mov		al, [ebx]			// get PixelInfo
				mov		edx, color
				cmp		edx, 0
				jz		_GetColorCode
				test	al, 0x0F			// lower four bit is not zero?
				jz		_DrawBlack
				cmp		edx, 1				// color mode 2 ?
				jnz		_DrawPixel
				jmp		_GetColorCode
_DrawBlack:
				mov		edx, 0
				jmp		_DrawPixel
				
_GetColorCode:
				cmp		ecx, 1
				jle		NotAddLeft
				or		al, [ebx-1]			// apply left pixel
				cmp		ecx, 2
				jle		NotAddLeft
				or		al, [ebx-2]			// apply left pixel
NotAddLeft:
				cmp		ecx, WIN_WIDTH-1
				jge		NotAddRight
				or		al, [ebx+1]			// apply right pixel
NotAddRight:
				and		eax, 0x0000000F
				shl		eax, 2				// eax*32
				mov		edx, [esi+eax]
//				mov		ax, dx
//				shl		edx, 16
//				mov		dx, ax

_DrawPixel:
				mov		eax, lPitch
				mov		[edi], edx
				cmp		bFillLine, 0
				jnz		_DrawNextLine
				mov		edx, 0
_DrawNextLine:
				mov		[edi+eax], edx			// nextLine
				inc		ebx
				add		edi, colorDepth			// depth in byte

				inc		ecx
				cmp		ecx, WIN_WIDTH
				jl		NextPixel1
				// NextPixel end
				mov		eax, WIN_WIDTH
				mov		ecx, colorDepth
				mul		ecx
				sub		edi, eax			// return to start position

				add		edi, lPitch			// next line( line += 2 )
				add		edi, lPitch			// next line( line += 2 )

				pop		ecx
				cmp		ecx, nextY
				jz		DrawEnd
				inc		ecx
				jmp		NextLine1
				// NextLine1 end
DrawEnd:
				mov		eax, nextY
				mov		y, eax
		}
	}
	
	lpddsBack->Unlock(NULL);

	if ( bScreenDirty )
	{
		if ( Present() == DDERR_SURFACELOST )
			Present();
		::Sleep(0);
	}

	m_iFrameCount++;


/*	
	if(result!=DD_OK){
		KillTimer(1);
		AfxMessageBox(DDErrorString(result));
		return;
	}
*/	
}


BOOL CScreen::InitDirectX()
{
	TRACE("Init start\n");
    LPDIRECTDRAWPALETTE pDDPal = NULL; 
    HRESULT	hr;

//	m_bInitializing = TRUE;		// It will cause WM_DISPLAY_CHANGE message,
								// but should not recreate direct draw
	CLockMgr<CCSWrapper> guard( m_Lock, TRUE );

	CMainFrame* pParent = (CMainFrame*)g_pBoard->m_lpwndMainFrame;

	SAFE_RELEASE( m_pSurfaceDisk );
	SAFE_RELEASE( m_pSurfaceMsg );
//	SAFE_RELEASE( m_pDisplay );
//	m_pDisplay = new CDisplay();
	if ( m_bWindowed )
	{
		pParent->ShowControlBar(&pParent->m_wndToolBar, TRUE, 0 );
		pParent->ShowControlBar(&pParent->m_wndStatusBar, TRUE, 0 );
		hr = m_pDisplay->CreateWindowedDisplay(m_hWnd, WIN_WIDTH, WIN_HEIGHT);
		if(hr!=DD_OK)
		{
			AfxMessageBox(DDErrorString(hr));
//			m_bInitializing = FALSE;
			return hr;
		}
		m_pDisplay->Clear();
		if ( !pParent->m_winRect.IsRectEmpty() )
		{
			::SetWindowPos( pParent->m_hWnd, NULL, m_stWindowPos.x, m_stWindowPos.y,
				m_stWindowPos.x + pParent->m_winRect.Width(), m_stWindowPos.y + pParent->m_winRect.Height(),
				SWP_NOZORDER | SWP_NOACTIVATE );
		}
		/*
		RECT rc, rcWork;
		::SystemParametersInfo( SPI_GETWORKAREA, 0, &rcWork, 0 );
		::GetWindowRect( pParent->m_hWnd, &rc );
		if( rc.left < rcWork.left ) rc.left = rcWork.left;
		if( rc.top  < rcWork.top )  rc.top  = rcWork.top;
		*/
	}
	else
	{
		RECT rc;
		::GetWindowRect( pParent->m_hWnd, &rc );
		m_stWindowPos.x = rc.left;
		m_stWindowPos.y = rc.top;
		pParent->ShowControlBar(&pParent->m_wndToolBar, FALSE, 0 );
		pParent->ShowControlBar(&pParent->m_wndStatusBar, FALSE, 0 );


		hr = m_pDisplay->CreateFullScreenDisplay( pParent->m_hWnd,
												FULL_WIDTH, FULL_HEIGHT, FULL_BPP );
		if(hr!=DD_OK)
		{
			AfxMessageBox(DDErrorString(hr));
			return hr;
		}

		// create disk status display surface and redraw
		m_pDisplay->CreateSurfaceFromBitmap( &m_pSurfaceDisk, MAKEINTRESOURCE(IDB_DISK),
			DISK_VIEW_WIDTH, DISK_VIEW_HEIGHT );
		// create message display surface.
		m_pDisplay->CreateSurface( &m_pSurfaceMsg, MSG_VIEW_WIDTH, MSG_VIEW_HEIGHT );
		m_pDisplay->Clear();
		m_pSurfaceDisk->Clear(COLORREF(0xFFFFFF));
		SetMessage("To return to windowed mode, press \"LEFT CTRL+LEFT ALT\" or \"CTRL+F9\" key.");
		pParent->m_wndStatusBar.SetDiskStatus( -1, 0 );	// redraw disk status

		::SetWindowPos( pParent->m_hWnd, NULL, 0, 0, FULL_WIDTH, FULL_HEIGHT,
			SWP_NOZORDER );

		g_cDIKeyboard.SetActive(TRUE);
		g_cDIMouse.SetActive(TRUE);
	}
	SetHSB(350, 205, 179);
	RedrawAll();
//	m_bInitializing = FALSE;
	TRACE("Init end\n");
	return TRUE;
	
}

char* CScreen::DDErrorString(HRESULT hr)
{
	switch (hr)
	{
	case DDERR_ALREADYINITIALIZED:           return "DDERR_ALREADYINITIALIZED";
	case DDERR_CANNOTATTACHSURFACE:          return "DDERR_CANNOTATTACHSURFACE";
	case DDERR_CANNOTDETACHSURFACE:          return "DDERR_CANNOTDETACHSURFACE";
	case DDERR_CURRENTLYNOTAVAIL:            return "DDERR_CURRENTLYNOTAVAIL";
	case DDERR_EXCEPTION:                    return "DDERR_EXCEPTION";
	case DDERR_GENERIC:                      return "DDERR_GENERIC";
	case DDERR_HEIGHTALIGN:                  return "DDERR_HEIGHTALIGN";
	case DDERR_INCOMPATIBLEPRIMARY:          return "DDERR_INCOMPATIBLEPRIMARY";
	case DDERR_INVALIDCAPS:                  return "DDERR_INVALIDCAPS";
	case DDERR_INVALIDCLIPLIST:              return "DDERR_INVALIDCLIPLIST";
	case DDERR_INVALIDMODE:                  return "DDERR_INVALIDMODE";
	case DDERR_INVALIDOBJECT:                return "DDERR_INVALIDOBJECT";
	case DDERR_INVALIDPARAMS:                return "DDERR_INVALIDPARAMS";
	case DDERR_INVALIDPIXELFORMAT:           return "DDERR_INVALIDPIXELFORMAT";
	case DDERR_INVALIDRECT:                  return "DDERR_INVALIDRECT";
	case DDERR_LOCKEDSURFACES:               return "DDERR_LOCKEDSURFACES";
	case DDERR_NO3D:                         return "DDERR_NO3D";
	case DDERR_NOALPHAHW:                    return "DDERR_NOALPHAHW";
	case DDERR_NOCLIPLIST:                   return "DDERR_NOCLIPLIST";
	case DDERR_NOCOLORCONVHW:                return "DDERR_NOCOLORCONVHW";
	case DDERR_NOCOOPERATIVELEVELSET:        return "DDERR_NOCOOPERATIVELEVELSET";
	case DDERR_NOCOLORKEY:                   return "DDERR_NOCOLORKEY";
	case DDERR_NOCOLORKEYHW:                 return "DDERR_NOCOLORKEYHW";
	case DDERR_NODIRECTDRAWSUPPORT:          return "DDERR_NODIRECTDRAWSUPPORT";
	case DDERR_NOEXCLUSIVEMODE:              return "DDERR_NOEXCLUSIVEMODE";
	case DDERR_NOFLIPHW:                     return "DDERR_NOFLIPHW";
	case DDERR_NOGDI:                        return "DDERR_NOGDI";
	case DDERR_NOMIRRORHW:                   return "DDERR_NOMIRRORHW";
	case DDERR_NOTFOUND:                     return "DDERR_NOTFOUND";
	case DDERR_NOOVERLAYHW:                  return "DDERR_NOOVERLAYHW";
	case DDERR_NORASTEROPHW:                 return "DDERR_NORASTEROPHW";
	case DDERR_NOROTATIONHW:                 return "DDERR_NOROTATIONHW";
	case DDERR_NOSTRETCHHW:                  return "DDERR_NOSTRETCHHW";
	case DDERR_NOT4BITCOLOR:                 return "DDERR_NOT4BITCOLOR";
	case DDERR_NOT4BITCOLORINDEX:            return "DDERR_NOT4BITCOLORINDEX";
	case DDERR_NOT8BITCOLOR:                 return "DDERR_NOT8BITCOLOR";
	case DDERR_NOTEXTUREHW:                  return "DDERR_NOTEXTUREHW";
	case DDERR_NOVSYNCHW:                    return "DDERR_NOVSYNCHW";
	case DDERR_NOZBUFFERHW:                  return "DDERR_NOZBUFFERHW";
	case DDERR_NOZOVERLAYHW:                 return "DDERR_NOZOVERLAYHW";
	case DDERR_OUTOFCAPS:                    return "DDERR_OUTOFCAPS";
	case DDERR_OUTOFMEMORY:                  return "DDERR_OUTOFMEMORY";
	case DDERR_OUTOFVIDEOMEMORY:             return "DDERR_OUTOFVIDEOMEMORY";
	case DDERR_OVERLAYCANTCLIP:              return "DDERR_OVERLAYCANTCLIP";
	case DDERR_OVERLAYCOLORKEYONLYONEACTIVE: return "DDERR_OVERLAYCOLORKEYONLYONEACTIVE";
	case DDERR_PALETTEBUSY:                  return "DDERR_PALETTEBUSY";
	case DDERR_COLORKEYNOTSET:               return "DDERR_COLORKEYNOTSET";
	case DDERR_SURFACEALREADYATTACHED:       return "DDERR_SURFACEALREADYATTACHED";
	case DDERR_SURFACEALREADYDEPENDENT:      return "DDERR_SURFACEALREADYDEPENDENT";
	case DDERR_SURFACEBUSY:                  return "DDERR_SURFACEBUSY";
	case DDERR_CANTLOCKSURFACE:              return "DDERR_CANTLOCKSURFACE";
	case DDERR_SURFACEISOBSCURED:            return "DDERR_SURFACEISOBSCURED";
	case DDERR_SURFACELOST:                  return "DDERR_SURFACELOST";
	case DDERR_SURFACENOTATTACHED:           return "DDERR_SURFACENOTATTACHED";
	case DDERR_TOOBIGHEIGHT:                 return "DDERR_TOOBIGHEIGHT";
	case DDERR_TOOBIGSIZE:                   return "DDERR_TOOBIGSIZE";
	case DDERR_TOOBIGWIDTH:                  return "DDERR_TOOBIGWIDTH";
	case DDERR_UNSUPPORTED:                  return "DDERR_UNSUPPORTED";
	case DDERR_UNSUPPORTEDFORMAT:            return "DDERR_UNSUPPORTEDFORMAT";
	case DDERR_UNSUPPORTEDMASK:              return "DDERR_UNSUPPORTEDMASK";
	case DDERR_VERTICALBLANKINPROGRESS:      return "DDERR_VERTICALBLANKINPROGRESS";
	case DDERR_WASSTILLDRAWING:              return "DDERR_WASSTILLDRAWING";
	case DDERR_XALIGN:                       return "DDERR_XALIGN";
	case DDERR_INVALIDDIRECTDRAWGUID:        return "DDERR_INVALIDDIRECTDRAWGUID";
	case DDERR_DIRECTDRAWALREADYCREATED:     return "DDERR_DIRECTDRAWALREADYCREATED";
	case DDERR_NODIRECTDRAWHW:               return "DDERR_NODIRECTDRAWHW";
	case DDERR_PRIMARYSURFACEALREADYEXISTS:  return "DDERR_PRIMARYSURFACEALREADYEXISTS";
	case DDERR_NOEMULATION:                  return "DDERR_NOEMULATION";
	case DDERR_REGIONTOOSMALL:               return "DDERR_REGIONTOOSMALL";
	case DDERR_CLIPPERISUSINGHWND:           return "DDERR_CLIPPERISUSINGHWND";
	case DDERR_NOCLIPPERATTACHED:            return "DDERR_NOCLIPPERATTACHED";
	case DDERR_NOHWND:                       return "DDERR_NOHWND";
	case DDERR_HWNDSUBCLASSED:               return "DDERR_HWNDSUBCLASSED";
	case DDERR_HWNDALREADYSET:               return "DDERR_HWNDALREADYSET";
	case DDERR_NOPALETTEATTACHED:            return "DDERR_NOPALETTEATTACHED";
	case DDERR_NOPALETTEHW:                  return "DDERR_NOPALETTEHW";
	case DDERR_BLTFASTCANTCLIP:              return "DDERR_BLTFASTCANTCLIP";
	case DDERR_NOBLTHW:                      return "DDERR_NOBLTHW";
	case DDERR_NODDROPSHW:                   return "DDERR_NODDROPSHW";
	case DDERR_OVERLAYNOTVISIBLE:            return "DDERR_OVERLAYNOTVISIBLE";
	case DDERR_NOOVERLAYDEST:                return "DDERR_NOOVERLAYDEST";
	case DDERR_INVALIDPOSITION:              return "DDERR_INVALIDPOSITION";
	case DDERR_NOTAOVERLAYSURFACE:           return "DDERR_NOTAOVERLAYSURFACE";
	case DDERR_EXCLUSIVEMODEALREADYSET:      return "DDERR_EXCLUSIVEMODEALREADYSET";
	case DDERR_NOTFLIPPABLE:                 return "DDERR_NOTFLIPPABLE";
	case DDERR_CANTDUPLICATE:                return "DDERR_CANTDUPLICATE";
	case DDERR_NOTLOCKED:                    return "DDERR_NOTLOCKED";
	case DDERR_CANTCREATEDC:                 return "DDERR_CANTCREATEDC";
	case DDERR_NODC:                         return "DDERR_NODC";
	case DDERR_WRONGMODE:                    return "DDERR_WRONGMODE";
	case DDERR_IMPLICITLYCREATED:            return "DDERR_IMPLICITLYCREATED";
	case DDERR_NOTPALETTIZED:                return "DDERR_NOTPALETTIZED";
	case DDERR_UNSUPPORTEDMODE:              return "DDERR_UNSUPPORTEDMODE";
	case DDERR_NOMIPMAPHW:                   return "DDERR_NOMIPMAPHW";
	case DDERR_INVALIDSURFACETYPE:           return "DDERR_INVALIDSURFACETYPE";
	case DDERR_DCALREADYCREATED:             return "DDERR_DCALREADYCREATED";
	case DDERR_CANTPAGELOCK:                 return "DDERR_CANTPAGELOCK";
	case DDERR_CANTPAGEUNLOCK:               return "DDERR_CANTPAGEUNLOCK";
	case DDERR_NOTPAGELOCKED:                return "DDERR_NOTPAGELOCKED";
	case DDERR_NOTINITIALIZED:               return "DDERR_NOTINITIALIZED";
	}
	return "Unknown Error";
}


int CScreen::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CWnd::OnCreate(lpCreateStruct) == -1)
		return -1;
	RECT rc;
	CMainFrame* pParent = g_pBoard->m_lpwndMainFrame;
	::GetWindowRect( pParent->m_hWnd, &rc );
	m_stWindowPos.x = rc.left;
	m_stWindowPos.y = rc.top;
	InitDirectX();	
	SetTimer(1, 3000, NULL);
	// TODO: Add your specialized creation code here
	return 0;
}

void CScreen::OnTimer(UINT nIDEvent) 
{
	// TODO: Add your message handler code here and/or call default
	// measure clock speed
	DWORD sec = GetTickCount();
	DWORD cpu_interval = sec - m_nTime;
	if ( cpu_interval > 3000 )
	{
		double frame = (double)( m_iFrameCount * 1000 ) / ( cpu_interval );
		m_iFrameCount = 0;
		m_nTime = sec;
		CMainFrame* pParent = g_pBoard->m_lpwndMainFrame;
		pParent->m_wndStatusBar.SetFrame( frame );
		pParent->m_wndStatusBar.SetSpeed( g_pBoard->m_dClockSpeed );
		switch( m_nMsgVisiable )
		{
		case 0:
			break;
		case 1:
			HideMessage();
			break;
		default:
			m_nMsgVisiable--;
			break;
		}
	}
	CWnd::OnTimer(nIDEvent);
}

void CScreen::OnDebug()
{
}

BOOL CScreen::OnEraseBkgnd(CDC* pDC) 
{
	// TODO: Add your message handler code here and/or call default
	return FALSE;
	//	return CWnd::OnEraseBkgnd(pDC);
}

void CScreen::OnPaint() 
{
	CPaintDC dc(this); // device context for painting
	
	// TODO: Add your message handler code here
//	int status = g_pBoard->GetAppleStatus();
	if ( Present() == DDERR_SURFACELOST )
		Present();
	// Do not call CWnd::OnPaint() for painting messages
}

HRESULT CScreen::Present()
{
	CLockMgr<CCSWrapper> guard( m_Lock, TRUE );

    HRESULT hr;

    if( NULL == m_pDisplay->GetFrontBuffer() && NULL == m_pDisplay->GetBackBuffer() )
        return E_POINTER;

    while( 1 )
    {
        if( m_pDisplay->IsWindowed() )
		{
			RECT rect;
			::GetClientRect(m_hWnd, &rect);
			POINT p1={rect.left, rect.top};
			POINT p2={rect.right, rect.bottom};
			::ClientToScreen(m_hWnd, &p1);
			::ClientToScreen(m_hWnd, &p2);
			rect.left = p1.x;
			rect.top = p1.y;
			rect.right = p2.x;
			rect.bottom = p2.y;
			
            hr = m_pDisplay->GetFrontBuffer()->Blt( &rect, m_pDisplay->GetBackBuffer(),
				NULL, DDBLT_WAIT, NULL );
		}
        else
		{
//			m_pDisplay->Blt( 0, 0, m_pSurfaceDisk, NULL );
			RECT rect = { 0, 0, FULL_WIDTH, FULL_HEIGHT };
            hr = m_pDisplay->GetFrontBuffer()->Blt( &rect, m_pDisplay->GetBackBuffer(),
				NULL, DDBLT_WAIT, NULL );

		}
		
        if( hr == DDERR_SURFACELOST )
        {
            m_pDisplay->GetFrontBuffer()->Restore();
            m_pDisplay->GetBackBuffer()->Restore();
        }
		
        if( hr != DDERR_WASSTILLDRAWING )
            return hr;
    }
}

BYTE CScreen::ChangeMode(WORD addr)
{
	switch(addr){
	case TXTCLR:
		m_iScrMode &= ~SS_TEXT;
		break;
	case TXTSET:
		m_iScrMode |= SS_TEXT;
		break;
	case MIXCLR:
		m_iScrMode &= ~SS_MIXED;
		break;
	case MIXSET:
		m_iScrMode |= SS_MIXED;
		break;
	case LOWSCR:
		m_iScrMode &= ~SS_PAGE2;
		break;
	case HISCR:
//		if ( !( m_iScrMode & SS_80COL ) )
		m_iScrMode |= SS_PAGE2;
		break;
	case LOWRES:
		m_iScrMode &= ~SS_HIRES;
		break;
	case HIRES:
		m_iScrMode |= SS_HIRES;
		break;
	case SETDHIRES:
		m_iScrMode |= SS_DHIRES;
		break;
	case CLRDHIRES:
		m_iScrMode &= ~SS_DHIRES;
		break;
	case SET80VID:
		m_iScrMode |= SS_80COL;
//		m_iScrMode &= ~SS_PAGE2;
		break;
	case CLR80VID:
		m_iScrMode &= ~SS_80COL;
		break;
	case SETALTCHAR:
		m_iScrMode |= SS_ALTCHAR;
		break;
	case CLRALTCHAR:
		m_iScrMode &= ~SS_ALTCHAR;
		break;
	}
	RedrawAll();
	return 0x00;
}

void CScreen::RedrawAll()
{
	CLockMgr<CCSWrapper> guard( m_Lock, TRUE );	
	for(int i=0; i<192; i++)
		m_aScanLine[i] = 1;
	if ( !g_pBoard->GetIsActive() || g_pBoard->GetIsSuspended() )
		Redraw();
}

void CScreen::setLookUp(BYTE *pMemory)
{
	int i,j;

	for(i=0;i<192;i++){
		m_awScanOT[i]=(i&7)*0x400 + ((i>>3)&7)*0x80 + (i>>6)*0x28;
		for(int j=0; j < 40; j++){
			m_aiScreenTable[m_awScanOT[i]+j][1] = i;
			m_aiScreenTable[m_awScanOT[i]+j][0] = j;
		}
	}
	
	for(i=0;i<192;i++){
		// page1, graphics = 0 
		m_apbScanAT[0][i]=m_awScanOT[i]+pMemory+0x2000;
		// page1, text
		// SS_TEXT, SS_TEXT|SS_MIXED
		m_apbScanAT[2][i]=m_apbScanAT[3][i]=m_awScanOT[i&~7]+pMemory+0x400;
		// page1, mixed
		// SS_MIXED
		m_apbScanAT[1][i]=m_apbScanAT[i>159?2:0][i];
		
		// page2, graphic
		// SS_PAGE2
		m_apbScanAT[4][i]=m_apbScanAT[0][i]+0x2000;
		// page2, text
		// SS_PAGE2|SS_TEXT, SS_PAGE2|SS_TEXT|SS_MIXED
		m_apbScanAT[6][i]=m_apbScanAT[7][i]=m_apbScanAT[2][i]+0x400;
		// page2, mixed
		// SS_PAGE2|SS_MIXED
		m_apbScanAT[5][i]=m_apbScanAT[i>159?6:4][i];
		
		m_aScanLine[i]=0;
	}
	
	for(i=0; i<3; i++)
	{
		for(j=0; j<40; j++)
			m_abPosTable[i*40+j] = (BYTE)(i<<6);
	}
	m_bPowerOn = TRUE;
}

int CScreen::GetMonitorType()
{
	return m_nVideoMode;
}

void CScreen::ChangeMonitorType()
{
	m_nVideoMode = (m_nVideoMode+1)&0x03;
	RedrawAll();
}

void CScreen::ChangeMonitorType( int type )
{
	m_nVideoMode = type & 0x03;
	RedrawAll();
}

unsigned int CScreen::HSB2RGB(int hue, BYTE saturation, BYTE bright)
{
	hue = hue*192/360;
	BYTE r, g, b;
	// c1 - bright color(for bright)
	// c2 - middle color(for hue)
	// c3 - dark color(for saturation)
	BYTE *c1, *c2, *c3;
	switch(hue>>5){
	case 0:
		c1 = &r;
		c2 = &g;
		c3 = &b;
		hue &= 0x1F;
		break;
	case 1:
		c1 = &g;
		c2 = &r;
		c3 = &b;
		hue &= 0x1F;
		hue = 0x20-hue;
		break;
	case 2:
		c1 = &g;
		c2 = &b;
		c3 = &r;
		hue &= 0x1F;
		break;
	case 3:
		c1 = &b;
		c2 = &g;
		c3 = &r;
		hue &= 0x1F;
		hue = 0x20-hue;
		break;
	case 4:
		c1 = &b;
		c2 = &r;
		c3 = &g;
		hue &= 0x1F;
		break;
	case 5:
	default:
		c1 = &r;
		c2 = &b;
		c3 = &g;
		hue &= 0x1F;
		hue = 0x20-hue;
	}
	*c1 = bright;
	*c3 = ((*c1*((~saturation)&0xFF))/255);
	*c2 = (hue*(*c1-*c3)/0x20+*c3);
	return (r<<16)|(g<<8)|b;
}

unsigned int CScreen::HSB2RGB(unsigned int hsb)
{
	BYTE h = (hsb>>16)*360/192;
	BYTE s = (hsb>>8)&0xFF;
	BYTE b = hsb&0xFF;
	return HSB2RGB(h, s, b);
}

unsigned int CScreen::ColorCompose(unsigned int c1, unsigned int c2)
{
	BYTE r1 = (c1>>16)&0xFF;
	BYTE g1 = (c1>>8)&0xFF;
	BYTE b1 = c1&0xFF;
	BYTE r2 = (c2>>16)&0xFF;
	BYTE g2 = (c2>>8)&0xFF;
	BYTE b2 = c2&0xFF;
	BYTE r3 = r1+(((256-r1)*r2)/256);
	BYTE g3 = g1+(((256-g1)*g2)/256);
	BYTE b3 = b1+(((256-b1)*b2)/256);
	return (r3<<16)|(g3<<8)|(b3);
}

void CScreen::SetHSB(int hue, BYTE saturation, BYTE bright)
{
	int i;
	unsigned int color[16];
	unsigned int color2[16];

	CLockMgr<CCSWrapper> guard( m_Lock, TRUE );	

	m_nHSB = (hue<<16) | (saturation<<8) | bright;

	color[0] = color2[0] = 0;
	color[8] = HSB2RGB((hue+90)%360, saturation, bright);
	color[4] = HSB2RGB((hue+180)%360, saturation, bright);
	color[2] = HSB2RGB((hue+270)%360, saturation, bright);
	color[1] = HSB2RGB(hue%360, saturation, bright);
	color[3] = ColorCompose(color[1], color[2]);
	color[5] = ColorCompose(color[1], color[4]);
	color[6] = ColorCompose(color[2], color[4]);
	color[7] = ColorCompose(color[1], color[6]);
	color[9] = ColorCompose(color[1], color[8]);
	color[10] = ColorCompose(color[2], color[8]);
	color[11] = ColorCompose(color[1], color[10]);
	color[12] = ColorCompose(color[4], color[8]);
	color[13] = ColorCompose(color[1], color[12]);
	color[14] = ColorCompose(color[2], color[12]);
	color[15] = color2[15] = ColorCompose(color[3], color[12]);
	color2[1] = BRIGHT24( color[1], 0.4 );
	color2[2] = BRIGHT24( color[2], 0.4 );
	color2[3] = BRIGHT24( color[3], 0.7 );
	color2[4] = BRIGHT24( color[4], 0.4 );
	color2[5] = BRIGHT24( color[5], 0.7 );
	color2[6] = BRIGHT24( color[6], 0.7 );
	color2[7] = BRIGHT24( color[7], 0.9 );
	color2[8] = BRIGHT24( color[8], 0.4 );
	color2[9] = BRIGHT24( color[9], 0.7 );
	color2[10] = BRIGHT24( color[10], 0.7 );
	color2[11] = BRIGHT24( color[11], 0.9 );
	color2[12] = BRIGHT24( color[12], 0.7 );
	color2[13] = BRIGHT24( color[13], 0.9 );
	color2[14] = BRIGHT24( color[14], 0.9 );

	DDPIXELFORMAT DDpf;
	DDpf.dwSize = sizeof(DDPIXELFORMAT);
	m_pDisplay->GetFrontBuffer()->GetPixelFormat(&DDpf);

	m_iColorDepth = DDpf.dwRGBBitCount;

	for( i = 0; i < 16; i++ )
	{
		m_crAppleColor[i] = ApplyRGBFormat( color2[i], &DDpf );
		m_crAppleColor2[i] = ApplyRGBFormat( color[i], &DDpf );
	}

	m_iWhiteColor = m_crAppleColor2[15];
	m_iGreenColor = ApplyRGBFormat( RGB24(150, 255, 150), &DDpf );

	RedrawAll();
}

unsigned int CScreen::GetHSB()
{
	return m_nHSB;
}

BYTE CScreen::CheckMode(WORD addr)
{
	BYTE mode = 0;
	switch( addr )
	{
	case RDTEXT:
		if ( m_iScrMode & SS_TEXT )
			mode = 0x80;
		break;
	case RDMIXED:
		if ( m_iScrMode & SS_MIXED )
			mode = 0x80;
		break;
	case RDPAGE2:
		if ( m_iScrMode & SS_PAGE2 )
			mode = 0x80;
		break;
	case RDHIRES:
		if ( m_iScrMode & SS_HIRES )
			mode = 0x80;
		break;
	case RD80STORE:
		if ( m_iScrMode & SS_80STORE )
			mode = 0x80;
		break;
	case RDVBLBAR:
		if ( !g_pBoard->IsVBlank() )		// not VBL, but VBL_BAR ( when Virtical Blank, it goes low )
			mode = 0x80;
		break;
	case RD80COL:
		if ( m_iScrMode & SS_80COL )
			mode = 0x80;
		break;
	case RDALTCHAR:
		if ( m_iScrMode & SS_ALTCHAR )
			mode = 0x80;
		break;
	}
	return mode;
}

void CScreen::Set80Store()
{
	m_iScrMode |= SS_80STORE;
}

void CScreen::Clr80Store()
{
	m_iScrMode &= ~SS_80STORE;
}

void CScreen::ReInitialize()
{
	HRESULT hr;
	
	CLockMgr<CCSWrapper> guard(m_Lock, TRUE);
    // Check the cooperative level before rendering
    if( FAILED( hr = m_pDisplay->GetDirectDraw()->TestCooperativeLevel() ) )
    {
		switch( hr )
		{
		case DDERR_EXCLUSIVEMODEALREADYSET:
		case DDERR_NOEXCLUSIVEMODE:
			// Do nothing because some other app has exclusive mode
//			Sleep(10);
			break;
		
		case DDERR_WRONGMODE:
			// The display mode changed on us. Update the
			// DirectDraw surfaces accordingly
			m_bWindowed = TRUE;			// is not exclusive mode
			TRACE("WRONGMODE\n");

			g_cDIKeyboard.SetActive(FALSE, FALSE);
			g_cDIMouse.SetActive(FALSE, FALSE);
			InitDirectX();
			break;
		default:
			break;
        }
    }
}

COLORREF CScreen::ApplyRGBFormat(COLORREF rgb32, LPDDPIXELFORMAT lpDDpf)
{
	COLORREF r = rgb32 & 0xFF0000;
	COLORREF g = rgb32 & 0x00FF00;
	COLORREF b = rgb32 & 0x0000FF;
	DWORD testbit;
	testbit = 0x800000;
	while ( testbit && !( lpDDpf->dwRBitMask & testbit ) )
	{
		testbit >>= 1;
		r >>= 1;
	}
	r &= lpDDpf->dwRBitMask;

	testbit = 0x008000;
	while ( testbit && !( lpDDpf->dwGBitMask & testbit ) )
	{
		testbit >>= 1;
		g >>= 1;
	}
	g &= lpDDpf->dwGBitMask;

	testbit = 0x000080;
	while ( testbit && !( lpDDpf->dwBBitMask & testbit ) )
	{
		testbit >>= 1;
		b >>= 1;
	}
	b &= lpDDpf->dwBBitMask;
	return( r | g | b );
}

void CScreen::writeMemory(WORD addr, BYTE data, BOOL aux)
{
	BYTE x = addr&0x7F;
	int addrPage=0;
	int curPage = ( m_iScrMode & SS_80STORE ) ? 0 : m_iScrMode & SS_PAGE2;
	if ( addr >= 0x2000 && addr < 0x6000 && x < 120 )
	{
		if ( addr & 0x4000 )
			addrPage = SS_PAGE2;
		
		if ( ( addrPage ^ curPage ) == 0 )		// same page
		{
			BOOL bDHIRES = ( m_iScrMode & ( SS_80COL | SS_DHIRES ) ) == ( SS_80COL | SS_DHIRES );
			if ( bDHIRES || !aux )		// dhires || aux == false
			{
				addr&=0x1FFF;
				int line = ( addr >> 10 ) | ( ( addr >> 4 ) & 0x38 ) | m_abPosTable[x];
				// when mode is graphics and not lo-res and mixed-graphics
				if ( !( m_iScrMode&SS_TEXT ) && m_iScrMode&SS_HIRES
					&& ( !( m_iScrMode & SS_MIXED ) || line < 160 ) )
				{
					// refresh only same page
					m_aScanLine[line]=1;
				}
			}
		}
	}
	else if ( addr >= 0x400 && addr < 0xC00 && x < 120 )
	{
		if ( addr & 0x800 )
			addrPage = SS_PAGE2;
		
		if( ( addrPage ^ curPage ) == 0 )		// same page
		{
			BOOL b80COL = m_iScrMode & SS_80COL;
			if ( b80COL || !aux )		// 80col || aux == false
			{
				int line = ( ( addr >> 4 ) & 0x38 ) | m_abPosTable[x];
				// when mode is text or lo-res or mixed-text
				if ( ( m_iScrMode&SS_TEXT || !( m_iScrMode&SS_HIRES )
					|| ( ( m_iScrMode & SS_MIXED ) && line >= 160 ) ) )
				{
					// refresh only same page
					addr&=0x3FF;
					m_aScanLine[line]=1;
				}
			}
		}
	}
}

void CScreen::Reset()
{
	m_iScrMode = SS_TEXT;
	RedrawAll();
}

void CScreen::OnLButtonUp(UINT nFlags, CPoint point) 
{
	// TODO: Add your message handler code here and/or call default
	g_cDIKeyboard.SetActive(TRUE);
	g_cDIMouse.SetActive(TRUE);

	CWnd::OnLButtonUp(nFlags, point);
}

void CScreen::OnMove(int x, int y) 
{
	CLockMgr<CCSWrapper> guard( m_Lock, TRUE );
	CWnd::OnMove(x, y);
	// TODO: Add your message handler code here
}

void CScreen::SetFullScreenMode(BOOL bFullScreen)
{
	BOOL bWindowed = !bFullScreen;
	if ( m_bWindowed != bWindowed )
	{
		BOOL suspended = g_pBoard->GetIsSuspended();
		if ( !suspended )
			g_pBoard->Suspend(TRUE);
		CLockMgr<CCSWrapper> guard( m_Lock, TRUE );
		m_bWindowed = bWindowed;
		InitDirectX();
		if ( !suspended )
			g_pBoard->Resume();
	}
}

CSurface* CScreen::GetDiskSurface()
{
	return m_pSurfaceDisk;
}

void CScreen::UpdateDiskSurface()
{
	CLockMgr<CCSWrapper> guard( m_Lock, TRUE );
	m_pDisplay->Blt( 550, 460, m_pSurfaceDisk, NULL );
	Present();
}

void CScreen::SetMessage(TCHAR *szText)
{
	CLockMgr<CCSWrapper> guard( m_Lock, TRUE );
	if ( szText != NULL )
		m_szMessage = szText;
	if ( !m_pSurfaceMsg )
		return;
	m_pSurfaceMsg->Clear();
	if ( m_hFont != NULL )
	{
		m_pSurfaceMsg->DrawText( m_hFont, m_szMessage, 0, 0, COLORREF(0), COLORREF(0xC0C0C0) );
	}
	m_pDisplay->Blt( 0, 460, m_pSurfaceMsg, NULL );
//	if ( !m_bInitializing )
		Present();
	m_nMsgVisiable = 2;
}

void CScreen::ToggleMessage()
{
	if ( !m_pSurfaceMsg )
		return;
	if ( m_nMsgVisiable )
		HideMessage();
	else
		SetMessage(NULL);
}

void CScreen::HideMessage()
{
	CLockMgr<CCSWrapper> guard( m_Lock, TRUE );
	if ( !m_pSurfaceMsg )
		return;
	m_pSurfaceMsg->Clear();
	m_pDisplay->Blt( 0, 460, m_pSurfaceMsg, NULL );
	Present();
	m_nMsgVisiable = 0;
}

// OnDisplayChange û  DirectDrawBuffer lock ɸ
//  ·  Լ  Ǿ dead lock Ѵ.
LRESULT CScreen::OnDisplayChanged(WPARAM wParam, LPARAM lParam)
{
	g_pBoard->m_pScreen->ReInitialize();
	return 0;
}

// WM_DISPLAYCHANGE ޽ ߻  DirectDrawBuffer lock
// ɸ · ѾǷ   CScreen lock ɸ
// dead lock ɸ  ִ.  window  ó 
// main thread  ̿  ó Ѵ.
LRESULT CScreen::OnDisplayChange(WPARAM wParam, LPARAM lParam)
{
	::PostMessage( m_hWnd, UM_DISPLAY_CHANGE, wParam, lParam );
	return 0;
}
