package org.argosdic.dictionary;

import java.util.List;

import junit.framework.Assert;
import junit.framework.TestCase;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.argosdic.preference.ArgosDicPreferenceStore;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.JFacePreferences;

/**
 * SearchHistoryTest.java
 * 
 * @author Xavier Cho
 * @version $Revision: 1.4 $ $Date: 2003/10/19 16:22:40 $
 */
public class SearchHistoryTest extends TestCase {
    private SearchHistory history;

    /**
     * Constructor for SearchHistoryTest.
     * 
     * @param name
     */
    public SearchHistoryTest(String name) {
        super(name);
    }

    /**
     * @see junit.framework.TestCase#setUp()
     */
    protected void setUp() throws Exception {
        super.setUp();

        IPreferenceStore preferences = new ArgosDicPreferenceStore();
        JFacePreferences.setPreferenceStore(preferences);

        this.history = new SearchHistory();

        history.add(createDocument("a", "a1")); //$NON-NLS-1$ //$NON-NLS-2$
        history.add(createDocument("b", "b1")); //$NON-NLS-1$ //$NON-NLS-2$
        history.add(createDocument("c", "c1")); //$NON-NLS-1$ //$NON-NLS-2$
    }

    /**
     * @see junit.framework.TestCase#tearDown()
     */
    protected void tearDown() throws Exception {
        super.tearDown();

        this.history = null;
    }

    public void testForward() {
        Assert.assertEquals("b", history.back()); //$NON-NLS-1$
        Assert.assertEquals("a", history.back()); //$NON-NLS-1$
        Assert.assertEquals("b", history.forward()); //$NON-NLS-1$
        Assert.assertEquals("c", history.forward()); //$NON-NLS-1$
    }

    public void testGo() {
        Assert.assertEquals("a", history.go(0)); //$NON-NLS-1$
        Assert.assertEquals("b", history.go(1)); //$NON-NLS-1$
        Assert.assertEquals("c", history.go(2)); //$NON-NLS-1$
    }

    public void testIsForwardEnabled() {
        Assert.assertTrue(history.isBackEnabled());
        Assert.assertFalse(history.isForwardEnabled());

        history.back();
        Assert.assertTrue(history.isBackEnabled());
        Assert.assertTrue(history.isForwardEnabled());

        history.back();
        Assert.assertFalse(history.isBackEnabled());
        Assert.assertTrue(history.isForwardEnabled());

        history.forward();
        Assert.assertTrue(history.isBackEnabled());
        Assert.assertTrue(history.isForwardEnabled());

        history.forward();
        Assert.assertTrue(history.isBackEnabled());
        Assert.assertFalse(history.isForwardEnabled());

        history.back();
        Assert.assertTrue(history.isBackEnabled());
        Assert.assertTrue(history.isForwardEnabled());

        history.add(createDocument("e", "e1")); //$NON-NLS-1$ //$NON-NLS-2$
        Assert.assertTrue(history.isBackEnabled());
        Assert.assertFalse(history.isForwardEnabled());
    }

    public void testClear() {
        history.clear();

        Assert.assertEquals(0, history.getSize());
        Assert.assertEquals(-1, history.getCurrentIndex());

        history.add(createDocument("a", "a1")); //$NON-NLS-1$ //$NON-NLS-2$
        Assert.assertFalse(history.isBackEnabled());
        Assert.assertFalse(history.isForwardEnabled());

        history.add(createDocument("b", "b1")); //$NON-NLS-1$ //$NON-NLS-2$
        Assert.assertTrue(history.isBackEnabled());
        Assert.assertFalse(history.isForwardEnabled());

        history.back();
        Assert.assertFalse(history.isBackEnabled());
        Assert.assertTrue(history.isForwardEnabled());
    }

    public void testGet() {
        Field field = history.get("a").getField("definition"); //$NON-NLS-1$ //$NON-NLS-2$
        Assert.assertEquals("a1", field.stringValue()); //$NON-NLS-1$

        field = history.get("b").getField("definition"); //$NON-NLS-1$ //$NON-NLS-2$
        Assert.assertEquals("b1", field.stringValue()); //$NON-NLS-1$

        field = history.get("c").getField("definition"); //$NON-NLS-1$ //$NON-NLS-2$
        Assert.assertEquals("c1", field.stringValue()); //$NON-NLS-1$
    }

    public void testGetSize() {
        Assert.assertEquals(3, history.getSize());

        history.add(createDocument("e", "e1")); //$NON-NLS-1$ //$NON-NLS-2$

        Assert.assertEquals(4, history.getSize());

        history.back();

        history.add(createDocument("f", "f1")); //$NON-NLS-1$ //$NON-NLS-2$
        Assert.assertEquals(4, history.getSize());

        history.back();
        history.back();

        history.add(createDocument("g", "g1")); //$NON-NLS-1$ //$NON-NLS-2$

        Assert.assertEquals(3, history.getSize());
    }

    public void testGetBackwardHistory() {
        List list = history.getBackwardHistory();
        Assert.assertNotNull(list);

        Assert.assertEquals(2, list.size());
        Assert.assertEquals("b", list.get(0)); //$NON-NLS-1$
        Assert.assertEquals("a", list.get(1)); //$NON-NLS-1$

        history.back();

        list = history.getBackwardHistory();
        Assert.assertNotNull(list);

        Assert.assertEquals(1, list.size());
        Assert.assertEquals("a", list.get(0)); //$NON-NLS-1$

        history.back();

        list = history.getBackwardHistory();
        Assert.assertNotNull(list);

        Assert.assertEquals(0, list.size());
    }

    public void testGetForwardHistory() {
        history.go(0);

        List list = history.getForwardHistory();
        Assert.assertNotNull(list);

        Assert.assertEquals(2, list.size());
        Assert.assertEquals("b", list.get(0)); //$NON-NLS-1$
        Assert.assertEquals("c", list.get(1)); //$NON-NLS-1$

        history.forward();

        list = history.getForwardHistory();
        Assert.assertNotNull(list);

        Assert.assertEquals(1, list.size());
        Assert.assertEquals("c", list.get(0)); //$NON-NLS-1$

        history.forward();

        list = history.getForwardHistory();
        Assert.assertNotNull(list);

        Assert.assertEquals(0, list.size());
    }

    private Document createDocument(String word, String definition) {
        Document document = new Document();

        Field field = Field.Text("word", word); //$NON-NLS-1$
        document.add(field);

        field = Field.Text("definition", definition); //$NON-NLS-1$
        document.add(field);

        return document;
    }
}
