package org.argosdic.dictionary;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.BindException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.URLDecoder;
import java.util.StringTokenizer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.StopAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.Filter;
import org.apache.lucene.search.Hits;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.argosdic.resource.ResourceManager;
import org.argosdic.util.MimeTypeMap;
import org.argosdic.util.SpecialFolders;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.JFacePreferences;
import org.eclipse.jface.util.Assert;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

/**
 * DictionaryServer.java
 * @author Xavier Cho
 * @version $Revision: 1.5 $ $Date: 2003/09/13 09:11:56 $
 */
public class DictionaryServer {
    private static Log log = LogFactory.getLog(DictionaryServer.class);

    private ServerSocket socket;
    private Analyzer analyzer;
    private IndexSearcher searcher;
    private String url;
    private String dataDir;
    private String htmlDir;
    private MimeTypeMap mimeMap;
    private int port;

    public DictionaryServer() {
        ResourceManager resources = ResourceManager.getInstance();

        IPreferenceStore preferences = JFacePreferences.getPreferenceStore();

        this.htmlDir = SpecialFolders.getHtmlDirectory();
        this.dataDir = preferences.getString("data.dir"); //$NON-NLS-1$
        this.port = preferences.getInt("server.port"); //$NON-NLS-1$
        this.url = "http://localhost:" + port; //$NON-NLS-1$
        this.mimeMap = new MimeTypeMap();

        setSearchIndexer();

        this.analyzer = new StopAnalyzer();

        DictionaryManager manager = DictionaryManager.getInstance();
        manager.getSelectedDictionary();
        manager.addDictionaryListener(new DictionaryListener() {
            public void dictionaryUpdated(DictionaryEvent event) {
            }

            public void dictionaryChanged(DictionaryEvent event) {
                setSearchIndexer();
            }
        });
    }

    private void setSearchIndexer() {
        this.searcher = null;

        DictionaryManager manager = DictionaryManager.getInstance();
        manager.getSelectedDictionary();

        Dictionary dictionary = manager.getSelectedDictionary();
        if (dictionary != null) {
            StringBuffer buffer = new StringBuffer();
            buffer.append(dataDir);
            buffer.append(File.separator);
            buffer.append(dictionary.getId());

            try {
                this.searcher = new IndexSearcher(buffer.toString());
            } catch (IOException e) {
                if (log.isWarnEnabled()) {
                    String msg = "Unable to use the selected dictionary."; //$NON-NLS-1$
                    log.warn(msg, e);
                }
            }
        }
    }

    public Hits search(String word) throws ParseException, IOException {
        if (word == null || word.trim().length() == 0) {
            return null;
        } else if (searcher == null) {
            ResourceManager resources = ResourceManager.getInstance();
            Shell parent = Display.getCurrent().getActiveShell();

            String title = resources.getString("dialog.title.warning"); //$NON-NLS-1$
            String msg = resources.getString("error.message.dictionary"); //$NON-NLS-1$

            MessageDialog.openWarning(parent, title, msg);

            return null;
        }

        Hits hits = null;

        try {
            Query query = parseQueryString(word.trim());
            hits = searcher.search(query, (Filter) null);
        } catch (ParseException e) {
            ResourceManager resources = ResourceManager.getInstance();

            Shell parent = Display.getCurrent().getActiveShell();
            String title = resources.getString("dialog.title.warning"); //$NON-NLS-1$
            String msg = resources.getString("error.message.query"); //$NON-NLS-1$

            MessageDialog.openWarning(parent, title, msg);

            throw e;
        }

        return hits;
    }

    protected Query parseQueryString(String word) throws ParseException {
        Assert.isNotNull(word);

        if (word.indexOf(" ") == -1 //$NON-NLS-1$
        && !word.endsWith("*") //$NON-NLS-1$
        && !word.endsWith("~") //$NON-NLS-1$
        && !word.endsWith("?")) { //$NON-NLS-1$
            word = word.concat("*"); //$NON-NLS-1$
        }

        return QueryParser.parse(word, "word", analyzer); //$NON-NLS-1$
    }

    /**
     * @return
     */
    public String getUrl() {
        return url;
    }

    public void start() throws IOException {
        if (socket == null) {
            if (log.isInfoEnabled()) {
                log.info("Starting dictionary server..."); //$NON-NLS-1$
            }

            synchronized (this) {
                int retry = 30;

                while ((socket == null || !socket.isBound()) && retry > 0) {
                    try {
                        this.socket = new ServerSocket(port);
                    } catch (BindException e) {
                        retry--;
                        port--;

                        if (log.isWarnEnabled()) {
                            log.warn("Unable to bind to the specified port."); //$NON-NLS-1$
                            log.warn("Retrying with port : " + port); //$NON-NLS-1$
                        }
                    }
                }

                if (socket == null || !socket.isBound()) {
                    String msg = "Unable to bind to the specified port range."; //$NON-NLS-1$
                    throw new IOException(msg);
                } else if (log.isInfoEnabled()) {
                    log.info("Dictionary server has been bound for port : " + port); //$NON-NLS-1$
                }
            }
        }

        Thread worker = new Thread(new ServerWorker());
        worker.start();
    }

    private class ServerWorker implements Runnable {
        public void run() {
            BufferedReader reader = null;
            OutputStream out = null;
            String word = null;

            if (searcher == null) {
                return;
            }

            while (!socket.isClosed()) {
                try {
                    Socket clientSocket = socket.accept();

                    if (log.isDebugEnabled()) {
                        log.debug("Accepting new client."); //$NON-NLS-1$
                    }

                    InputStream in = clientSocket.getInputStream();
                    out = clientSocket.getOutputStream();

                    reader = new BufferedReader(new InputStreamReader(in));

                    String resource = null;

                    String line = null;
                    while ((line = reader.readLine()) != null) {
                        if (line.trim().length() == 0) {
                            break;
                        }

                        if (line.startsWith("GET")) { //$NON-NLS-1$
                            StringTokenizer tokenizer =
                                new StringTokenizer(line);
                            if (tokenizer.countTokens() > 2) {
                                tokenizer.nextToken();
                                resource = tokenizer.nextToken();

                                if (resource.startsWith("/")) { //$NON-NLS-1$
                                    resource = resource.substring(1);
                                }

                                break;
                            }
                        }
                    }

                    resource = URLDecoder.decode(resource, "euc-kr"); //$NON-NLS-1$

                    if (log.isDebugEnabled()) {
                        log.debug("Finished reading client request."); //$NON-NLS-1$
                    }

                    if (resource != null) {
                        if (resource.endsWith(".dic")) { //$NON-NLS-1$
                            word = resource.substring(0, resource.length() - 4);

                            if (log.isDebugEnabled()) {
                                log.debug("Searching for : " + word); //$NON-NLS-1$
                            }

                            Query query = QueryParser.parse(word, "word", new StopAnalyzer()); //$NON-NLS-1$

                            Hits hits = searcher.search(query, (Filter) null);

                            if (hits.length() > 0) {
                                Document document = hits.doc(0);

                                if (log.isDebugEnabled()) {
                                    log.debug("Writing server response..."); //$NON-NLS-1$
                                }

                                StringBuffer buffer = new StringBuffer();
                                buffer.append("<html><head><style>body {font-size: 10pt;}</style><meta http-equiv=\"Content-Type\" "); //$NON-NLS-1$
                                buffer.append("content=\"text/html; charset=euc-kr\"></head><body>"); //$NON-NLS-1$
                                buffer.append(document.getField("definition") //$NON-NLS-1$
                                .stringValue());
                                buffer.append("</body></html>"); //$NON-NLS-1$

                                String source = buffer.toString();

                                out.write("HTTP/1.0 200 OK".getBytes()); //$NON-NLS-1$
                                out.write("\r\n".getBytes()); //$NON-NLS-1$
                                out.write("Content-type: text/html;charset=euc-kr" //$NON-NLS-1$
                                .getBytes());
                                out.write("\r\n".getBytes()); //$NON-NLS-1$
                                out.write("\r\n".getBytes()); //$NON-NLS-1$

                                out.write(source.getBytes());
                            }
                        } else {
                            if (log.isDebugEnabled()) {
                                log.debug("Processing resource : " + resource); //$NON-NLS-1$
                            }

                            StringBuffer buffer = new StringBuffer(htmlDir);
                            buffer.append(File.separator);
                            buffer.append(
                                resource.replace('/', File.separatorChar));

                            File file = new File(buffer.toString());

                            if (log.isDebugEnabled()) {
                                log.debug("File name : " + file.getAbsolutePath()); //$NON-NLS-1$
                            }

                            if (!file.exists() || file.isDirectory()) {
                                System.out.println("The specified file has not been found : " //$NON-NLS-1$
                                +file.getAbsolutePath());

                                out.write("HTTP/1.0 404\r\n".getBytes()); //$NON-NLS-1$
                                out.write("Content-type: text/html;charset=euc-kr" //$NON-NLS-1$
                                .getBytes());
                                out.write("\r\n\r\n".getBytes()); //$NON-NLS-1$
                                out.write("The specified resource has not been found." //$NON-NLS-1$
                                .getBytes());
                            } else {
                                if (log.isDebugEnabled()) {
                                    log.debug("Writing file content..."); //$NON-NLS-1$
                                }

                                out.write("HTTP/1.0 200 OK".getBytes()); //$NON-NLS-1$
                                out.write("\r\n".getBytes()); //$NON-NLS-1$
                                out.write("Accept-Ranges: bytes".getBytes()); //$NON-NLS-1$
                                out.write("\r\n".getBytes()); //$NON-NLS-1$
                                out.write("Content-type: ".getBytes()); //$NON-NLS-1$
                                out.write(mimeMap.getContentTypeFor(resource).getBytes()); //$NON-NLS-1$
                                out.write("\r\n".getBytes()); //$NON-NLS-1$
                                out.write(("Content-Length: " + file.length()) //$NON-NLS-1$
                                .getBytes());
                                out.write("\r\n\r\n".getBytes()); //$NON-NLS-1$

                                DataOutputStream dout = null;
                                InputStream fin = null;

                                try {
                                    in =
                                        new DataInputStream(
                                            new FileInputStream(file));
                                    in = new BufferedInputStream(in);

                                    out = new BufferedOutputStream(out);
                                    dout = new DataOutputStream(out);

                                    int data = -1;
                                    while ((data = in.read()) != -1) {
                                        dout.writeByte(data);
                                    }

                                    dout.flush();
                                } finally {
                                    if (fin != null) {
                                        try {
                                            fin.close();
                                        } catch (IOException ie) {
                                        }
                                    }

                                    if (dout != null) {
                                        try {
                                            dout.close();
                                        } catch (IOException ie) {
                                        }
                                    }
                                }
                            }
                        }
                    }
                } catch (ParseException e) {
                    ResourceManager resources = ResourceManager.getInstance();

                    Shell parent = Display.getCurrent().getActiveShell();
                    String title = resources.getString("dialog.title.warning"); //$NON-NLS-1$
                    String msg = resources.getString("error.message.query"); //$NON-NLS-1$

                    MessageDialog.openWarning(parent, title, msg);
                } catch (IOException e) {
                    e.printStackTrace();
                } finally {
                    if (reader != null) {
                        try {
                            reader.close();
                        } catch (IOException e) {
                        }
                    }

                    if (out != null) {
                        try {
                            out.close();
                        } catch (Exception e) {
                        }
                    }
                }
            }
        }
    }

    public static void main(String[] args) {
        try {
            DictionaryServer server = new DictionaryServer();
            server.start();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}
