package net.kldp.beat.action;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.kldp.beat.annotation.After;
import net.kldp.beat.annotation.Before;
import net.kldp.beat.annotation.BeforeResult;
import net.kldp.beat.annotation.Result;
import net.kldp.beat.annotation.Results;
import net.kldp.beat.exception.InterceptorException;
import net.kldp.beat.exception.ResultException;
import net.kldp.beat.interceptor.AfterInterceptor;
import net.kldp.beat.interceptor.BeforeInterceptor;
import net.kldp.beat.interceptor.Interceptor;
import net.kldp.beat.interceptor.SystemInterceptor;
import net.kldp.beat.interceptor.UserInterceptor;
import net.kldp.beat.web.interceptor.InterceptorFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * 액션에 정의된 인터셉터를 분류합니다. 이들 인터셉터들은 ActionService객체에서 요청되어 차례차례 실행됩니다.
 */
public class InterceptorStack {
	private Log logger = LogFactory.getLog(InterceptorStack.class);

	private List<Result> results = new ArrayList<Result>();
	private Map<UserInterceptor, Annotation> beforeInterceptors = new HashMap<UserInterceptor, Annotation>();
	private Map<UserInterceptor, Annotation> afterInterceptors = new HashMap<UserInterceptor, Annotation>();
	private Map<SystemInterceptor, Annotation> systemInterceptors = new HashMap<SystemInterceptor, Annotation>();;
	private Method beforeMethod;
	private Method afterMethod;
	private Method resultBeforeMethod;

	public InterceptorStack(Class<? extends Object> actionClass) throws ResultException {
		AnnotationFinder finder = new AnnotationFinder(actionClass);
		finder.find();
		for (Annotation annotation : finder.getAnnotations()) {
			System.out.println(annotation);
			classifyAnnotations(annotation);
		}
		classifyMethods(actionClass.getMethods());
	}

	/**
	 * Action클래스안에서의 특별한 메서드들을 분류합니다. 이들 메서드는 Before, BeforResult, After를 선언한
	 * 메서드입니다.
	 * 
	 * @param methods
	 */
	private void classifyMethods(Method[] methods) {
		for (Method method : methods) {
			Before before = method.getAnnotation(Before.class);
			After after = method.getAnnotation(After.class);
			BeforeResult resultBefore = method.getAnnotation(BeforeResult.class);
			if (before != null && beforeMethod == null) {
				this.beforeMethod = method;
			} else if (resultBefore != null && resultBeforeMethod == null) {
				this.resultBeforeMethod = method;
			} else if (after != null && afterMethod == null) {
				this.afterMethod = method;
			}
		}
	}

	/**
	 * 어노테이션들을 Result또는 인터셉터로 분류합니다.
	 * 
	 * @param annotation
	 * @throws ResultException
	 */
	private void classifyAnnotations(Annotation annotation) throws ResultException {
		if (annotation instanceof Result) {
			addResult((Result) annotation);
		} else if (annotation instanceof Results) {
			Results results = (Results) annotation;
			for (Result result : results.value()) {
				addResult(result);
			}
		} else {
			classifyInterceptors(annotation);
		}
	}

	/**
	 * Result를 추가합니다. Result의 value는 반드시 /로 시작되어야 합니다.
	 * 
	 * @param result
	 * @throws ResultException
	 */
	private void addResult(Result result) throws ResultException {
		if (!result.value().equals("") && result.value().charAt(0) != '/')
			throw new ResultException("value is must start at '/' in Result annotation");
		results.add(result);
	}

	/**
	 * 인터셉터들을 분류합니다.
	 * 
	 * @param annotation
	 */
	private void classifyInterceptors(Annotation annotation) {
		try {
			Interceptor interceptor = InterceptorFactory.getInterceptor(annotation);
			if (interceptor instanceof SystemInterceptor) {
				systemInterceptors.put((SystemInterceptor) interceptor, annotation);
			} else {
				if (interceptor instanceof BeforeInterceptor) {
					beforeInterceptors.put((UserInterceptor) interceptor, annotation);
				}
				if (interceptor instanceof AfterInterceptor) {
					afterInterceptors.put((UserInterceptor) interceptor, annotation);
				}
			}
		} catch (InterceptorException e) {
			logger.error(e);
		}
	}

	/**
	 * 정의된 Result를 리턴합니다.
	 * 
	 * @return
	 */
	public List<Result> getResults() {
		return results;
	}

	public boolean hasResults() {
		return results.size() > 0 ? true : false;
	}

	public Result getResult(String resultName) throws ResultException {
		for (Result result : results) {
			if (result.name().equals(resultName))
				return result;
		}
		throw new ResultException("can not found Result for name=" + resultName);
	}

	/**
	 * 액션이 정의한 시스템 인터셉터를 리턴합니다.
	 * 
	 * @return
	 */
	public Map<SystemInterceptor, Annotation> getSystemInterceptors() {
		return systemInterceptors;
	}

	/**
	 * 액션이 정의한 사용자 전처리 인터셉터를 리턴합니다.
	 * 
	 * @return
	 */
	public Map<UserInterceptor, Annotation> getBeforeInterceptors() {
		return beforeInterceptors;
	}

	/**
	 * 액션이 정의한 사용자 후처리 인터셉터를 리턴합니다.
	 * 
	 * @return
	 */
	public Map<UserInterceptor, Annotation> getAfterInterceptors() {
		return afterInterceptors;
	}

	/**
	 * Before 메서드를 리턴합니다.
	 * 
	 * @return
	 */
	public Method getBeforeMethod() {
		return beforeMethod;
	}

	/**
	 * After 메서드를 리턴합니다.
	 * 
	 * @return
	 */
	public Method getAfterMethod() {
		return afterMethod;
	}

	/**
	 * BeforeResult 메서드를 리턴합니다.
	 * 
	 * @return
	 */
	public Method getBeforeResultMethod() {
		return resultBeforeMethod;
	}
}