package net.kldp.beat.dispatcher;

import java.io.IOException;
import java.util.Map;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.kldp.beat.configuration.Config;

public abstract class ViewDispatcher implements Dispatch {

	protected ServletContext context;
	protected HttpServletRequest request;
	protected HttpServletResponse response;
	protected static final String REDIRECT = "redirect";
	protected static final String FORWARD = "forward";
	protected static final String NOTHING = "nothing";

	public static ViewDispatcher create(ServletContext context, HttpServletRequest request,
			HttpServletResponse response) throws ServletException {
		ViewDispatcher dispatcher;
		if ("jsp".equals(Config.viewType())) {
			dispatcher = new JspViewDispatcher();
		} else if ("velocity".equals(Config.viewType())) {
			dispatcher = new JspViewDispatcher();
		} else if ("freemaker".equals(Config.viewType())) {
			dispatcher = new JspViewDispatcher();
		} else {
			throw new ServletException("can not found dispatcher. check configuration file");
		}

		dispatcher.context = context;
		dispatcher.request = request;
		dispatcher.response = response;
		return dispatcher;
	}

	@Override
	public void dispatch(String url, String type) throws ServletException, IOException {
		if (type.equals(REDIRECT)) {
			response.sendRedirect(url);
		} else if (type.equals(FORWARD)) {
			RequestDispatcher dispatcher = context.getRequestDispatcher(url);
			dispatcher.forward(request, response);
		} else if (type.equals(NOTHING)) {
			// nothing
		} else {
			throw new ServletException("can not dispath Result. unknown type - " + type);
		}
	}

	@Override
	public String getQueryString() {
		return request.getQueryString();
	}

	@Override
	public String getRequestURI() {
		return request.getRequestURI();
	}

	protected String getNamespace() {
		String path = request.getServletPath();
		int index = path.indexOf('/', 1);
		if (index > 0)
			return path.substring(0, path.indexOf('/', 1));
		else
			return "";
	}

	protected String interpretString(String result, Map<String, Object> map) {
		for (String key : map.keySet()) {
			Object value = map.get(key);
			if (value != null)
				result = result.replace("${" + key + "}", value.toString());
		}
		return result;
	}

	protected String arrayToParam(String[] params) {
		return arrayToParam(params, "&");
	}

	protected String arrayToParam(String[] params, String sep) {
		StringBuffer buffer = new StringBuffer();
		for (String param : params)
			buffer.append(param + sep);
		if (buffer.length() < 1)
			return "";
		return buffer.substring(0, buffer.length() - 1);
	}

	protected String[] parseHeaders(String[] headers, Map<String, Object> map) {
		String header = interpretString(arrayToParam(headers, "\n"), map);
		return header.split("\n");
	}

	protected void addHeaders(String[] headers, HttpServletResponse response) {
		for (String header : headers) {
			String[] split = header.split(":");
			response.addHeader(split[0], split[1]);
		}
	}

	protected String getViewPath() {
		return Config.getMapping(getNamespace()).getViewLocation();
	}
}