------------------------------------------------------------------------------------
--
-- CensusPlus
-- A WoW UI customization by Cooper Sellers
--
--
------------------------------------------------------------------------------------

----------------------------------------------------------------------------------
--
-- EURO vs US localization problem workaround for common server names
-- 
---------------------------------------------------------------------------------
local g_InterfaceVersion = 10900.2;
g_CensusPlusLocale = "US";							--  Must read either US or EU
g_CensusPlusTZOffset = -999;
local g_LocaleSet = false;
local g_TZWarningSent = false;


----------------------------------------------------------------------------------
--
-- Constants
-- 
---------------------------------------------------------------------------------
local CensusPlus_VERSION = "3.2kr5"; 				-- version
local CensusKr_DataVersion = 1;                                -- data version
local CensusPlus_MAXBARHEIGHT = 128;			-- Length of blue bars
local CensusPlus_NUMGUILDBUTTONS = 10;			-- How many guild buttons are on the UI?
local MAX_CHARACTER_LEVEL = 60;					-- Maximum level a PC can attain
local MAX_WHO_RESULTS = 49;						-- Maximum number of who results the server will return
CensusPlus_GUILDBUTTONSIZEY = 16;
local CensusPlus_UPDATEDELAY = 5;				-- Delay time between /who messages
local CP_MAX_TIMES = 50;

local g_ServerPrefix = "";						--  US VERSION!!
--local g_ServerPrefix = "EU-";					--  EU VERSION!!

	
----------------------------------------------------------------------------------
--
-- Print a string to the chat frame
--	msg - message to print
-- 
---------------------------------------------------------------------------------
function CensusPlus_Msg(msg)
	ChatFrame1:AddMessage("Census+: "..msg, 1.0, 1.0, 0.5);
end

local function CensusPlus_Msg2( msg )
	ChatFrame2:AddMessage("Census+: "..msg, 0.5, 1.0, 1.0);
end

----------------------------------------------------------------------------------
--
-- Global scope variables
-- 
---------------------------------------------------------------------------------
CensusPlus_Database = {};							-- Database of all CensusPlus results

----------------------------------------------------------------------------------
--
-- File scope variables
-- 
---------------------------------------------------------------------------------
local g_CensusPlusInitialized;						-- Is CensusPlus initialized?
local g_JobQueue = {};							-- The queue of pending jobs
local g_CurrentJob = {};						-- Current job being executed
local g_IsCensusPlusInProgress = false;			-- Is a CensusPlus in progress?
local g_CensusPlusPaused = false;               -- Is CensusPlus in progress paused?
local g_CensusPlusManuallyPaused = false;       -- Is CensusPlus in progress manually paused?
local g_WhoAutoClose = 0;                       -- AutoClose who window?

local g_NumNewCharacters = 0;					-- How many new characters found this CensusPlus
local g_NumUpdatedCharacters = 0;				-- How many characters were updated during this CensusPlus

local g_MobXPByLevel = {};						-- XP earned for killing
local g_CharacterXPByLevel = {};				-- XP required to advance through the given level
local g_TotalCharacterXPPerLevel = {};			-- Total XP required to attain the given level

local g_Guilds = {};							-- All known guild

local g_TotalCharacterXP = 0;					-- Total character XP for currently selected search
local g_TotalCount = 0;							-- Total number of characters which meet search criteria
local g_RaceCount = {};							-- Totals for each race given search criteria
local g_ClassCount = {};						-- Totals for each class given search criteria
local g_LevelCount = {};						-- Totals for each level given search criteria
local g_TempCount  = {}; -- 직업
local g_TempRaceCount  = {}; -- 종족
local g_TempZoneCount = {};

local g_GuildSelected = 0;						-- Search criteria: Currently selected guild, 0 indicates none
local g_RaceSelected = 0;						-- Search criteria: Currently selected race, 0 indicates none
local g_ClassSelected = 0;						-- Search criteria: Currently selected class, 0 indicates none
local g_LevelSelected = 0;

local g_LastOnUpdateTime = 0;					-- Last time OnUpdate was called
local g_WaitingForWhoUpdate = false;			-- Are we waiting for a who update event?

local g_WhoAttempts = 0;                        -- Counter for detecting stuck who results
local g_MiniOnStart = 1;                        -- Flag to have the mini-censusP displayed on startup

local g_CompleteCensusStarted = false;          -- Flag for counter
local g_TakeHour = 0;                           -- Our timing hour 
local g_TimeDatabase = {};                      -- Time database
local g_ResetHour = true;                       -- Rest hour
local g_VariablesLoaded = false;                -- flag to tell us if vars are loaded
local g_FirstRun = true;
local g_LastCensusRun = time() - 1500;			--  timer used if auto census is turned on

local g_Pre_FriendsFrameOnHideOverride = nil;		--  override for friend's frame to stop the close window sound
local g_Pre_FriendsFrameOnShowOverride = nil;		--  override for friend's frame to stop the close window sound
local g_Pre_WhoList_UpdateOverride = nil;		--  override for friend's frame to stop the close window sound
local g_Pre_WhoHandler = nil;						--  override for submiting a who
local g_Pre_FriendsFrame_Update = nil;
local CP_updatingGuild  = nil;
g_CensusPlusLastTarget = nil;
g_CensusPlusLastTargetName = nil;
local g_CurrentlyInBG = false;

local g_AccumulatedPruneData = {};

g_RaceClassList = { };						-- Used to pick the right icon
g_RaceClassList[CENSUSPlus_DRUID]		= 10;
g_RaceClassList[CENSUSPlus_HUNTER]		= 11;
g_RaceClassList[CENSUSPlus_MAGE]		= 12;
g_RaceClassList[CENSUSPlus_PRIEST]		= 13;
g_RaceClassList[CENSUSPlus_ROGUE]		= 14;
g_RaceClassList[CENSUSPlus_WARLOCK]	    = 15;
g_RaceClassList[CENSUSPlus_WARRIOR]	    = 16;
g_RaceClassList[CENSUSPlus_SHAMAN]		= 17;
g_RaceClassList[CENSUSPlus_PALADIN]	    = 18;

g_RaceClassList[CENSUSPlus_DWARF]		= 20;
g_RaceClassList[CENSUSPlus_GNOME]		= 21;
g_RaceClassList[CENSUSPlus_HUMAN]		= 22;
g_RaceClassList[CENSUSPlus_NIGHTELF]	= 23;
g_RaceClassList[CENSUSPlus_ORC]		    = 24;
g_RaceClassList[CENSUSPlus_TAUREN]		= 25;
g_RaceClassList[CENSUSPlus_TROLL]		= 26;
g_RaceClassList[CENSUSPlus_UNDEAD]		= 27;

local g_TimeDatabase = {};                      -- Time database
g_TimeDatabase[CENSUSPlus_DRUID]		= 0;
g_TimeDatabase[CENSUSPlus_HUNTER]		= 0;
g_TimeDatabase[CENSUSPlus_MAGE]			= 0;
g_TimeDatabase[CENSUSPlus_PRIEST]		= 0;
g_TimeDatabase[CENSUSPlus_ROGUE]		= 0;
g_TimeDatabase[CENSUSPlus_WARLOCK]	    = 0;
g_TimeDatabase[CENSUSPlus_WARRIOR]	    = 0;
g_TimeDatabase[CENSUSPlus_SHAMAN]		= 0;
g_TimeDatabase[CENSUSPlus_PALADIN]	    = 0;
g_TimeDatabase["전쟁노래 협곡"]			= 0;
g_TimeDatabase["알터랙 계곡"]		= 0;
g_TimeDatabase["아라시 분지"]			= 0;
g_TimeDatabase[CENSUSPlus_DWARF]	= 0;
g_TimeDatabase[CENSUSPlus_GNOME]	= 0;
g_TimeDatabase[CENSUSPlus_HUMAN]	= 0;
g_TimeDatabase[CENSUSPlus_NIGHTELF]	= 0;
g_TimeDatabase[CENSUSPlus_ORC]		= 0;
g_TimeDatabase[CENSUSPlus_TAUREN]	= 0;
g_TimeDatabase[CENSUSPlus_TROLL]		= 0;
g_TimeDatabase[CENSUSPlus_UNDEAD]	= 0;


local g_FactionCheck = {};

g_FactionCheck[CENSUSPlus_ORC]		= CENSUSPlus_HORDE;

g_FactionCheck[CENSUSPlus_TAUREN]	= CENSUSPlus_HORDE;

g_FactionCheck[CENSUSPlus_TROLL]	= CENSUSPlus_HORDE;

g_FactionCheck[CENSUSPlus_UNDEAD]	= CENSUSPlus_HORDE;


g_FactionCheck[CENSUSPlus_DWARF]	= CENSUSPlus_ALLIANCE;

g_FactionCheck[CENSUSPlus_GNOME]	= CENSUSPlus_ALLIANCE;

g_FactionCheck[CENSUSPlus_HUMAN]	= CENSUSPlus_ALLIANCE;

g_FactionCheck[CENSUSPlus_NIGHTELF]	= CENSUSPlus_ALLIANCE;

local g_TimeStamp = nil;
local g_ProcessedZone = nil;
local g_ProcessZoneCount = 0;


-----------------------------------------------------------------------------------
--
-- Insert a job at the end of the job queue
--
-----------------------------------------------------------------------------------
local function InsertJobIntoQueue(job)
	table.insert(g_JobQueue, job);
end

-----------------------------------------------------------------------------------
--
-- Initialize the tables of constants for XP calculations
--
-----------------------------------------------------------------------------------
local function InitConstantTables()
	--
	-- XP earned for killing
	--
	for i = 1, MAX_CHARACTER_LEVEL, 1 do
		g_MobXPByLevel[i] = (i * 5) + 45;
	end
	
	--
	-- XP required to advance through the given level
	--
	for i = 1, MAX_CHARACTER_LEVEL, 1 do
		g_CharacterXPByLevel[i] = ((8 * i * g_MobXPByLevel[i]) / 100) * 100;
	end

	--
	-- Total XP required to attain the given level
	--
	local totalCharacterXP = 0;
	for i = 1, MAX_CHARACTER_LEVEL, 1 do
		g_TotalCharacterXPPerLevel[i] = totalCharacterXP;
		totalCharacterXP = totalCharacterXP + g_CharacterXPByLevel[i];
	end
end

-----------------------------------------------------------------------------------
--
-- Return a table of races for the input faction
--
-----------------------------------------------------------------------------------
local function GetFactionRaces(faction)
	local ret = {};
	if (faction == CENSUSPlus_HORDE) then
		ret = {CENSUSPlus_ORC, CENSUSPlus_TAUREN, CENSUSPlus_TROLL, CENSUSPlus_UNDEAD};
	elseif (faction == CENSUSPlus_ALLIANCE) then
		ret = {CENSUSPlus_DWARF, CENSUSPlus_GNOME, CENSUSPlus_HUMAN, CENSUSPlus_NIGHTELF};
	end
	return ret;
end

-----------------------------------------------------------------------------------
--
-- Return a table of classes for the input faction
--
-----------------------------------------------------------------------------------
local function GetFactionClasses(faction)
	local ret = {};
	if (faction == CENSUSPlus_HORDE) then
		ret = {CENSUSPlus_DRUID, CENSUSPlus_HUNTER, CENSUSPlus_MAGE, CENSUSPlus_PRIEST, CENSUSPlus_ROGUE, CENSUSPlus_WARLOCK, CENSUSPlus_WARRIOR, CENSUSPlus_SHAMAN};
	elseif (faction == CENSUSPlus_ALLIANCE) then
		ret = {CENSUSPlus_DRUID, CENSUSPlus_HUNTER, CENSUSPlus_MAGE, CENSUSPlus_PRIEST, CENSUSPlus_ROGUE, CENSUSPlus_WARLOCK, CENSUSPlus_WARRIOR, CENSUSPlus_PALADIN};
	end
	return ret;
end

-----------------------------------------------------------------------------------
--
-- Return a table of classes for the input race
--
-----------------------------------------------------------------------------------
local function GetRaceClasses(race)
	local ret = {};
	if (race == CENSUSPlus_ORC) then
		ret = {CENSUSPlus_WARRIOR, CENSUSPlus_HUNTER, CENSUSPlus_ROGUE, CENSUSPlus_SHAMAN, CENSUSPlus_WARLOCK};
	elseif (race == CENSUSPlus_TAUREN) then
		ret = {CENSUSPlus_WARRIOR, CENSUSPlus_HUNTER, CENSUSPlus_SHAMAN, CENSUSPlus_DRUID};
	elseif (race == CENSUSPlus_TROLL) then
		ret = {CENSUSPlus_WARRIOR, CENSUSPlus_HUNTER, CENSUSPlus_ROGUE, CENSUSPlus_PRIEST, CENSUSPlus_SHAMAN, CENSUSPlus_MAGE};
	elseif (race == CENSUSPlus_UNDEAD) then
		ret = {CENSUSPlus_WARRIOR, CENSUSPlus_ROGUE, CENSUSPlus_PRIEST, CENSUSPlus_MAGE, CENSUSPlus_WARLOCK};
	elseif (race == CENSUSPlus_DWARF) then
		ret = {CENSUSPlus_WARRIOR, CENSUSPlus_PALADIN, CENSUSPlus_HUNTER, CENSUSPlus_ROGUE, CENSUSPlus_PRIEST};
	elseif (race == CENSUSPlus_GNOME) then
		ret = {CENSUSPlus_WARRIOR, CENSUSPlus_ROGUE, CENSUSPlus_MAGE, CENSUSPlus_WARLOCK};
	elseif (race == CENSUSPlus_HUMAN) then
		ret = {CENSUSPlus_WARRIOR, CENSUSPlus_PALADIN, CENSUSPlus_ROGUE, CENSUSPlus_PRIEST, CENSUSPlus_MAGE, CENSUSPlus_WARLOCK};
	elseif (race == CENSUSPlus_NIGHTELF) then
		ret = {CENSUSPlus_WARRIOR, CENSUSPlus_HUNTER, CENSUSPlus_ROGUE, CENSUSPlus_PRIEST, CENSUSPlus_DRUID};
	end
	return ret;
end

-----------------------------------------------------------------------------------
--
-- Return common letters found in zone names
--
-----------------------------------------------------------------------------------
--[[
    -- 칼림도어
    "다르나서스",
    "텔드랏실",
    "어둠의 해안",
    "악령의 숲",
    "여명의 설원",
    "잿빛 골짜기",
    "아즈샤라",
    "돌발톱 산맥",
    "불모의 땅",
    "오그리마",
    "듀로타",
    "잊혀진 땅",
    "썬더 블러프",
    "멀고어",
    "페랄라스",
    "먼지진흙 습지대",
    "버섯구름 봉우리",
    "실리더스",
    "운고로 분화구",
    "타나리스",
    -- 동부대륙
    "검은바위 산", 
    "언더시티",
    "아이언포지",
    "스톰윈드",
    "티리스팔 숲",
    "서부 역병지대",
    "동부 역병지대",
    "은빛소나무 숲",
    "알터랙 산맥",
    "힐스브래드 구릉지",
    "동부 내륙지",
    "아라시 고원",
    "저습지",
    "던 모로",
    "모단 호수",
    "이글거리는 협곡",
    "황야의 땅",
    "불타는 평원",
    "엘윈 숲",
    "붉은마루 산맥",
    "서부 몰락지대",
    "그늘숲",
    "죽음의 고개",
    "슬픔의 늪",
    "가시덤불 골짜기",
    "저주받은 땅",
    -- 던젼
    "성난불길 협곡",
    "통곡의 동굴",
    "검은심연의 나락",
    "가시덩쿨 우리",
    "가시덩쿨 구릉",
    "줄파락",
    "줄구룹",
    "마라우돈",
    "혈투의 전장",
    "오닉시아의 둥지",
    "죽음의 폐광",
    "그림자송곳니 성채",
    "스톰윈드 지하감옥",
    "놈리건",
    "붉은십자군 수도원",
    "울다만",
    "아탈학카르 신전",
    "검은바위 나락",
    "검은바위 첨탑",
    "스칼로맨스",
    "스트라솔름",
    "화산 심장부",
    "검은날개 둥지",
    "안퀴라즈",
    -- 전장
    "전쟁노래 협곡",
    "알터랙 계곡",
    "아라시 분지"
]]

local function GetZoneLetters()
	--return {"t", "d", "g", "f", "h", "b", "x", "gulch", "valley", "basin" };
	-- 한글 지역
	return {
            "나", -- 검은바위 나락, 검은심연의 나락, 다르나서스, 은빛소나무 숲, 타나리스 
            "곡", -- 성난불길 협곡, 알터랙 계곡, 이글거리는 협곡, 전쟁노래 협곡, 통곡의 동굴 
            "즈", -- 아즈샤라, 안퀴라즈, 안퀴라즈 사원, 안퀴라즈 성문, 안퀴라즈 폐허 
            "구", -- 가시덩쿨 구릉, 버섯구름 봉우리, 운고로 분화구, 줄구룹, 힐스브래드 구릉지 
            "땅", -- 불모의 땅, 잊혀진 땅, 저주받은 땅, 황야의 땅 
            "지대", -- 동부 역병지대, 먼지진흙 습지대, 서부 몰락지대, 서부 역병지대 
            "윈", -- 스톰윈드, 스톰윈드 지하감옥, 엘윈 숲 
            "맥", -- 돌발톱 산맥, 붉은마루 산맥, 알터랙 산맥 
            "더", -- 실리더스, 썬더 블러프, 언더시티 
            "라시", -- 아라시 고원, 아라시 분지 
            "음", -- 죽음의 고개, 죽음의 폐광 
            "자", -- 그림자송곳니 성채, 붉은십자군 수도원 
            "짜", -- 가시덤불 골짜기, 잿빛 골짜기 
            "장", -- 혈투의 전장, 화산 심장부 
            "어", -- 멀고어, 어둠의 해안 
            "검은바위 산", -- 검은바위 산 
            "검은바위 첨탑", -- 검은바위 첨탑 
            "검은날개 둥지", -- 검은날개 둥지 
            "슬픔의 늪", -- 슬픔의 늪 
            "악령의 숲", -- 악령의 숲 
            "여명의 설원", -- 여명의 설원 
            "오닉시아의 둥지", -- 오닉시아의 둥지 
            "울다만", -- 울다만 
            "아탈학카르 신전", -- 아탈학카르 신전 
            "마라우돈", -- 마라우돈 
            "스트라솔름", -- 스트라솔름 
            "티리스팔 숲", -- 티리스팔 숲 
            "줄파락", -- 줄파락 
            "불타는 평원", -- 불타는 평원 
            "페랄라스", -- 페랄라스 
            "텔드랏실", -- 텔드랏실 
            "가시덩쿨 우리", -- 가시덩쿨 우리 
            "모단 호수", -- 모단 호수 
            "놈리건", -- 놈리건 
            "저습지", -- 저습지 
            "스칼로맨스", -- 스칼로맨스 
            "듀로타", -- 듀로타 
            "동부 내륙지", -- 동부 내륙지 
            "던 모로", -- 던 모로 
            "그늘숲", -- 그늘숲 
            "아이언포지", -- 아이언포지 
            "오그리마", -- 오그리마 
        };
end
	
-----------------------------------------------------------------------------------
--
-- Return common letters found in names, may override this for other languages
--   Worst case scenario is to do it for every letter in the alphabet
--
-----------------------------------------------------------------------------------
local function GetNameLetters()
	return { "a", "b", "c", "d", "e", "f", "g", "i", "o", "p", "r", "s", "t", "u", "y" };
end

---------------------------------------------------------------------------------
--
-- Register with Cosmos UI
-- 
---------------------------------------------------------------------------------
local function CensusPlus_RegisterCosmos()
	--
	-- If Cosmos is installed, add a button to the Cosmos page to activate CensusPlus
	--
	if ( Earth) then
		EarthFeature_AddButton(
			{
				id = "CensusPlus";
				name = CENSUSPlus_BUTTON_TEXT;
				subtext = CENSUSPlus_BUTTON_SUBTEXT;
				tooltip = CENSUSPlus_BUTTON_TIP;
				icon = "Interface\\AddOns\\CensusPlus\\Skin\\CensusPlus_Icon";
				callback = CensusPlus_Toggle;
			}
		);
	elseif ( Cosmos_RegisterButton ) then 
			Cosmos_RegisterButton(CENSUSPlus_BUTTON_TEXT, CENSUSPlus_BUTTON_SUBTEXT, CENSUSPlus_BUTTON_TIP, "Interface\\AddOns\\CensusPlus\\Skin\\CensusPlus_Icon", CensusPlus_Toggle);
	end
end


----------------------------------------------------------------------------------
--
-- Called when the main window is shown
-- 
---------------------------------------------------------------------------------
function CensusPlus_OnShow()
	-- Initialize if this is the first OnShow event
	if (g_CensusPlusInitialized == false) then
		g_CensusPlusInitialized = true;
	end
	CensusPlus_UpdateView();
end

----------------------------------------------------------------------------------
--
-- Toggle hidden status
-- 
---------------------------------------------------------------------------------
function CensusPlus_Toggle() 
	if ( CensusPlus:IsVisible() ) then 
		CensusPlus:Hide();
	else
		CensusPlus:Show();
	end
end

----------------------------------------------------------------------------------
--
-- Toggle options pane
-- 
---------------------------------------------------------------------------------
function CensusPlus_ToggleOptions() 
	if ( CP_OptionsWindow:IsVisible() ) then 
		CP_OptionsWindow:Hide();
	else
		CP_OptionsWindow:Show();
	end
end

-----------------------------------------------------------------------------------
--
-- Called once on load
--
-----------------------------------------------------------------------------------
function CensusPlus_OnLoad()
	--
	-- Update the version number
	--
	CensusPlusText:SetText("Census+ v"..CensusPlus_VERSION .. " " .. g_CensusPlusLocale );
    CensusPlusText2:SetText( CENSUSPlus_UPLOAD );

	--
	-- Init constant tables
	--
	InitConstantTables();

	--
	-- Register with Cosmos, if it is installed
	--
	CensusPlus_RegisterCosmos();
	
	--
	-- Register for events
	--
	this:RegisterEvent("VARIABLES_LOADED");
	this:RegisterEvent("WHO_LIST_UPDATE");
--	this:RegisterEvent("GUILD_ROSTER_SHOW");
--	this:RegisterEvent("GUILD_ROSTER_UPDATE");
--	this:RegisterEvent("TRAINER_SHOW");
--	this:RegisterEvent("TRAINER_CLOSED");
--	this:RegisterEvent("MERCHANT_SHOW");
--	this:RegisterEvent("MERCHANT_CLOSED");
--	this:RegisterEvent("GUILD_REGISTRAR_SHOW");
--	this:RegisterEvent("GUILD_REGISTRAR_CLOSED");
--	this:RegisterEvent("TRADE_SHOW");
--	this:RegisterEvent("TRADE_CLOSED");
--	this:RegisterEvent("AUCTION_HOUSE_SHOW");
--	this:RegisterEvent("AUCTION_HOUSE_CLOSED");
--	this:RegisterEvent("BANKFRAME_OPENED");
--	this:RegisterEvent("BANKFRAME_CLOSED");

--	this:RegisterEvent("QUEST_GREETING");
--	this:RegisterEvent("QUEST_DETAIL");
--	this:RegisterEvent("QUEST_PROGRESS");
--	this:RegisterEvent("QUEST_COMPLETE");
--	this:RegisterEvent("QUEST_FINISHED");
--	this:RegisterEvent("QUEST_ITEM_UPDATE");
	
--	this:RegisterEvent("QUEST_ACCEPT_CONFIRM");
--	this:RegisterEvent("QUEST_LOG_UPDATE");
	
	this:RegisterEvent("UNIT_FOCUS");	
	this:RegisterEvent("PLAYER_TARGET_CHANGED" );
	this:RegisterEvent("UPDATE_MOUSEOVER_UNIT");	
		
	this:RegisterEvent("PLAYER_PVP_KILLS_CHANGED");		
	this:RegisterEvent("INSPECT_HONOR_UPDATE");		

	this:RegisterEvent("ZONE_CHANGED_NEW_AREA");		

	--
	-- Register a slash command
	--
	SLASH_CensusPlusCMD1 = "/Census";
        SLASH_CensusPlusCMD2 = "/CensusPlus";
	SLASH_CensusPlusCMD3 = "/Census+";
        SLASH_CensusPlusCMD4 = "/인구조사";
	SlashCmdList["CensusPlusCMD"] = CensusPlus_Command;

    SLASH_CensusPlusVerbose1 = "/censusverbose";	
	SlashCmdList["CensusPlusVerbose"] = CensusPlus_Verbose;
	
	--
	--  Set the auto close to true
	--
	CensusPlus_AutoCloseWho( 1 );
	--AutoClose:SetChecked( 1 );

	g_Pre_FriendsFrameOnHideOverride = FriendsFrame_OnHide;
	FriendsFrame_OnHide = CensusPlus_FriendsFrame_OnHide;
	
	g_Pre_FriendsFrameOnShowOverride = FriendsFrame_OnShow;
	FriendsFrame_OnShow = CensusPlus_FriendsFrame_OnShow;
	
	g_Pre_WhoList_UpdateOverride = WhoList_Update;
	WhoList_Update = CensusPlus_WhoList_Update;	
	
	g_Pre_FriendsFrame_Update  = FriendsFrame_Update;
	FriendsFrame_Update = CensusPlus_FriendsFrame_Update;

	g_Pre_WhoHandler = SlashCmdList["WHO"];
	SlashCmdList["WHO"] = CensusPlus_WhoHandler;
	
	CensusPlus_CheckForBattleground();

end

-----------------------------------------------------------------------------------
--
-- Load Handler for options box
--
-----------------------------------------------------------------------------------
function CP_OptionsOnShow()
	CP_OptionAutoClose:SetChecked(g_WhoAutoClose);
	CP_OptionVerboseButton:SetChecked(CensusPlus_Database["Info"]["Verbose"]);
end

-----------------------------------------------------------------------------------
--
-- CensusPlus Friends Frame override to stop the window close sound
--
-----------------------------------------------------------------------------------
function CensusPlus_FriendsFrame_OnHide()
	g_Pre_FriendsFrameOnHideOverride();
end

-----------------------------------------------------------------------------------
--
-- CensusPlus Friends Frame override to stop the window close sound
--
-----------------------------------------------------------------------------------
function CensusPlus_FriendsFrame_OnShow()
	g_Pre_FriendsFrameOnShowOverride();
end

function CensusPlus_WhoList_Update()
	if( g_IsCensusPlusInProgress == true and g_WhoAutoClose ) then
		local numWhos, totalCount = GetNumWhoResults();
		local name, guild, level, race, class, zone, group;
		local button;
		local columnTable;
		local whoOffset = FauxScrollFrame_GetOffset(WhoListScrollFrame);
		local whoIndex;
		local showScrollBar = nil;
		if ( numWhos > WHOS_TO_DISPLAY ) then
			showScrollBar = 1;
		end
		local displayedText = "";
		if ( totalCount > MAX_WHOS_FROM_SERVER ) then
			displayedText = format(WHO_FRAME_SHOWN_TEMPLATE, MAX_WHOS_FROM_SERVER);
		end
		WhoFrameTotals:SetText(format(GetText("WHO_FRAME_TOTAL_TEMPLATE", nil, totalCount), totalCount).."  "..displayedText);
		for i=1, WHOS_TO_DISPLAY, 1 do
			whoIndex = whoOffset + i;
			button = getglobal("WhoFrameButton"..i);
			button.whoIndex = whoIndex;
			name, guild, level, race, class, zone, group = GetWhoInfo(whoIndex);
			columnTable = { zone, guild, race };
			getglobal("WhoFrameButton"..i.."Name"):SetText(name);
			getglobal("WhoFrameButton"..i.."Level"):SetText(level);
			getglobal("WhoFrameButton"..i.."Class"):SetText(class);
			local variableText = getglobal("WhoFrameButton"..i.."Variable");
			variableText:SetText(columnTable[UIDropDownMenu_GetSelectedID(WhoFrameDropDown)]);
			if ( not group  ) then
				group = "";
			end
			--getglobal("WhoFrameButton"..i.."Group"):SetText(getglobal(strupper(group)));
			
			-- If need scrollbar resize columns
			if ( showScrollBar ) then
				variableText:SetWidth(95);
			else
				variableText:SetWidth(110);
			end

			-- Highlight the correct who
			if ( WhoFrame.selectedWho == whoIndex ) then
				button:LockHighlight();
			else
				button:UnlockHighlight();
			end
			
			if ( whoIndex > numWhos ) then
				button:Hide();
			else
				button:Show();
			end
		end

		if ( not WhoFrame.selectedWho ) then
			WhoFrameGroupInviteButton:Disable();
			WhoFrameAddFriendButton:Disable();
		else
			WhoFrameGroupInviteButton:Enable();
			WhoFrameAddFriendButton:Enable();
			WhoFrame.selectedName = GetWhoInfo(WhoFrame.selectedWho); 
		end

		-- If need scrollbar resize columns
		if ( showScrollBar ) then
			WhoFrameColumn_SetWidth(105, WhoFrameColumnHeader2);
			UIDropDownMenu_SetWidth(80, WhoFrameDropDown);
		else
			WhoFrameColumn_SetWidth(120, WhoFrameColumnHeader2);
			UIDropDownMenu_SetWidth(95, WhoFrameDropDown);
		end

		-- ScrollFrame update
		FauxScrollFrame_Update(WhoListScrollFrame, numWhos, WHOS_TO_DISPLAY, FRIENDS_FRAME_WHO_HEIGHT );
	else
		g_Pre_WhoList_UpdateOverride();
	end
end


function CensusPlus_FriendsFrame_Update()
	if ( FriendsFrame.selectedTab == 3 and g_IsCensusPlusInProgress == true and g_WhoAutoClose ) then
		FriendsFrameTopLeft:SetTexture("Interface\\ClassTrainerFrame\\UI-ClassTrainer-TopLeft");
		FriendsFrameTopRight:SetTexture("Interface\\ClassTrainerFrame\\UI-ClassTrainer-TopRight");
		FriendsFrameBottomLeft:SetTexture("Interface\\FriendsFrame\\GuildFrame-BotLeft");
		FriendsFrameBottomRight:SetTexture("Interface\\FriendsFrame\\GuildFrame-BotRight");
		local guildName;
		guildName = GetGuildInfo("player");
		FriendsFrameTitleText:SetText(guildName);
		FriendsFrame_ShowSubFrame("GuildFrame");
	else
		g_Pre_FriendsFrame_Update();
	end
end

-----------------------------------------------------------------------------------
--
-- CensusPlus Who Handler
--
-----------------------------------------------------------------------------------
function CensusPlus_WhoHandler( msg )
	if( g_IsCensusPlusInProgress == true ) then
		if ( msg == "" ) then
			msg = WhoFrame_GetDefaultWhoCommand();
			ShowWhoPanel();
		elseif ( msg == "cheat" ) then
			-- Remove the "cheat" part later!
			ShowWhoPanel();
		end
		SendWho(msg);
	else
		g_Pre_WhoHandler(msg);
	end
end

-----------------------------------------------------------------------------------
--
-- CensusPlus command
--
-----------------------------------------------------------------------------------
function CensusPlus_Command( param )
    local  firsti, lasti, command, value = string.find (param, "(%w+) (%w+)") ;

--	if( string.lower(param) == "locale" ) then
--		CP_EU_US_Version:Show();
--	else
	if( string.lower(param) == "options" ) then
		CP_OptionsWindow:Show();
--	elseif( string.lower(param) == "tz" ) then
--		CensusPlus_DetermineServerDate();
	elseif( string.lower(param) == "bufftest" ) then
		showAllUnitBuffs("player");
	elseif( string.lower(param) == "verbose" ) then
		CensusPlus_Verbose();
	else
		CensusPlus_DisplayUsage();
	end
end

-----------------------------------------------------------------------------------
--
-- CensusPlus Display Usage
--
-----------------------------------------------------------------------------------
function CensusPlus_DisplayUsage()
    local text;

	CensusPlus:Show();

--[[
    CensusPlus_Msg("Usage:\n  /CensusPlus \n");
    CensusPlus_Msg("  /censusPlus verbose Toggle verbose mode off/on\n");
--    CensusPlus_Msg("  /CensusPlus locale  Bring up the locale selection dialog - (WARNING -- CHANGING YOUR LOCALE WILL PURGE YOUR DATABASE)\n");
    CensusPlus_Msg("  /CensusPlus options Bring up the Option window\n");
    CensusPlus_Msg("  /CensusPlus prune X Prune the database by removing characters not seen in X days\n");
    CensusPlus_Msg("  /CensusPlus serverprune Prune the database by removing all data from servers other than the one you are currently on.\n");
]]

    CensusPlus_Msg("Usage:\n  /CensusPlus, /census+, /census, /인구조사 \n");
    CensusPlus_Msg("  /censusPlus verbose 대화모드를 켜거나 끕니다. \n");
    CensusPlus_Msg("  /CensusPlus options 옵션 창을 엽니다. \n");

end

-----------------------------------------------------------------------------------
--
-- CensusPlus Verbose option
--
-----------------------------------------------------------------------------------
function CensusPlus_Verbose()
    if( CensusPlus_Database["Info"]["Verbose"] == true ) then
        CensusPlus_Msg( "Verbose Mode : OFF" );
        CensusPlus_Database["Info"]["Verbose"] = false;
	else
        CensusPlus_Msg( "Verbose Mode : ON" );
        CensusPlus_Database["Info"]["Verbose"] = true;
    end
end

-----------------------------------------------------------------------------------
--
-- Minimize the window
--
-----------------------------------------------------------------------------------
function CensusPlus_OnClickMinimize()
    if( CensusPlus:IsVisible() ) then
        MiniCensusPlus:Show();
        CensusPlus:Hide();
    end
end

-----------------------------------------------------------------------------------
--
-- Minimize the window
--
-----------------------------------------------------------------------------------
function CensusPlus_OnClickMaximize()
    if( MiniCensusPlus:IsVisible() ) then
        MiniCensusPlus:Hide();
        CensusPlus:Show();
    end
end

-----------------------------------------------------------------------------------
--
-- Take or pause a census depending on current status
--
-----------------------------------------------------------------------------------
function CensusPlus_Take_OnClick()
	if (g_IsCensusPlusInProgress) then
	    CensusPlus_TogglePause();
	else
		CensusPlus_StartCensus();
	end
end

-----------------------------------------------------------------------------------
--
-- Display a tooltip for the take button
--
-----------------------------------------------------------------------------------
function CensusPlus_Take_OnEnter()
	if (g_IsCensusPlusInProgress) then
		if (g_CensusPlusManuallyPaused) then
			GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
			GameTooltip:SetText(CENSUSPlus_UNPAUSECENSUS, 1.0, 1.0, 1.0);
			GameTooltip:Show();					
		else
			GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
			GameTooltip:SetText(CENSUSPlus_PAUSECENSUS, 1.0, 1.0, 1.0);
			GameTooltip:Show();					
		end
	else
		GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
		GameTooltip:SetText(CENSUSPlus_TAKECENSUS, 1.0, 1.0, 1.0);
		GameTooltip:Show();					
	end
end

-----------------------------------------------------------------------------------
--
-- Pause the current census
--
-----------------------------------------------------------------------------------
function CensusPlus_TogglePause()
	if (g_IsCensusPlusInProgress == true) then
	    if( g_CensusPlusManuallyPaused == true ) then
	        CensusPlusTakeButton:SetText( CENSUSPlus_PAUSE );
            g_CensusPlusManuallyPaused = false;
	    else
	        CensusPlusTakeButton:SetText( CENSUSPlus_UNPAUSE );
            g_CensusPlusManuallyPaused = true;
        end
    end
end
-----------------------------------------------------------------------------------
--
-- Purge the database for this realm and faction
--
-----------------------------------------------------------------------------------
function CensusPlus_Purge()
	if( CensusPlus_Database["Servers"] ~= nil ) then
		CensusPlus_Database["Servers"] = nil;
	end
	CensusPlus_Database["Servers"] = {};
	CensusPlus_UpdateView();
	CensusPlus_Msg(CENSUSPlus_PURGEMSG);	
	
	if( CensusPlus_Database["Guilds"] ~= nil ) then
		CensusPlus_Database["Guilds"] = nil;
	end
	CensusPlus_Database["Guilds"] = {};

	if( CensusPlus_Database["TimesPlus"] ~= nil ) then
		CensusPlus_Database["TimesPlus"] = nil;
	end
	CensusPlus_Database["TimesPlus"] = {};
	
end


-----------------------------------------------------------------------------------
--
-- Handler for auto close checkbox
--
-----------------------------------------------------------------------------------
function CensusPlus_AutoCloseWho(close)
    g_WhoAutoClose = close;
end

-----------------------------------------------------------------------------------
--
-- Take a CensusPlus
--
-----------------------------------------------------------------------------------
function CensusPlus_StartCensus()

    if (g_IsCensusPlusInProgress) then
        if( g_CensusPlusManuallyPaused == true ) then
            g_CensusPlusManuallyPaused = false;
            CensusPlusPauseButton:SetText( CENSUSPlus_PAUSE );
        else
            -- Do not initiate a new CensusPlus while one is in progress
            CensusPlus_Msg(CENSUSPlus_ISINPROGRESS);
        end
    elseif( g_CurrentlyInBG ) then
        g_LastCensusRun = time()-600;    
        CensusPlus_Msg(CENSUSPlus_ISINBG);
    else
        -- who 결과를 ui로 보냄
        SetWhoToUI(1);

        --
        -- Initialize the job queue and counters
        --
        CensusPlus_Msg(CENSUSPlus_TAKINGONLINE);
        g_NumNewCharacters = 0;
        g_NumUpdatedCharacters = 0;
        g_JobQueue = {};

        g_TempCount = nil;
        g_TempCount = {};

        -- timestamp 초기화
        g_TimeStamp = nil;
        CensusKr_setTimeStamp(); 

        -- 진행상태
        g_ProcessedZone = {};

        local counter = 0;
        -- 레벨 대신 지역 별로 조회
        --[[
        for outer = 0, 10, 1 do
        local job = {m_MinLevel=outer*5+1, m_MaxLevel=outer*5+5};
        InsertJobIntoQueue(job);
        end
        local job = {m_MinLevel=56, m_MaxLevel=59};
        InsertJobIntoQueue(job);
        job = {m_MinLevel=60, m_MaxLevel=60};
        InsertJobIntoQueue(job);
        ]]
        local zoneLetters = GetZoneLetters();
        g_ProcessZoneCount = table.getn(zoneLetters); -- 진행 상태 처리용.
        local minLevel = 1;
        local maxLevel = MAX_CHARACTER_LEVEL; -- 60
        for i=1, table.getn(zoneLetters), 1 do
        local job = {m_zoneLetter = zoneLetters[i], m_MinLevel = minLevel, m_MaxLevel = maxLevel};
        InsertJobIntoQueue(job);
        end

        g_IsCensusPlusInProgress = true;
        g_WaitingForWhoUpdate = false;
        g_CensusPlusManuallyPaused = false;

        local hour, minute = GetGameTime();
        g_TakeHour = hour;
        g_ResetHour = true;
    end

    CensusPlusTakeButton:SetText( CENSUSPlus_PAUSE );
end

-----------------------------------------------------------------------------------
--
-- Stop a CensusPlus
--
-----------------------------------------------------------------------------------
function CensusPlus_StopCensus(  )
	if (g_IsCensusPlusInProgress) then
        CensusPlusTakeButton:SetText( CENSUSPlus_TAKE );
        g_CensusPlusManuallyPaused = false;
        
		g_JobQueue = {};
		g_JobQueue = nil;
		g_JobQueue = {};


		CensusPlus_DisplayResults( );
	else
		CensusPlus_Msg(CENSUSPlus_NOCENSUS);
	end
end

-----------------------------------------------------------------------------------
--
-- Display Census results
--
-----------------------------------------------------------------------------------
function CensusPlus_DisplayResults(  )
	--
	-- We are all done, report our results
	--
	g_IsCensusPlusInProgress = false;
	--CensusPlus_UpdateView();
	
        -- 자신의 정보 추가하는 루틴. 이거말고 대체할 방법 고려.
	--CensusPlus_ProcessMyHonor();

	CensusPlus_Msg(format(CENSUSPlus_FINISHED, g_NumUpdatedCharacters));
	ChatFrame1:AddMessage(CENSUSPlus_UPLOAD, 0.5, 1.0, 1.0);
	
	CensusPlus_UpdateView();
	g_LastCensusRun = time();

        -- who 결과 ui 반응을 원래 상태로 돌림. 
        SetWhoToUI(0);

	
    CensusPlusTakeButton:SetText( CENSUSPlus_TAKE );
end

-----------------------------------------------------------------------------------
--
-- Create a who command text for the input job
--
-----------------------------------------------------------------------------------
function CensusPlus_CreateWhoText(job)
	local whoText = "";
	local race = job.m_Race;
	if (race ~= nil) then
		whoText = whoText.." r-\""..race.."\"";
	end

	local class = job.m_Class;
	if (class ~= nil) then
		whoText = whoText.." c-\""..class.."\"";
	end

	local minLevel = job.m_MinLevel;
	if (minLevel == nil) then
		minLevel = 1;
	end
	local maxLevel = job.m_MaxLevel;
	if (maxLevel == nil) then
		maxLevel = 60;
	end
	whoText = whoText.." "..minLevel.."-"..maxLevel;

	local zoneLetter = job.m_zoneLetter;
	if ( zoneLetter ~= nil) then
		whoText = whoText.." z-\""..zoneLetter.."\"";
	end

	--[[ 한글처리 안됨
	local letter = job.m_Letter;
	if( letter ~= nil ) then
		whoText = whoText.." n-"..letter;
	end
	--]]
	return whoText;
end

function CensusKr_getProcess()
    return math.ceil(table.getn(g_ProcessedZone) / g_ProcessZoneCount * 100) .. "%";
end
-----------------------------------------------------------------------------------
--
-- Called on events
--
-----------------------------------------------------------------------------------
function CensusPlus_OnEvent(event,arg1,arg2,arg3,arg4,arg5,arg6,arg7,arg8,arg9)

	if( arg1 == nil ) then
		arg1 = "nil"
	end
	if( arg2 == nil ) then
		arg2 = "nil"
	end
	if( arg3 == nil ) then
		arg3 = "nil"
	end
	if( arg4 == nil ) then
		arg4 = "nil"
	end
		
	--
	-- If we have not been initialized, do nothing
	--
	if (g_CensusPlusInitialized == false) then
		return
	end
	
--	CensusPlus_Msg( "Message =>" .. event );
	
	--
	-- WHO_LIST_UPDATE
	--
	if( event == "TRAINER_SHOW" or event == "MERCHANT_SHOW" or event == "TRADE_SHOW" or event == "GUILD_REGISTRAR_SHOW" 
	            or event == "AUCTION_HOUSE_SHOW" or event == "BANKFRAME_OPENED" or event == "QUEST_DETAIL" ) then
	    if( g_IsCensusPlusInProgress ) then
	        g_CensusPlusPaused = true;
	    end
	elseif( event == "TRAINER_CLOSED" or event == "MERCHANT_CLOSED" or event == "TRADE_CLOSED" or event == "GUILD_REGISTRAR_CLOSED" 
	            or event == "AUCTION_HOUSE_CLOSED" or event == "BANKFRAME_CLOSED" or event == "QUEST_FINISHED" ) then
	    if( g_IsCensusPlusInProgress ) then
	        g_CensusPlusPaused = false;
	    end
	elseif(event == "WHO_LIST_UPDATE") then
		--
		-- Only process who results if a CensusPlus is in progress
		--
		if (g_IsCensusPlusInProgress) then
			local numWhoResults = GetNumWhoResults();
			--CensusKr_Print("debug: WHO_LIST_UPDATE fired. ret - " .. numWhoResults);
			CensusPlus_CheckForBattleground();
			CensusPlus_ProcessWhoResults();
			if (numWhoResults > MAX_WHO_RESULTS) then
				--
				-- Who list is overflowed, split the query to make the return smaller
                                -- 지역/종족/직업/레벨
				--
				local minLevel = g_CurrentJob.m_MinLevel;
				local maxLevel = g_CurrentJob.m_MaxLevel;
				local race = g_CurrentJob.m_Race;
				local class = g_CurrentJob.m_Class;
				local zoneLetter = g_CurrentJob.m_zoneLetter;
				local letter = g_CurrentJob.m_Letter;
                                if (zoneLetter == nil) then
                                        --
                                        -- This job does not specify zone, so split it that way, making more jobs
                                        --
                                        local zoneLetters = GetZoneLetters();
                                        for i=1, table.getn(zoneLetters), 1 do
                                                local job = {m_zoneLetter = zoneLetters[i], m_MinLevel = minLevel, m_MaxLevel = maxLevel};
                                                InsertJobIntoQueue(job);
                                        end
                                else
                                    --
                                    -- We cannot split the level range any more
                                    --
                                    local factionGroup = UnitFactionGroup("player");
                                    if (race == nil) then
                                            --
                                            -- This job does not specify race, so split it that way, making four new jobs
                                            --
                                            local thisFactionRaces = GetFactionRaces(factionGroup);
                                            local numRaces = table.getn(thisFactionRaces);
                                            for i = 1, numRaces, 1 do
                                                    local job = {m_zoneLetter = zoneLetter, m_Race = thisFactionRaces[i], m_MinLevel = minLevel, m_MaxLevel = maxLevel};
                                                    InsertJobIntoQueue(job);
                                            end	
                                    else             
                                        if (class == nil) then
                                                --
                                                -- This job does not specify class, so split it that way, making more jobs
                                                --
                                                local thisRaceClasses = GetRaceClasses(race);
                                                local numClasses = table.getn(thisRaceClasses);
                                                for i = 1, numClasses, 1 do
                                                        local job = {m_zoneLetter = zoneLetter, m_Race = race, m_Class = thisRaceClasses[i], m_MinLevel = minLevel, m_MaxLevel = maxLevel};
                                                        InsertJobIntoQueue(job);
                                                end	
                                        else                               
                                            if (minLevel ~= maxLevel) then
                                                    --
                                                    -- The level range is greater than a single level, so split it in half and submit the two jobs
                                                    --
                                                    local pivot = floor((minLevel + maxLevel) / 2);
                                                    local jobLower = {m_zoneLetter = zoneLetter, m_Race = race, m_Class = class, m_MinLevel = minLevel, m_MaxLevel = pivot};
                                                    InsertJobIntoQueue(jobLower);
                                                    local jobUpper = {m_zoneLetter = zoneLetter, m_Race = race, m_Class = class, m_MinLevel = pivot + 1, m_MaxLevel = maxLevel};
                                                    InsertJobIntoQueue(jobUpper);
                                            else
                                                    --
                                                    -- There are too many characters with a single level, class, race and letter, give up
                                                    --
                                                    local whoText = CensusPlus_CreateWhoText(g_CurrentJob);
                                                    if( CensusPlus_Database["Info"]["Verbose"] == true ) then
                                                            CensusPlus_Msg(format(CENSUSPlus_TOOMANY, whoText));
                                                    end
                                            end -- end level
                                        end -- end class
                                    end -- end race
                                end -- end zone
                            end
		else
		    --
		    --  This is just a random /who done by the player
		    --  인구조사 진행 중인 경우가 아니면 조사 하지 않음.
			--CensusPlus_ProcessWhoResults();
		end
		--
		-- We got the who update
		--
		g_WaitingForWhoUpdate = false;
	elseif ( event == "VARIABLES_LOADED" ) then
	    --
	    --  Initialize our variables
	    --
	    CensusPlus_InitializeVariables();
--[[		
	elseif ((event == "UNIT_FOCUS" or event=="PLAYER_TARGET_CHANGED" )and arg1 ~= "player") then
		if( UnitIsPlayer( "target" ) and not UnitIsUnit("player", "target") ) then
			if( g_CensusPlusLastTarget ~= nil ) then
				ClearInspectPlayer();
				g_CensusPlusLastTargetName = nil;
				g_CensusPlusLastTarget = nil;
				if( CensusPlus_IsInspectLoaded() ) then
					InspectFrame.unit = nil;
				end
			end
			CensusPlus_ProcessTarget("target");
		end
		]]--
	elseif( event == "ZONE_CHANGED_NEW_AREA" ) then
		--
		--  We need to check to see if we entered a battleground
		--
		CensusPlus_CheckForBattleground();
	end
end

-----------------------------------------------------------------------------------
--
-- ProcessTarget --  called when UNIT_FOCUS event is fired
--
-----------------------------------------------------------------------------------
function CensusPlus_ProcessTarget( unit )
        CensusKr_Print("ProcessTarget() invoked");
	if ( UnitIsPlayer(unit) == nil or (not UnitIsPlayer(unit)) or unit == "player" or unit == nil ) then
		return;
	end
	
	local sightingData = CensusPlus_CollectSightingData( unit );
	if( sightingData == nil or sightingData.faction == nil ) then
		return
	end

	if (sightingData ~= nil and (sightingData.faction == "Alliance" or sightingData.faction == "Horde")) then
	
		--
		--  Do a quick check to see if this is an MC'd person
		--
		if( sightingData.faction ~= g_FactionCheck[sightingData.race] ) then
			return;
		end
	
	
		if( sightingData.guild == nil ) then
			sightingData.guild = "";
		end
		--
		-- Get the portion of the database for this server
		--
		local realmName = g_CensusPlusLocale .. GetCVar("realmName");
		local realmDatabase = CensusPlus_Database["Servers"][realmName];
		if (realmDatabase == nil) then
			CensusPlus_Database["Servers"][realmName] = {};
			realmDatabase = CensusPlus_Database["Servers"][realmName];
		end

		--
		-- Get the portion of the database for this faction
		--
		local factionDatabase = realmDatabase[sightingData.faction];
		if (factionDatabase == nil) then
			realmDatabase[sightingData.faction] = {};
			factionDatabase = realmDatabase[sightingData.faction];
		end
		
		--
		-- Get racial database
		--
		local raceDatabase = factionDatabase[sightingData.race];
		if (raceDatabase == nil) then
			factionDatabase[sightingData.race] = {};
			raceDatabase = factionDatabase[sightingData.race];
		end

		--
		-- Get class database
		--
		local classDatabase = raceDatabase[sightingData.class];
		if (classDatabase == nil) then
			raceDatabase[sightingData.class] = {};
			classDatabase = raceDatabase[sightingData.class];
		end

		--
		-- Get this player's entry
		--
		local entry = classDatabase[sightingData.name];
		if (entry == nil) then
			classDatabase[sightingData.name] = {};
			entry = classDatabase[sightingData.name];
		end
		
		--
		-- Update the information
		--
		entry[1] = sightingData.level;
		entry[2] = sightingData.guild;
		entry[3] = CensusPlus_DetermineServerDate() .. "";

		entry[7] = CensusPlus_DetermineServerDate() .. "";
		
		--
		--  Update their rank info
		--
		rankNumber = UnitPVPRank(unit);
		if( rankNumber ~= 0 ) then
--			rankName= GetPVPRankInfo( rankNumber ) 		
			entry[8] = rankNumber;  --  slot 8 will be current rank
		else
			entry[8] = 0;  --  slot 8 will be current rank
		end
		
		--
		--  Trigger an update on the inspect honor frame to get honor information
		--		
		if ( UnitIsPlayer( unit ) and sightingData.level >= 30  
				and CheckInteractDistance( unit, 1) 
				and not UnitIsUnit("player", unit) and unit == "target" 
				and g_CensusPlusLastTarget == nil 
				and g_CensusPlusLastTargetName == nil 
				and CensusPlus_IsInspectLoaded() ) then
			NotifyInspect(unit);
			InspectFrame.unit = unit;
			if ( not HasInspectHonorData() ) then
				g_CensusPlusLastTarget = entry;
			    g_CensusPlusLastTargetName = sightingData.name; 
				RequestInspectHonorData();
			else
				InspectHonorFrame_Update();
			end
		end		
	end
end


-----------------------------------------------------------------------------------
--
-- Gather targeting data
--
-----------------------------------------------------------------------------------
function CensusPlus_CollectSightingData(unit)
	if ( UnitIsPlayer(unit) and UnitName(unit) ~= "Unknown Entity" ) then
		return {
			name=UnitName(unit), 
			level=UnitLevel(unit),
			sex=UnitSex(unit), 
			race=UnitRace(unit),
			class=UnitClass(unit), 
			guild=GetGuildInfo(unit),
			faction=UnitFactionGroup(unit) 
		};
	else
		return nil;
	end
end


-----------------------------------------------------------------------------------
--
-- Initialize our primary save variables --  called when VARIABLES_LOADED event is fired
--
-----------------------------------------------------------------------------------
function CensusPlus_InitializeVariables()
    -- 
	--  Check for the mini-start
	--
    
    if( CensusPlus_Database["Servers"] == nil ) then 
            CensusPlus_Database["Servers"] = {};
    end	    
    
        --[[
	if( CensusPlus_Database["Times"] ~= nil ) then
		CensusPlus_Database["Times"] = nil;
	end
        ]]

        -- 조사 시간대별로 결과 기록
        if( CensusPlus_Database["Times"] == nil ) then
		CensusPlus_Database["Times"] = {};
	end

	if( CensusPlus_Database["TimesPlus"] == nil ) then
	    CensusPlus_Database["TimesPlus"] = {};
	end
	
	--
	--  Times should now be initted, so let's sort them
	--
	table.sort( CensusPlus_Database["TimesPlus"] );

    --
    --  Make sure info is last so it will be first in the output so we can grab the version number
    --
	if( CensusPlus_Database["Info"] == nil ) then
	    CensusPlus_Database["Info"] = {};
	    CensusPlus_Database["Info"]["Verbose"] = true;
	end

        -- 이전 버젼인 경우. 저장기록 초기화. 
        if (CensusPlus_Database["Info"]["DataVersion"] == nil) then
            CensusPlus_Msg("자료구조가 변경되어서 이전 버젼에서 작성된 조사기록을 초기화합니다.");
            CensusPlus_Purge();
        end

    CensusPlus_Database["Info"]["Version"] = CensusPlus_VERSION;
    CensusPlus_Database["Info"]["DataVersion"] = CensusKr_DataVersion;
    CensusPlus_Database["Info"]["ClientLocale"] = GetLocale();
	if( CensusPlus_Database["Info"]["ClientLocale"] == "enUS" and GetCVar("realmList") == "eu.logon.worldofwarcraft.com" ) then
	    CensusPlus_Database["Info"]["ClientLocale"] = "enGB";
	end
	if( CensusPlus_Database["Info"]["LoginServer"] ~= nil ) then
		--  already present, make sure it equals, and if
		--		not, force a purge
		if( CensusPlus_Database["Info"]["LoginServer"] ~= GetCVar("realmList") ) then
			--[[
				We have to nuke the data in the case that someone is playing on both
				US and EU servers
			]]--
			CensusPlus_Purge()
		end
	end
    CensusPlus_Database["Info"]["LoginServer"] = GetCVar("realmList");

    local firstVersionRun = CensusPlus_Database["Info"][g_InterfaceVersion];
    local localeSetting = CensusPlus_Database["Info"]["Locale"];
    if( localeSetting == "??" ) then
		--  We had problems previously.. we must purge =(
		CensusPlus_Purge();
		localeSetting = nil;
    end
    
    if( localeSetting == nil ) then
		--
		--  Have a new way to detect locale, yay!
		--
		if( CensusPlus_Database["Info"]["ClientLocale"] == "enUS" ) then
			CensusPlus_Database["Info"]["Locale"] = "US";
		elseif( CensusPlus_Database["Info"]["ClientLocale"] == "enGB" ) then
			CensusPlus_Database["Info"]["Locale"] = "EU";
		elseif( CensusPlus_Database["Info"]["ClientLocale"] == "frFR" ) then
			CensusPlus_Database["Info"]["Locale"] = "EU";
		elseif( CensusPlus_Database["Info"]["ClientLocale"] == "deDE" ) then
			CensusPlus_Database["Info"]["Locale"] = "EU";
		elseif( CensusPlus_Database["Info"]["ClientLocale"] == "koKR" ) then
			CensusPlus_Database["Info"]["Locale"] = "KR";
		else
			CensusPlus_Database["Info"]["Locale"] = "??";
		end
    end
    
    if( firstVersionRun == nil and g_InterfaceVersion == 10900.2 ) then
		--[[
			Unfortunately we have to purge all our PVP data due to data problems
		]]--
		CensusPlus_Database["TimesPlus"] = nil;
		CensusPlus_Database["TimesPlus"] = {};		
--		CensusPlus_Msg( "Due to a data discrepancy, all prior data has been purged, sorry!" );
--		CensusPlus_Purge();
		CensusPlus_Database["Info"][g_InterfaceVersion] = true;
    end
    
    local locale = CensusPlus_Database["Info"]["Locale"];
    if( locale ~= nil ) then
		CensusPlus_SelectLocale( CensusPlus_Database["Info"]["Locale"], true );
    end
    
	local miniStart = CensusPlus_Database["Info"]["MiniStart"];
	if( miniStart == nil ) then
	    miniStart = 0;
	end
	
	if( CensusPlus_Database["Info"]["CensusButtonPosition"] == nil ) then
	    CensusPlus_Database["Info"]["CensusButtonPosition"] = 370;
	end
	
	if( CensusPlus_Database["Info"]["CensusButtonShown"] == nil ) then
	    CensusPlus_Database["Info"]["CensusButtonShown"] = 1;
	end
	
	if( CensusPlus_Database["Info"]["CensusButtonShown"] == 1 ) then
		CensusButtonFrame:Show();
	else
		CensusButtonFrame:Hide();
	end
	
    if( miniStart ) then
		CensusPlus_Msg(" V"..CensusPlus_VERSION..CENSUSPlus_MSG1);
    end
    
    g_VariablesLoaded = true;
    
    CensusPlus_CheckTZ();
    
    InitConstantTables();
    
    CP_OptionAutoShowMinimapButton:SetChecked(CensusPlus_Database["Info"]["CensusButtonShown"]);
    CP_SliderButtonPos:SetValue(CensusPlus_Database["Info"]["CensusButtonPosition"]);
    

end

-----------------------------------------------------------------------------------
--
-- Call on the update event
--
-----------------------------------------------------------------------------------
function CensusPlus_OnUpdate()
        -- autocensus 옵션 기능. autocensus가 활성화 되어 있으면 자동으로 인구조사 시작. 
        -- kr5에서 막음. 
        --[[
	if( g_VariablesLoaded and g_IsCensusPlusInProgress == false and g_LastCensusRun < time() - 1800 ) then
		CensusPlus_Take_OnClick();
	end
        ]]
	
	if (g_IsCensusPlusInProgress == true and g_CensusPlusPaused == false and g_CensusPlusManuallyPaused == false ) then
		local now = GetTime();
		local delta = now - g_LastOnUpdateTime;
		if (delta > CensusPlus_UPDATEDELAY) then
			g_LastOnUpdateTime = now;
			if (g_WaitingForWhoUpdate == true) then
				--
				-- Resend /who command
				--
				g_WhoAttempts = g_WhoAttempts + 1;
				local whoText = CensusPlus_CreateWhoText(g_CurrentJob);
				--CensusKr_Print("debug: OnUpdate() invoked");
				--CensusKr_Print("debug: whoText - " .. whoText);
				--CensusKr_Print("debug: WhoAttempts - " .. g_WhoAttempts);
				if( CensusPlus_Database["Info"]["Verbose"] == true ) then
				    CensusPlus_Msg(CENSUSPlus_WAITING);
				end
				if( g_WhoAttempts < 3 ) then
					if( g_WhoAttempts == 2 ) then
						g_CurrentJob.m_MinLevel = g_CurrentJob.m_MinLevel - 2;
						whoText = CensusPlus_CreateWhoText(g_CurrentJob);
						SendWho(whoText);
					else
						SendWho(whoText);
					end
				else
				    g_WaitingForWhoUpdate = false;
				end
			else
				--
				-- Determine if there is any more work to do
				--
				local numJobs = table.getn(g_JobQueue);
				if (numJobs > 0) then
					--
					-- Remove the top job from the queue and send it
					--
					local job = g_JobQueue[numJobs];
					table.remove(g_JobQueue);
					local whoText = CensusPlus_CreateWhoText(job);
					g_CurrentJob = job;
					g_WaitingForWhoUpdate = true;
					if( CensusPlus_Database["Info"]["Verbose"] == true ) then
                                            -- 진행율
                                            if (g_ProcessedZone[g_CurrentJob.m_zoneLetter] == nil) then
                                                g_ProcessedZone[g_CurrentJob.m_zoneLetter] = g_CurrentJob.m_zoneLetter;
                                                g_ProcessedZone.n = table.getn(g_ProcessedZone) + 1;
                                            end
					    CensusPlus_Msg(format(CENSUSPlus_SENDING, whoText, CensusKr_getProcess()));
					end
					SendWho(whoText);
				    g_WhoAttempts = 0;
				else
					--
					-- We are all done, hide the friends frame and report our results
					--
					CensusPlus_DoTimeCounts();					
                                        --CensusKr_SaveTimesData();
					CensusPlus_DisplayResults();
				end
			end
		end
	end
end

-- 시간대별 조사기록 저장
function CensusKr_SaveTimesDate()
    local realm = GetRealmName();
    local faction = UnitFactionGroup("player");
    local hour, minute = GetGameTime();
    local date = CensusPlus_DetermineServerDate();
    local times = date .. "&" .. hour .. ":" .. minute .. ":00";
    if (CensusPlus_Database["Times"] == nil) then
        CensusPlus_Database["Times"] = {};
    end
    if (CensusPlus_Database["Times"][realm] == nil) then
        CensusPlus_Database["Times"][realm] = {};
    end
    if (CensusPlus_Database["Times"][realm][faction] == nil) then
        CensusPlus_Database["Times"][realm][faction] = {};
    end
    CensusKr_Print("g_TotalCount = " .. g_TotalCount);
    CensusKr_Print("g_TempCount = " .. table.getn(g_TempCount));
    CensusPlus_Database["Times"][realm][faction][times] = g_NumNewCharacters .. "&" ..g_NumUpdatedCharacters;
end

function CensusKr_Print(msg)
    if (DEFAULT_CHAT_FRAME) then
        DEFAULT_CHAT_FRAME:AddMessage(msg);
    end
end

-----------------------------------------------------------------------------------
--
-- Take final tally
--
-----------------------------------------------------------------------------------
function CensusPlus_DoTimeCounts()

	if( g_CurrentlyInBG == true ) then
		return;
	end

	--  Zero out the times
	g_TimeDatabase[CENSUSPlus_DRUID]		= 0;
	g_TimeDatabase[CENSUSPlus_HUNTER]		= 0;
	g_TimeDatabase[CENSUSPlus_MAGE]			= 0;
	g_TimeDatabase[CENSUSPlus_PRIEST]		= 0;
	g_TimeDatabase[CENSUSPlus_ROGUE]		= 0;
	g_TimeDatabase[CENSUSPlus_WARLOCK]	    = 0;
	g_TimeDatabase[CENSUSPlus_WARRIOR]	    = 0;
	g_TimeDatabase[CENSUSPlus_SHAMAN]		= 0;
	g_TimeDatabase[CENSUSPlus_PALADIN]	    = 0;
 
        g_TimeDatabase[CENSUSPlus_DWARF]        = 0;
        g_TimeDatabase[CENSUSPlus_GNOME]        = 0;
        g_TimeDatabase[CENSUSPlus_HUMAN]        = 0;
        g_TimeDatabase[CENSUSPlus_NIGHTELF]     = 0;
        g_TimeDatabase[CENSUSPlus_ORC]          = 0;
        g_TimeDatabase[CENSUSPlus_TAUREN]       = 0;
        g_TimeDatabase[CENSUSPlus_TROLL]                = 0;
        g_TimeDatabase[CENSUSPlus_UNDEAD]       = 0;	

	local realmName = g_CensusPlusLocale .. GetCVar("realmName");
	if( CensusPlus_Database["TimesPlus"][realmName] == nil ) then
		CensusPlus_Database["TimesPlus"][realmName]= {};
	end					

        local factionGroup = UnitFactionGroup("player");
	if( CensusPlus_Database["TimesPlus"][realmName][factionGroup] == nil ) then
            CensusPlus_Database["TimesPlus"][realmName][factionGroup] = {};
	end

        local realmDB = CensusPlus_Database["Servers"][realmName];
        if (realmDB == nil) then
            CensusKr_Print("DB Error");
            return;
        end

        local factionDB = realmDB[factionGroup];
        if (factionDB == nil) then
            CensusKr_Print("db error");
            return;
        end

        local newDB = factionDB[g_TimeStamp];
        if (newDB == nil) then
            CensusKr_Print("db error3");
            return;
        end

        for raceName, raceDB in pairs(newDB) do
            for className, classDB in pairs(raceDB) do 
                for guildName, guildDB in pairs(classDB) do
                    for levelName, levelDB in pairs(guildDB) do
                        for zoneName, census in pairs(levelDB) do
                            g_TimeDatabase[raceName] = g_TimeDatabase[raceName] + census;
                            g_TimeDatabase[className] = g_TimeDatabase[className] + census;
                        end
                    end
                end
            end
        end
	
	local numTimes = table.getn( CensusPlus_Database["TimesPlus"][realmName][UnitFactionGroup("player")] );
	if (numTimes > CP_MAX_TIMES-1 ) then
		--
		-- Remove the top time from the queue 
		--
		table.remove( CensusPlus_Database["TimesPlus"][realmName][UnitFactionGroup("player")] );
	end

	local hour, minute = GetGameTime();
--					CensusPlus_Database["TimesPlus"][realmName][UnitFactionGroup("player")]["" .. hour .. ""] = g_TimeDatabase;
	--CensusPlus_Database["TimesPlus"][realmName][UnitFactionGroup("player")][CensusPlus_DetermineServerDate() .. "&" .. hour .. ":" .. minute .. ":00"] = 
        CensusPlus_Database["TimesPlus"][realmName][factionGroup][g_TimeStamp] = 
		g_TimeDatabase[CENSUSPlus_DRUID] .. "&" ..
		g_TimeDatabase[CENSUSPlus_HUNTER] .. "&" ..
		g_TimeDatabase[CENSUSPlus_MAGE] .. "&" ..
		g_TimeDatabase[CENSUSPlus_PRIEST] .. "&" ..
		g_TimeDatabase[CENSUSPlus_ROGUE] .. "&" ..
		g_TimeDatabase[CENSUSPlus_WARLOCK] .. "&" ..
		g_TimeDatabase[CENSUSPlus_WARRIOR] .. "&" ..
		g_TimeDatabase[CENSUSPlus_SHAMAN] .. "&" ..
		g_TimeDatabase[CENSUSPlus_PALADIN] .. "&"..
                0 .. "&" ..
                0 .. "&" ..
                0 .. "&" ..
                g_TimeDatabase[CENSUSPlus_DWARF] .. "&" ..
                g_TimeDatabase[CENSUSPlus_GNOME] .. "&" ..
                g_TimeDatabase[CENSUSPlus_HUMAN] .. "&" ..
                g_TimeDatabase[CENSUSPlus_NIGHTELF] .. "&" ..
                g_TimeDatabase[CENSUSPlus_ORC] .. "&" ..
                g_TimeDatabase[CENSUSPlus_TAUREN] .. "&" ..
                g_TimeDatabase[CENSUSPlus_TROLL] .. "&" ..
                g_TimeDatabase[CENSUSPlus_UNDEAD];                
end

-----------------------------------------------------------------------------------
--
-- Add the contents of the guild results to the database
--
-----------------------------------------------------------------------------------
function CensusPlus_ProcessGuildResults()

    if( g_VariablesLoaded == false ) then
        return;
    end
    
    if( CensusPlus_Database["Info"]["Locale"] == nil ) then
		return;
	end
   
    --
    --  Grab temp var
    --
	local showOfflineTemp = GetGuildRosterShowOffline();
	SetGuildRosterShowOffline(1);
    
    
	--
	-- Walk through the guild info
	--
    local numGuildMembers = GetNumGuildMembers();
--	CensusPlus_Msg("Processing "..numGuildMembers.." guild members.");

    local realmName = g_CensusPlusLocale .. GetCVar("realmName");
    CensusPlus_Database["Guilds"] = nil;
    if( CensusPlus_Database["Guilds"] == nil ) then
		CensusPlus_Database["Guilds"] = {};
    end
    
	if (CensusPlus_Database["Guilds"][realmName] == nil) then
		CensusPlus_Database["Guilds"][realmName] = {};
	end
    
	local guildRealmDatabase = CensusPlus_Database["Guilds"][realmName];
	if (guildRealmDatabase == nil) then
		CensusPlus_Database["Guilds"][realmName] = {};
		guildRealmDatabase = CensusPlus_Database["Guilds"][realmName];
	end

	local factionGroup = UnitFactionGroup("player");
	if( factionGroup == nil ) then
	    CensusPlus_Database["Guilds"] = nil;
	    return;
	end

	local factionDatabase = guildRealmDatabase[factionGroup];
	if (factionDatabase == nil) then
		guildRealmDatabase[factionGroup] = {};
		factionDatabase = guildRealmDatabase[factionGroup];
	end
	
	CensusPlus_Database["Guilds"][realmName][factionGroup] = nil;
	CensusPlus_Database["Guilds"][realmName][factionGroup] = {};
	
	factionDatabase = CensusPlus_Database["Guilds"][realmName][factionGroup];	
	
    local Ginfo = GetGuildInfo("player");
	if( Ginfo == nil ) then
	    CensusPlus_Database["Guilds"] = nil;
	    return;
	end
	local guildDatabase = factionDatabase[Ginfo];
	if (guildDatabase == nil) then
		factionDatabase[Ginfo] = {};
		guildDatabase = factionDatabase[Ginfo];
	end

	local info = guildDatabase["GuildInfo"];
	if (info == nil) then
		guildDatabase["GuildInfo"] = {};
		info = guildDatabase["GuildInfo"];
	end
	
	info["Update"] = date( "%m-%d-%Y", time()) .. "";
	info["ShowOnline"] = 1;  --  Variable comes from FriendsFrame
	
	guildDatabase["Members"] = nil;
	guildDatabase["Members"] = {};
	
	local members = guildDatabase["Members"];

    for index = 1, numGuildMembers, 1 do 
        local name, rank, rankIndex, level, class, zone, group, note, officernote, online = GetGuildRosterInfo(index); 
        
        if( members[name] == nil ) then
            members[name] = {};
        end
        
--        CensusPlus_Msg( "Name =>" .. name );
--        CensusPlus_Msg( "rank =>" .. rank );
--        CensusPlus_Msg( "rankIndex =>" .. rankIndex );
--        CensusPlus_Msg( "level =>" .. level );
--        CensusPlus_Msg( "class =>" .. class );
        members[name]["Rank"] = rank; 
        members[name]["RankIndex"] = rankIndex;
        members[name]["Level"]= level; 
        members[name]["Class"]= class;
    end
    
	SetGuildRosterShowOffline(showOfflineTemp);
end

function CensusKr_setTimeStamp()
    local hour, minute = GetGameTime();
    local ts = CensusPlus_DetermineServerDate() .. "&" .. hour .. ":" .. minute .. ":00";
    g_TimeStamp = ts;
    --return ts;
end

function CensusKr_getLastTimeStamp(realm, faction)
    local CensusDB = CensusPlus_Database["Servers"];
    if (CensusDB[realm] == nil) then
        return;
    end
    if (CensusDB[realm][faction] == nil) then
        return;
    end
    local last = "";
    for ts, tsdb in pairs(CensusDB[realm][faction]) do
        if (ts > last) then 
            last = ts;
        end
    end
    return last;
end
-----------------------------------------------------------------------------------
--
-- Add the contents of the who results to the database
--
-----------------------------------------------------------------------------------
function CensusPlus_ProcessWhoResults()

	--
	--  If we are in a BG then stop a census
	--
    if( g_CurrentlyInBG and g_IsCensusPlusInProgress ) then
		g_LastCensusRun = time()-600;    
		CensusPlus_Msg(CENSUSPlus_ISINBG);
		CensusPlus_StopCensus( );
	end


	--
	-- Get the portion of the database for this server
	--
	local realmName = g_CensusPlusLocale .. GetCVar("realmName");
	local realmDatabase = CensusPlus_Database["Servers"][realmName];
	if (realmDatabase == nil) then
		CensusPlus_Database["Servers"][realmName] = {};
		realmDatabase = CensusPlus_Database["Servers"][realmName];
	end

	--
	-- Get the portion of the database for this faction
	--
	local factionGroup = UnitFactionGroup("player");
	local factionDatabase = realmDatabase[factionGroup];
	if (factionDatabase == nil) then
		realmDatabase[factionGroup] = {};
		factionDatabase = realmDatabase[factionGroup];
	end

        if (g_TimeStamp == nil) then
            CensusKr_setTimeStamp();
        end
        local timeStamp = g_TimeStamp;
        local timeDatabase = factionDatabase[timeStamp];
        if (timeDatabase == nil) then 
            factionDatabase[timeStamp] = {};
            timeDatabase = factionDatabase[timeStamp];
        end
	
	--
	-- Walk through all the who results
	--
	local numWhoResults = GetNumWhoResults();
	--CensusKr_Print("debug: WhoResults - " ..numWhoResults);
	if( CensusPlus_Database["Info"]["Verbose"] == true ) then
	    CensusPlus_Msg(format(CENSUSPlus_PROCESSING, numWhoResults));
	end
	for i = 1, numWhoResults, 1 do
		--
		-- Get who result entry
		--
		local name, guild, level, race, class, zone, group = GetWhoInfo(i);
		
		--
		--  Test the name for possible color coding
		--
		--  for example |cffff0000Rollie|r
        local karma_check = string.find( name, "|cff" );
        if( karma_check ~= nil ) then
			name = string.sub( name, 11, -3 );
        end	
		
		--
		-- Get racial database
		--
		local raceDatabase = timeDatabase[race];
		if (raceDatabase == nil) then
			timeDatabase[race] = {};
			raceDatabase = timeDatabase[race];
		end

		--
		-- Get class database
		--
		local classDatabase = raceDatabase[class];
		if (classDatabase == nil) then
			raceDatabase[class] = {};
			classDatabase = raceDatabase[class];
		end

                -- 길드
                local guildDatabase = classDatabase[guild];
                if (guildDatabase == nil) then
                    classDatabase[guild] = {};
                    guildDatabase = classDatabase[guild];
                end

                -- 레벨
                local levelDatabase = guildDatabase[level];
                if (levelDatabase == nil) then
                    guildDatabase[level] = {};
                    levelDatabase = guildDatabase[level];
                end

                -- 지역
                --local zoneDatabase = levelDatabase[zone];
                if (levelDatabase[zone] == nil) then
                    levelDatabase[zone] = 0;
                end

		--
		-- 지역정보, 인구수, 중복 캐릭터 정보는 제외함. 
		--
                if (g_TempCount[name] == nil) then
                    levelDatabase[zone] = levelDatabase[zone] + 1;
                    g_NumUpdatedCharacters = g_NumUpdatedCharacters + 1;
        		g_TempCount[name] = class;            
                end
	end
--	CensusPlus_UpdateView();
end

----------------------------------------------------------------------------------
--
-- Find a guild in the g_Guilds array by name
-- 
---------------------------------------------------------------------------------
local function FindGuildByName(name)
	local i;
	local size = table.getn(g_Guilds);
	for i = 1, size, 1 do
		local entry = g_Guilds[i];
		if (entry.m_Name == name) then
			return i;
		end
	end
	return nil;
end

----------------------------------------------------------------------------------
--
-- Add up the total character XP and count
-- 
---------------------------------------------------------------------------------
local g_AccumulateGuildTotals = true;
local function TotalsAccumulator(guild, census)
	g_TotalCount = g_TotalCount + census;
	if (g_AccumulateGuildTotals and (guild ~= nil)) then
		local index = FindGuildByName(guild);
		if (index == nil) then
			local size = table.getn(g_Guilds);
			index = size + 1;
			g_Guilds[index] = {m_Name = guild, m_TotalCharacterXP = 0, m_Count = 0};
		end
		local entry = g_Guilds[index];
		entry.m_Count = entry.m_Count + census;
	end
end

----------------------------------------------------------------------------------
--
-- Predicate function which can be used to compare two guilds for sorting
-- 
---------------------------------------------------------------------------------
local function GuildPredicate(lhs, rhs)
	--
	-- nil references are always less than
	--
	if (lhs == nil) then
		if (rhs == nil) then
			return false;
		else
			return true;
		end
	elseif (rhs == nil) then
		return false;	
	end
	--
	-- Sort by total XP first
        -- 경험치 대신 인원 수로 정렬
	--
	if (rhs.m_Count < lhs.m_Count) then
		return true;
	elseif (lhs.m_Count < rhs.m_Count) then
		return false;
	end
	--
	-- Sort by name
	--
	if (lhs.m_Name < rhs.m_Name) then
		return true;
	elseif (rhs.m_Name < lhs.m_Name) then
		return false;
	end
	
	--
	-- identical
	--
	return false;	
end


----------------------------------------------------------------------------------
--
-- Another accumulator for adding up XP and counts
-- name: 길드 정보처리 위해 보내는 것일 뿐, 전체 데이터 처리에선 상관없음. 
---------------------------------------------------------------------------------
local g_AccumulatorCount = 0;
local g_AccumulatorXPTotal = 0;
local function CensusPlus_Accumulator(name, census)
	g_AccumulatorCount = g_AccumulatorCount + census;
end

----------------------------------------------------------------------------------
--
-- Reset the above accumulator
-- 
---------------------------------------------------------------------------------
function CensusPlus_ResetAccumulator()
	g_AccumulatorCount = 0;
	g_AccumulatorXPTotal = 0;
end


----------------------------------------------------------------------------------
--
-- Search the character database using the search criteria and update display
-- 
---------------------------------------------------------------------------------
function CensusPlus_UpdateView()
	--
	-- Get realm and faction
	--
	local realmName = g_CensusPlusLocale .. GetCVar("realmName");
	if( realmName == nil ) then
		return;
	end
	CensusPlusRealmName:SetText(format(CENSUSPlus_REALMNAME, realmName));
	
	local factionGroup = UnitFactionGroup("player");
	if( factionGroup == nil ) then
		return;
	end

	CensusPlusFactionName:SetText(format(CENSUSPlus_FACTION, factionGroup));

	if( CensusPlus_Database["Info"]["Locale"] ~= nil ) then
		CensusPlusLocaleName:SetText(format(CENSUSPlus_LOCALE, CensusPlus_Database["Info"]["Locale"]));
	end

        local timeKey = g_TimeStamp;
        if (timeKey == nil) then 
            timeKey = CensusKr_getLastTimeStamp(realmName, factionGroup);
        end
	

	local guildKey = nil;
	local raceKey = nil;
	local classKey = nil;
	local levelKey = nil;
	g_TotalCharacterXP = 0;
	g_TotalCount = 0;
	
	--
	-- Has the user selected a guild?
	--
	if (g_GuildSelected > 0) then
		guildKey = g_Guilds[g_GuildSelected].m_Name;
	end
	if (g_RaceSelected > 0) then
		local thisFactionRaces = GetFactionRaces(factionGroup);
		raceKey = thisFactionRaces[g_RaceSelected];
	end
	if (g_ClassSelected > 0) then
		local thisFactionClasses = GetFactionClasses(factionGroup);
		classKey = thisFactionClasses[g_ClassSelected];
	end
	if (g_LevelSelected > 0 or g_LevelSelected < 0) then
		levelKey = g_LevelSelected;
	end
	
	--
	-- Has the user added any search criteria?
	--
	if ((guildKey ~= nil) or (raceKey ~= nil) or (classKey ~= nil) or (levelKey ~= nil)) then
		--
		-- Get totals for this criteria
		--
		g_AccumulateGuildTotals = false;
		CensusPlus_ForAllCharacters(realmName, factionGroup, timeKey, raceKey, classKey, guildKey, levelKey, TotalsAccumulator);
	else
		--
		-- Get the overall totals and find guild information
		--
		g_Guilds = {};
		g_AccumulateGuildTotals = true;
		CensusPlus_ForAllCharacters(realmName, factionGroup, timeKey, nil, nil, nil, nil, TotalsAccumulator);
		local size = table.getn(g_Guilds);
		if (size) then
			table.sort(g_Guilds, GuildPredicate);
		end
	end
	local levelSearch = nil;
	if (levelKey ~= nil) then
		levelSearch = "  ("..CENSUSPlus_LEVEL..": ";
		local level = levelKey;
		if (levelKey < 0) then
			levelSearch = levelSearch.."!";
			level = 0 - levelKey;
		end
		levelSearch = levelSearch..level..")";
	end

	local totalCharactersText = nil;
	if (levelSearch ~= nil) then
		totalCharactersText = format(CENSUSPlus_TOTALCHAR, g_TotalCount)..levelSearch;
	else
		totalCharactersText = format(CENSUSPlus_TOTALCHAR, g_TotalCount);
	end
	CensusPlusTotalCharacters:SetText(totalCharactersText);
	CensusPlusTotalCharacterXP:SetText(format(CENSUSPlus_TOTALCHARXP, g_TotalCharacterXP));	
	CensusPlus_UpdateGuildButtons();
	
	--
	-- Accumulate totals for each race
	--
	local maxCount = 0;
	local thisFactionRaces = GetFactionRaces(factionGroup);
	local numRaces = table.getn(thisFactionRaces);
	for i = 1, numRaces, 1 do
		local race = thisFactionRaces[i];
		CensusPlus_ResetAccumulator();
		if ((raceKey == nil) or (raceKey == race)) then
			CensusPlus_ForAllCharacters(realmName, factionGroup, timeKey, race, classKey, guildKey, levelKey, CensusPlus_Accumulator);
		end
		if (g_AccumulatorCount > maxCount) then
			maxCount = g_AccumulatorCount;
		end
		g_RaceCount[i] = g_AccumulatorCount;
	end

	--
	-- Update race bars
	--
	for i = 1, numRaces, 1 do
		local race = thisFactionRaces[i];
		local buttonName = "CensusPlusRaceBar"..i;
		local button = getglobal(buttonName);
		local thisCount = g_RaceCount[i];
		if ((thisCount ~= nil) and (thisCount > 0) and (maxCount > 0)) then
			local height = floor((thisCount / maxCount) * CensusPlus_MAXBARHEIGHT);
			if (height < 1 or height == nil ) then height = 1; end
			button:SetHeight(height);
			button:Show();
		else
			button:Hide();
		end
		local normalTextureName="Interface\\AddOns\\CensusPlus\\Skin\\CensusPlus_"..g_RaceClassList[race];
		local legendName = "CensusPlusRaceLegend"..i;
		local legend = getglobal(legendName);
		legend:SetNormalTexture(normalTextureName);
		if (g_RaceSelected == i) then
			legend:LockHighlight();
		else
			legend:UnlockHighlight();
		end
	end	

	--
	-- Accumulate totals for each class
	--
	local maxCount = 0;
	local thisFactionClasss = GetFactionClasses(factionGroup);
	local numClasses = table.getn(thisFactionClasss);
	for i = 1, numClasses, 1 do
		local class = thisFactionClasss[i];
		CensusPlus_ResetAccumulator();
		if ((classKey == nil) or (classKey == class)) then
			CensusPlus_ForAllCharacters(realmName, factionGroup, timeKey, raceKey, class, guildKey, levelKey, CensusPlus_Accumulator);
		end
		if (g_AccumulatorCount > maxCount) then
			maxCount = g_AccumulatorCount;
		end
		g_ClassCount[i] = g_AccumulatorCount;
	end

	--
	-- Update class bars
	--
	for i = 1, numClasses, 1 do
		local class = thisFactionClasss[i];
			
		local buttonName = "CensusPlusClassBar"..i;
		local button = getglobal(buttonName);
		local thisCount = g_ClassCount[i];
		if ((thisCount ~= nil) and (thisCount > 0) and (maxCount > 0)) then
			local height = floor((thisCount / maxCount) * CensusPlus_MAXBARHEIGHT);
			if (height < 1 or height == nil ) then height = 1; end
			button:SetHeight(height);
			button:Show();
		else
			button:Hide();
		end
		
		local normalTextureName="Interface\\AddOns\\CensusPlus\\Skin\\CensusPlus_"..g_RaceClassList[class];
		local legendName = "CensusPlusClassLegend"..i;
		local legend = getglobal(legendName);
		legend:SetNormalTexture(normalTextureName);
		if (g_ClassSelected == i) then
			legend:LockHighlight();
		else
			legend:UnlockHighlight();
		end
	end	
	
	--
	-- Accumulate totals for each level
	--
	local maxCount = 0;
	for i = 1, MAX_CHARACTER_LEVEL, 1 do
	    if ((levelKey == nil) or (levelKey == i) or (levelKey < 0 and levelKey + i ~= 0)) then
			CensusPlus_ResetAccumulator();
			CensusPlus_ForAllCharacters(realmName, factionGroup, timeKey, raceKey, classKey, guildKey, i, CensusPlus_Accumulator);
			if (g_AccumulatorCount > maxCount) then
				maxCount = g_AccumulatorCount;
			end
			g_LevelCount[i] = g_AccumulatorCount;
		else
			g_LevelCount[i] = 0;
		end
	end

	--
	-- Update level bars
	--
	for i = 1, MAX_CHARACTER_LEVEL, 1 do
		local buttonName = "CensusPlusLevelBar"..i;
		local buttonEmptyName = "CensusPlusLevelBarEmpty"..i;
		local button = getglobal(buttonName);
		local emptyButton = getglobal(buttonEmptyName);
		local thisCount = g_LevelCount[i];
		if ((thisCount ~= nil) and (thisCount > 0) and (maxCount > 0)) then
			local height = floor((thisCount / maxCount) * CensusPlus_MAXBARHEIGHT);
			if (height < 1 or height == nil ) then height = 1; end
			button:SetHeight(height);
			button:Show();
			if (emptyButton ~= nil) then
				emptyButton:Hide();
			end
		else
			button:Hide();
			if (emptyButton ~= nil) then
				emptyButton:SetHeight(CensusPlus_MAXBARHEIGHT);
				emptyButton:Show();
			end
		end
	end	
end

----------------------------------------------------------------------------------
--
-- Walk the character database and call the callback function for every entry that matches the search criteria
-- 
---------------------------------------------------------------------------------
function CensusPlus_ForAllCharacters(realmKey, factionKey, timeKey, raceKey, classKey, guildKey, levelKey, callback)
    if (timeKey == nil) then 
        return;
    end
    for realmName, realmDatabase in pairs(CensusPlus_Database["Servers"]) do
        if ((realmKey == nil) or (realmKey == realmName)) then
            for factionName, factionDatabase in pairs(realmDatabase) do
                if ((factionKey == nil) or (factionKey == factionName)) then
                    for timeStamp, timeDatabase in pairs(factionDatabase) do
                        if (timeStamp == timeKey) then
                            for raceName, raceDatabase in pairs(timeDatabase) do
                                if ((raceKey == nil) or (raceKey == raceName)) then
                                    for className, classDatabase in pairs(raceDatabase) do
                                        if ((classKey == nil) or (classKey == className)) then
                                            for guildName, guildDatabase in pairs(classDatabase) do
                                                if ((guildKey == nil) or (guildKey == guildName)) then
                                                    for levelName, levelDatabase in pairs(guildDatabase) do
                                                        --if ((leveName == nil) or (levelKey == levelName)) then
                                                        if ((levelKey == nil) 
                                                        or (levelKey == levelName) 
                                                        or (levelKey < 0 and levelKey + levelName ~= 0)) then
                                                            for zoneName, census in (levelDatabase) do
                                                                callback(guildName, census);
                                                            end -- for zone
                                                        end -- if level
                                                    end -- for level
                                                end -- if guild
                                            end -- for guild
                                        end -- if class
                                    end -- for class
                                end -- if race
                            end -- for race
                        end -- if times
                    end -- for times
                end -- if faction
            end -- for faction
        end -- if realm 
    end -- for realm
end -- end func

----------------------------------------------------------------------------------
--
-- Race legend clicked
-- 
---------------------------------------------------------------------------------
function CensusPlus_OnClickRace()
	local id = this:GetID();
	if (id == g_RaceSelected) then
		g_RaceSelected = 0;
	else
		g_RaceSelected = id;
	end
	CensusPlus_UpdateView();
end

----------------------------------------------------------------------------------
--
-- Class legend clicked
-- 
---------------------------------------------------------------------------------
function CensusPlus_OnClickClass()
	local id = this:GetID();
	if (id == g_ClassSelected) then
		g_ClassSelected = 0;
	else
		g_ClassSelected = id;
	end
	CensusPlus_UpdateView();
end


----------------------------------------------------------------------------------
--
-- Level bar loaded
-- 
---------------------------------------------------------------------------------
function CensusPlus_OnLoadLevel()
	this:RegisterForClicks("LeftButtonUp","RightButtonUp");
end

----------------------------------------------------------------------------------
--
-- Level bar clicked
-- 
---------------------------------------------------------------------------------
function CensusPlus_OnClickLevel(button)
	local id = this:GetID();
	if (((button == "LeftButton") and (id == g_LevelSelected)) or ((button == "RightButton") and (id + g_LevelSelected == 0))) then
		g_LevelSelected = 0;
	elseif (button == "RightButton") then
		g_LevelSelected = 0 - id;
	else
		g_LevelSelected = id;
	end
	CensusPlus_UpdateView();
end

----------------------------------------------------------------------------------
--
-- Race tooltip
-- 
---------------------------------------------------------------------------------
function CensusPlus_OnEnterRace()
	local factionGroup = UnitFactionGroup("player");
	local thisFactionRaces = GetFactionRaces(factionGroup);
	local id = this:GetID();
	local raceName = thisFactionRaces[id];
	local count = g_RaceCount[id];
	if (count ~= nil) then
	    local percent = floor((count / g_TotalCount) * 100);
	    GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
	    GameTooltip:SetText(raceName.."\n"..count.."\n"..percent.."%", 1.0, 1.0, 1.0);
	    GameTooltip:Show();
	end
end

----------------------------------------------------------------------------------
--
-- Class tooltip
-- 
---------------------------------------------------------------------------------
function CensusPlus_OnEnterClass()
	local factionGroup = UnitFactionGroup("player");
	local thisFactionClasses = GetFactionClasses(factionGroup);
	local id = this:GetID();
	local className = thisFactionClasses[id];
	local count = g_ClassCount[id];
	if (count ~= nil) then
	    local percent = floor((count / g_TotalCount) * 100);
	    GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
	    GameTooltip:SetText(className.."\n"..count.."\n"..percent.."%", 1.0, 1.0, 1.0);
	    GameTooltip:Show();
	end
end

----------------------------------------------------------------------------------
--
-- Level tooltip
-- 
---------------------------------------------------------------------------------
function CensusPlus_OnEnterLevel()
	local id = this:GetID();
	local count = g_LevelCount[id];
	if (count ~= nil) then
		local percent = floor((count / g_TotalCount) * 100);
		GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
		GameTooltip:SetText("Level "..id.."\n"..count.."\n"..percent.."%", 1.0, 1.0, 1.0);
		GameTooltip:Show();
	end
end

----------------------------------------------------------------------------------
--
-- Clicked a guild button
-- 
---------------------------------------------------------------------------------
function CensusPlus_GuildButton_OnClick()
	local id = this:GetID();
	local offset = FauxScrollFrame_GetOffset(CensusPlusGuildScrollFrame);
	local newSelection = id + offset;
	if (g_GuildSelected ~= newSelection) then
		g_GuildSelected = newSelection;
	else
		g_GuildSelected = 0;
	end
	CensusPlus_UpdateView();
end

----------------------------------------------------------------------------------
--
-- Update the guild button contents
-- 
---------------------------------------------------------------------------------
function CensusPlus_UpdateGuildButtons()
	--
	-- Determine where the scroll bar is
	--
	local offset = FauxScrollFrame_GetOffset(CensusPlusGuildScrollFrame);
	--
	-- Walk through all the rows in the frame
	--
	local size = table.getn(g_Guilds);
	local i = 1;
	while (i <= CensusPlus_NUMGUILDBUTTONS) do
		--
		-- Get the index to the ad displayed in this row
		--
		local iGuild = i + offset;
		--
		-- Get the button on this row
		--
		local button = getglobal("CensusPlusGuildButton"..i);
		--
		-- Is there a valid guild on this row?
		--
		if (iGuild <= size) then
			local guild = g_Guilds[iGuild];
			--
			-- Update the button text
			--
			button:Show();
			local textField = "CensusPlusGuildButton"..i.."Text";
			if (guild.m_Name == "") then
				getglobal(textField):SetText(CENSUSPlus_UNGUILDED);
			else
				getglobal(textField):SetText(guild.m_Name);
			end
			--
			-- If this is the guild, highlight it
			--
			if (g_GuildSelected == iGuild) then
				button:LockHighlight();
			else
				button:UnlockHighlight();
			end
		else
			--
			-- Hide the button
			--
			button:Hide();
		end
		--
		-- Next row
		--
		i = i + 1;
	end
	--
	-- Update the scroll bar
	--
	FauxScrollFrame_Update(CensusPlusGuildScrollFrame, size, CensusPlus_NUMGUILDBUTTONS, CensusPlus_GUILDBUTTONSIZEY);
end



----------------------------------------------------------------------------------
--
-- CensusPlus_SelectLocale - Set the locale (US or EU)
-- 
---------------------------------------------------------------------------------
function CensusPlus_SelectLocale( locale, auto )

	if( not auto ) then
		CensusPlus_Msg( "You have set your locale to " .. locale .. " from " .. g_CensusPlusLocale );
	end

	g_CensusPlusLocale = locale;
    if( g_CensusPlusLocale == "EU" ) then
		g_CensusPlusLocale = g_CensusPlusLocale .. "-";
	else
		g_CensusPlusLocale = "";
    end
    -- 한국서버에 한정해서 사용되어야 하므로, 다른 지역은 배제함.
    g_CensusPlusLocale = "";

	
	if( CensusPlus_Database["Info"]["Locale"] ~= locale ) then
		if( not ( CensusPlus_Database["Info"]["Locale"] == nil and locale == "US" ) ) then
			CensusPlus_Msg( "Locale differs from previous setting, purging database." );
			CensusPlus_Purge();
			CensusPlus_Database["Info"]["Locale"] = locale;
		end
	end
	CensusPlus_Database["Info"]["Locale"] = locale;
	
	textLine = getglobal("CensusPlusText");
	textLine:SetText("Census+ v"..CensusPlus_VERSION .. " " .. g_CensusPlusLocale );
	
    if(( CENSUSPlus_DWARF == "Nain" or CENSUSPlus_DWARF == "Zwerg" ) and GetLocale() == "usEN") then
		CensusPlus_Msg( "You appear to have a US Census version, yet your localization is set to French or German." );
		CensusPlus_Msg( "Please do not upload stats to WarcraftRealms until this has been resolved." );
		CensusPlus_Msg( "If this is incorrect, please let Rollie know at www.WarcraftRealms.com about your situation so he can make corrections." );
    end
    
	CP_EU_US_Version:Hide();    
    
end

function CensusPlus_CheckForBattleground()
	local battlefieldTime = GetBattlefieldInstanceRunTime();
	if( battlefieldTime > 0 ) then
		--
		--  We are in a battleground so cancel the current take
		--
		g_CurrentlyInBG = true;
	else
		if( GetBattlefieldStatInfo(1) ~= nil ) then
			g_CurrentlyInBG = true;
		else
			g_CurrentlyInBG = false;
		end			
	end

end

function CensusPlus_IsInspectLoaded()
	if (IsAddOnLoaded("Blizzard_InspectUI")) then
		--ChatFrame1:AddMessage("Inspect Loaded");
		return true;
	end
	
	if (CensusPlus_Database["Info"]["LoadInspect"] ~= nil and CensusPlus_Database["Info"]["LoadInspect"] == true) then
		--ChatFrame1:AddMessage("Loading Inspect Frame");
		LoadAddOn("Blizzard_InspectUI");
	end

	--ChatFrame1:AddMessage("Inspect Not Loaded");
	return false;
end

function CensusPlus_IsTalentLoaded()
	if (IsAddOnLoaded("Blizzard_TalentUI")) then
		--ChatFrame1:AddMessage("Talent Loaded");
		return true;
	end
	
	if (CensusPlus_Database["Info"]["LoadTalent"] ~= nil and CensusPlus_Database["Info"]["LoadTalent"] == true) then
		--ChatFrame1:AddMessage("Loading Talent Frame");
		LoadAddOn("Blizzard_TalentUI");
	end

	--ChatFrame1:AddMessage("Talent Not Loaded");
	return false;
end



function showAllUnitBuffs(sUnitname) 
  local iIterator = 1
  DEFAULT_CHAT_FRAME:AddMessage(format("[%s] Buffs", sUnitname))
  while (UnitBuff(sUnitname, iIterator)) do
    DEFAULT_CHAT_FRAME:AddMessage(UnitBuff(sUnitname, iIterator), 1, 1, 0)    
    iIterator = iIterator + 1
  end
  DEFAULT_CHAT_FRAME:AddMessage("---", 1, 1, 0)    
end

function CensusPlus_GetUTCDateTimeStr()
	return date( "!%Y-%m-%d %H:%M", time() );
end

-----------------------------------------------------------------------------------
--
-- CensusPlus_DetermineServerDate 
--
-----------------------------------------------------------------------------------
function CensusPlus_DetermineServerDate()

	CensusPlus_CheckTZ();
	
	local strDate;
	local TZOffset = g_CensusPlusTZOffset;
	
	--
	--  Timezone offsets should fall into distinct numbers for now
	--		And now that we know if they are playing on US or EU servers
	--		we can be even better estimates
	--
	
	--[[
		For US servers, the offset should be either -9 to -5 or 16/19 depending on DST for NA times
			and for oceana it is +11/-13
		EU servers are either +1 or 0 or -23 depending on DST
	]]--
	
	if( CensusPlus_Database["Info"]["Locale"] == "US" ) then
		if( TZOffset > 12 ) then
			-- NA server times but wrong day
			TZOffset = TZOffset - 24;
		elseif( TZOffset < -11 ) then
			--  Oceana times but wrong day
			TZOffset = 24 - TZOffset;
		end
	else
		if( TZOffset == -23 ) then
			TZOffset = 1;
		end
	end
	
	--  Now, take the TZOffset and modify our time to give us server date
	strDate = date( "!%Y-%m-%d", time() + (TZOffset * 3600 ) );

--	local strDate2 = date( "%Y-%m-%d : %H:%M", time() );
--	CensusPlus_Msg("Server date = " .. strDate .. " for TZOffset : " .. TZOffset .. " curr local: " .. strDate2 );

	return strDate;
end

-----------------------------------------------------------------------------------
--
-- Check time zone
--
-----------------------------------------------------------------------------------
function CensusPlus_CheckTZ()

    local UTCTimeHour = date( "!%H", time() );
    local LocTimeHour = date( "%H", time() );
	local hour, minute = GetGameTime();
	
	local locDiff  = LocTimeHour - UTCTimeHour;
	local servDiff = hour - UTCTimeHour;
--[[	
	if(( servDiff == 1) or ( servDiff ==-23) or ( servDiff ==0) or ( servDiff ==-24)) then
--		CensusPlus_Msg("Guessing European (EU) Servers");
		if( g_CensusPlusLocale ~= "EU" and g_TZWarningSent == false) then
			CensusPlus_Msg("I guessed you are playing on EURO servers but your locale is not set to EU, are you sure this is correct?  Type /census locale to change locale setting.");
			g_TZWarningSent = true;
		end
	elseif(( servDiff >= 16 and servDiff <= 20 ) or ( servDiff >= -8 and servDiff <= -4 )) then
--		CensusPlus_Msg("Guessing North American (US) Servers" );
		if( g_CensusPlusLocale == "EU" and g_TZWarningSent == false ) then
			CensusPlus_Msg("I guessed you are playing on US servers but your locale not set to US, are you sure this is correct?  Type /census locale to change locale setting.");
			g_TZWarningSent = true;
		end
	elseif( g_TZWarningSent == false ) then
		CensusPlus_Msg("Unable to determine locale");
		g_TZWarningSent = true;
	end
]]--	
	g_CensusPlusTZOffset = servDiff;
end

