/*  -
   Copyright (C) 2006 Weongyo Jeong (weongyo@gmail.com)

This file is part of ROVM.

ROVM is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

ROVM is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.  */

/*
  Ʒ ڵ Apache MPM  Worker MPM    ڵԴϴ.  Ʒ
  ڵ  Apache ̼ ϴ.
 */

/*
  Ʒ ڵ Apache MPM  Worker MPM   ڵν, Apache ڵ ,
  ROVM κ Ʈũ  ״ ̱⿡ ڵ尡 ʹ , ټ 
   , Ʒ  ּ ȭϿ Ͽ.

   ִ Apache Worker MPM ü   Ͽ.
 */
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>

#include "types.h"
#include "mpool.h"
#include "tktree.h"
#include "sha1.h"
#include "listen.h"
#include "rovm.h"

#include "thread.h"
#include "thread_mutex.h"
#include "thread_cond.h"

#include "mpm_worker_fdqueue.h"
#include "mpm_worker_pod.h"
#include "mpm_worker.h"

#include "connection.h"
#include "request.h"

#include "log.h"
#include "utils.h"

/* The LISTENER_SIGNAL signal will be sent from the main thread to the
 * listener thread to wake it up for graceful termination (what a child
 * process from an old generation does when the admin does "apachectl
 * graceful").  This signal will be blocked in all threads of a child
 * process except for the listener thread.
 */
#define LISTENER_SIGNAL     SIGHUP

static void
sig_term (int sig)
{
  rovm_log (NULL, ROVMLOG_INFO, ROVMLOG_MARK,
            "sig_term: sig = %d", sig);
}

static void
restart (int sig)
{
  rovm_log (NULL, ROVMLOG_INFO, ROVMLOG_MARK,
            "restart: sig = %d", sig);
}

static void
set_signals (struct rovm *r)
{
#ifndef NO_USE_SIGACTION
    struct sigaction sa;
#endif

    if (!CONF_ONE_PROCESS (ROVM_CONF (r))) 
      {
        //ap_fatal_signal_setup(ap_server_conf, pconf);
      }
    
#ifndef NO_USE_SIGACTION
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = 0;
    
    sa.sa_handler = sig_term;
    if (sigaction(SIGTERM, &sa, NULL) < 0)
      rovm_log (r, ROVMLOG_WARNING, ROVMLOG_MARK,
                "sigaction(SIGTERM)");
#ifdef SIGINT
    if (sigaction(SIGINT, &sa, NULL) < 0)
      rovm_log (r, ROVMLOG_WARNING, ROVMLOG_MARK,
                "sigaction(SIGINT)");
#endif
#ifdef SIGXCPU
    sa.sa_handler = SIG_DFL;
    if (sigaction(SIGXCPU, &sa, NULL) < 0)
      rovm_log (r, ROVMLOG_WARNING, ROVMLOG_MARK,
                "sigaction(SIGXCPU)");
#endif
#ifdef SIGXFSZ
    sa.sa_handler = SIG_DFL;
    if (sigaction(SIGXFSZ, &sa, NULL) < 0)
      rovm_log (r, ROVMLOG_WARNING, ROVMLOG_MARK,
                "sigaction(SIGXFSZ)");
#endif
#ifdef SIGPIPE
    sa.sa_handler = SIG_IGN;
    if (sigaction(SIGPIPE, &sa, NULL) < 0)
      rovm_log (r, ROVMLOG_WARNING, ROVMLOG_MARK,
                "sigaction(SIGPIPE)");
#endif
    
    /* we want to ignore HUPs and AP_SIG_GRACEFUL while we're busy
     * processing one */
    sigaddset(&sa.sa_mask, SIGHUP);
    sa.sa_handler = restart;
    if (sigaction(SIGHUP, &sa, NULL) < 0)
      rovm_log (r, ROVMLOG_WARNING, ROVMLOG_MARK,
                   "sigaction(SIGHUP)");
#else
    if (!CONF_ONE_PROCESS (ROVM_CONF (r))) 
      {
#ifdef SIGXCPU
        rc_signal (SIGXCPU, SIG_DFL);
#endif /* SIGXCPU */
#ifdef SIGXFSZ
        rc_signal (SIGXFSZ, SIG_DFL);
#endif /* SIGXFSZ */
      }
    
    rc_signal (SIGTERM, sig_term);
#ifdef SIGHUP
    rc_signal (SIGHUP, restart);
#endif /* SIGHUP */
#ifdef SIGPIPE
    rc_signal (SIGPIPE, SIG_IGN);
#endif /* SIGPIPE */

#endif
}


static void
close_worker_sockets (thread_starter *ts)
{
  int i;
  struct rovm *r = (struct rovm *) ts->arg;

  for (i = 0; i < CONF_THREADS_PER_CHILD (ROVM_CONF (r)); i++) 
    {
      if (ts->worker_sockets[i]) 
        {
          rc_socket_close (ts->worker_sockets[i]);
          ts->worker_sockets[i] = NULL;
        }
    }
}

static void clean_child_exit (rc_pool_t *pchild, int code)
{
  if (pchild) 
    mp_destroy (pchild);

  exit(code);
}

static void 
join_start_thread (rc_thread_t *start_thread_id)
{
  rc_status_t rv, thread_rv;
  
  rv = rc_thread_join (&thread_rv, start_thread_id);
  if (rv != RC_SUCCESS) 
    {
      rovm_log (NULL, ROVMLOG_CRIT, ROVMLOG_MARK,
		"rc_thread_join: unable to join the start "
		"thread");
    }
}

static void
wakeup_listener (thread_starter *ts)
{
/* The LISTENER_SIGNAL signal will be sent from the main thread to the
 * listener thread to wake it up for graceful termination (what a child
 * process from an old generation does when the admin does "apachectl
 * graceful").  This signal will be blocked in all threads of a child
 * process except for the listener thread.
 */
#define LISTENER_SIGNAL     SIGHUP

  ts->listener_may_exit = 1;

  if (!ts->listener_os_thread) 
    {
      /* XXX there is an obscure path that this doesn't handle perfectly:
       *     right after listener thread is created but before
       *     listener_os_thread is set, the first worker thread hits an
       *     error and starts graceful termination
       */
      return;
    }
  /*
   * we should just be able to "kill(ap_my_pid, LISTENER_SIGNAL)" on all
   * platforms and wake up the listener thread since it is the only thread
   * with SIGHUP unblocked, but that doesn't work on Linux
   */
#ifdef HAVE_PTHREAD_KILL
  pthread_kill(*ts->listener_os_thread, LISTENER_SIGNAL);
#else
  kill (ts->my_pid, LISTENER_SIGNAL);
#endif
}

#define ST_INIT              0
#define ST_GRACEFUL          1
#define ST_UNGRACEFUL        2

static void
signal_threads (thread_starter *ts, int mode)
{
  if (ts->terminate_mode == mode) 
    return;

  ts->terminate_mode = mode;
  
  /* in case we weren't called from the listener thread, wake up the
   * listener thread
   */
  wakeup_listener (ts);
  
  /* for ungraceful termination, let the workers exit now;
   * for graceful termination, the listener thread will notify the
   * workers to exit once it has stopped accepting new connections
   */
  if (mode == ST_UNGRACEFUL) 
    {
      ts->workers_may_exit = 1;
      rc_queue_interrupt_all (ts->worker_queue);
      rc_queue_info_term (ts->worker_queue_info);
      close_worker_sockets (ts);
    }
}

static void
join_workers (thread_starter *ts, 
	      rc_thread_t *listener, 
	      rc_thread_t **threads)
{
  int i;
  rc_status_t rv, thread_rv;
  
  if (listener) 
    {
      int iter;
      
      /* deal with a rare timing window which affects waking up the
       * listener thread...  if the signal sent to the listener thread
       * is delivered between the time it verifies that the
       * listener_may_exit flag is clear and the time it enters a
       * blocking syscall, the signal didn't do any good...  work around
       * that by sleeping briefly and sending it again
       */
      iter = 0;
      while (iter < 10 &&
#ifdef HAVE_PTHREAD_KILL
	     pthread_kill (*ts->listener_os_thread, 0)
#else
	     kill (ts->my_pid, 0)
#endif
	     == 0) 
	{
	  /* listener not dead yet */
	  rc_sleep (rc_time_make (0, 500000));
	  wakeup_listener (ts);
	  ++iter;
        }
      if (iter >= 10) 
	rovm_log (NULL, ROVMLOG_DEBUG, ROVMLOG_MARK, "the listener thread didn't exit");
      else 
	{
	  rv = rc_thread_join (&thread_rv, listener);
	  if (rv != RC_SUCCESS) 
	    {
	      rovm_log (NULL, ROVMLOG_CRIT, ROVMLOG_MARK,
			"rc_thread_join: unable to join listener thread");
            }
        }
    }
  
  for (i = 0; i < CONF_THREADS_PER_CHILD (ROVM_CONF ((struct rovm *) ts->arg)); i++) 
    {
      if (threads[i])
	{ /* if we ever created this thread */
	  rv = rc_thread_join (&thread_rv, threads[i]);
	  if (rv != RC_SUCCESS)
	    {
	      rovm_log (NULL, ROVMLOG_CRIT, ROVMLOG_MARK,
			"rc_thread_join: unable to join worker "
			"thread %d",
			i);
            }
        }
    }
}

static void 
unblock_signal(int sig)
{
  sigset_t sig_mask;
  
  sigemptyset (&sig_mask);
  sigaddset (&sig_mask, sig);
#if defined(SIGPROCMASK_SETS_THREAD_MASK)
  sigprocmask (SIG_UNBLOCK, &sig_mask, NULL);
#else
  pthread_sigmask (SIG_UNBLOCK, &sig_mask, NULL);
#endif
}

static void
dummy_signal_handler (int sig)
{
  /* XXX If specifying SIG_IGN is guaranteed to unblock a syscall,
         then we don't need this goofy function.  */
}

static void
rc_close_listeners (struct rovm *r)
{
  rc_listen_rec *lr;

  for (lr = ROVM_LISTENERS (r); lr; lr = lr->next) 
    {
      rc_socket_close (lr->sd);
      lr->active = 0;
    }
}

static void *
listener_thread (rc_thread_t *thd, void *arg)
{
  void *csd = NULL;
  int have_idle_worker = 0;
  rc_status_t rv;
  rc_pool_t *ptrans = NULL;            /* Pool for per-transaction stuff */
  proc_info *ti = (proc_info *) arg;
  thread_starter *ts = (thread_starter *) ti->ts;
  struct rovm *r = (struct rovm *) ts->arg;
  rc_listen_rec *lr;

  /* Unblock the signal used to wake this thread up, and set a handler for
     it.  */
  unblock_signal (LISTENER_SIGNAL);
  rc_signal (LISTENER_SIGNAL, dummy_signal_handler);

  /*
     listener_thread  multi-listening  ϰ  ʴ.  ׷ ׿ 
    poll  apache  , ⿡  ʴ´.
   */

  while (1)
    {
      if (ts->listener_may_exit)
        break;
      
      if (!have_idle_worker) 
        {
          /* the following pops a recycled ptrans pool off a stack
           * if there is one, in addition to reserving a worker thread
           */
          rv = rc_queue_info_wait_for_idler (ts->worker_queue_info, &ptrans);
          if (RC_STATUS_IS_EOF (rv))
            break; /* we've been signaled to die now */
          else if (rv != RC_SUCCESS) 
            {
              rovm_log (NULL, ROVMLOG_EMERG, ROVMLOG_MARK,
                        "rc_queue_info_wait failed. Attempting to "
                        " shutdown process gracefully.");
              signal_threads (ts, ST_GRACEFUL);
              break;
            }
          have_idle_worker = 1;
        }

      if (!ROVM_LISTENERS (r)->next)
        /* Only one listener, so skip the poll */
        lr = ROVM_LISTENERS (r);
      else
        {
          rovm_log (NULL, ROVMLOG_EMERG, ROVMLOG_MARK,
                    "Not supported multi listening yet.");
          signal_threads (ts, ST_GRACEFUL);
          break;
        }
      
      if (!ts->listener_may_exit)
        {
          if (ptrans == NULL) 
            {
              /* we can't use a recycled transaction pool this time.
               * create a new transaction pool */
              rc_allocator_t *allocator;
              
              rc_allocator_create (&allocator);
              mp_create_ex (&ptrans, PROCREC_PCONF (ROVM_PROCESS (r)), NULL, allocator);
              rc_allocator_owner_set (allocator, ptrans);
            }
          mp_tag (ptrans, "transaction");
          rv = lr->accept_func (&csd, lr, ptrans);
          if (rv == RC_EGENERAL) 
            {
              /* E[NM]FILE, ENOMEM, etc */
              ts->resource_shortage = 1;
              signal_threads (ts, ST_GRACEFUL);
            }
          if (csd != NULL) 
            {
              rv = rc_queue_push (ts->worker_queue, csd, ptrans);
              if (rv) 
                {
                  /* trash the connection; we couldn't queue the connected
                   * socket to a worker
                   */
                  rc_socket_close(csd);
                  rovm_log (NULL, ROVMLOG_CRIT, ROVMLOG_MARK, "ap_queue_push failed");
                }
              else 
                have_idle_worker = 0;
            }
        }
    }

  rc_close_listeners (r);
  rc_queue_term (ts->worker_queue);

  /* wake up the main thread */
  kill (ts->my_pid, SIGTERM);

  rc_thread_exit (thd, RC_SUCCESS);
  
  return NULL;
}

/*****************************************************************
  Child process main loop.

   Լ ȣ Ǿٴ  Ʒ   Ѵٰ    ̴.

    1) Listener Thread  Worker Thread   ߵǰ ִ.
    2) Socket   , ̸ ̿Ͽ  ڰ  
         ִ.
    3)  ڰ  buffer  ó ð̴.
    4) ޸ POOL  P  ̿ϸ ó     
        ̴.  ׸    ̴.
*/

static void
process_socket (thread_starter *ts, rc_pool_t *p, rc_socket_t *sock)
{
  conn_rec *current_conn;

  current_conn = rc_create_connection (ts, p, sock);
  if (current_conn)
    {
      rc_process_connection (current_conn);
      rc_close_connection (current_conn);
    }
}

static void * 
worker_thread (rc_thread_t *thd, void *arg)
{
  proc_info *ti = (proc_info *) arg;
  thread_starter *ts = ti->ts;
  int thread_slot = ti->tid;
  int is_idle = 0;
  rc_status_t rv;
  rc_socket_t *csd = NULL;
  rc_pool_t *last_ptrans = NULL;
  rc_pool_t *ptrans;                /* Pool for per-transaction stuff */

  while (!ts->workers_may_exit) 
    {
      if (!is_idle) 
        {
          rv = rc_queue_info_set_idle (ts->worker_queue_info, last_ptrans);
          last_ptrans = NULL;
          if (rv != RC_SUCCESS) 
            {
              rovm_log (NULL, ROVMLOG_EMERG, ROVMLOG_MARK,
                        "rc_queue_info_set_idle failed. Attempting to "
                        "shutdown process gracefully.");
              signal_threads (ts, ST_GRACEFUL);
              break;
            }
          is_idle = 1;
        }

worker_pop:
      if (ts->workers_may_exit)
        break;

      rv = rc_queue_pop (ts->worker_queue, &csd, &ptrans);
      
      if (rv != RC_SUCCESS) 
        {
          /* We get RC_EOF during a graceful shutdown once all the connections
           * accepted by this server process have been handled.
           */
          if (RC_STATUS_IS_EOF (rv)) 
            break;

          /* We get RC_EINTR whenever ap_queue_pop() has been interrupted
           * from an explicit call to ap_queue_interrupt_all(). This allows
           * us to unblock threads stuck in ap_queue_pop() when a shutdown
           * is pending.
           *
           * If workers_may_exit is set and this is ungraceful termination/
           * restart, we are bound to get an error on some systems (e.g.,
           * AIX, which sanity-checks mutex operations) since the queue
           * may have already been cleaned up.  Don't log the "error" if
           * workers_may_exit is set.
           */
          else if (RC_STATUS_IS_EINTR (rv))
            goto worker_pop;
          /* We got some other error. */
          else if (!ts->workers_may_exit)
            rovm_log (NULL, ROVMLOG_CRIT, ROVMLOG_MARK, "ap_queue_pop failed");

          continue;
        }

      is_idle = 0;
      ts->worker_sockets[thread_slot] = csd;
      process_socket (ts, ptrans, csd);
      ts->worker_sockets[thread_slot] = NULL;
      mp_clear (ptrans);
      last_ptrans = ptrans;
    }

  rc_thread_exit (thd, RC_SUCCESS);
  return NULL;
}

static void
create_listener_thread (thread_starter *ts)
{
  int my_child_num = ts->child_num_arg;
  rc_threadattr_t *thread_attr = ts->threadattr;
  proc_info *my_info;
  rc_status_t rv;

  my_info = (proc_info *) malloc (sizeof (proc_info));
  my_info->pid = my_child_num;
  my_info->tid = -1; /* listener thread doesn't have a thread slot */
  my_info->sd = 0;
  my_info->ts = ts;

  rv = rc_thread_create (&ts->listener, thread_attr, listener_thread, my_info, ts->p);
  if (rv != RC_SUCCESS) 
    {
      rovm_log (NULL, ROVMLOG_ALERT, ROVMLOG_MARK,
		"rc_thread_create: unable to create listener thread");
      clean_child_exit (ts->p, RCEXIT_CHILDSICK);
    }

  rc_os_thread_get (&ts->listener_os_thread, ts->listener);
}

/**
   ϳ μ  CONF_THREADS_PER_CHILD  شϴ worker thread 
   ϳ listener thread  open Ѵ.
 */
static void * 
start_threads (rc_thread_t *thd, void *arg)
{
  rc_status_t rv;
  proc_info *my_info;
  int i, listener_started = 0, threads_created = 0;
  int loops = 0, prev_threads_created = 0;
  thread_starter *ts = (thread_starter *) arg;
  int child_num_arg = ts->child_num_arg;
  int my_child_num = child_num_arg;
  rc_thread_t **threads = ts->threads;
  rc_threadattr_t *thread_attr = ts->threadattr;
  rc_pool_t *pchild = ts->p;
  struct rovm *r = (struct rovm *) ts->arg;

  /* We must create the fd queues before we start up the listener
   * and worker threads. */
  ts->worker_queue = mp_calloc (pchild, sizeof (*ts->worker_queue));
  rv = rc_queue_init (ts->worker_queue, CONF_THREADS_PER_CHILD (ROVM_CONF (r)), pchild);
  if (rv != RC_SUCCESS) 
    {
      rovm_log (NULL, ROVMLOG_ALERT, ROVMLOG_MARK, "rc_queue_init () failed");
      clean_child_exit (pchild, RCEXIT_CHILDFATAL);
    }

  rv = rc_queue_info_create (&ts->worker_queue_info, pchild, CONF_THREADS_PER_CHILD (ROVM_CONF (r)));
  if (rv != RC_SUCCESS) 
    {
      rovm_log (NULL, ROVMLOG_ALERT, ROVMLOG_MARK, "ap_queue_info_create() failed");
      clean_child_exit (pchild, RCEXIT_CHILDFATAL);
    }

  ts->worker_sockets = mp_calloc (pchild, CONF_THREADS_PER_CHILD (ROVM_CONF (r)) * sizeof(rc_socket_t *));
  
  while (1) 
    {
      /* threads_per_child does not include the listener thread */
      for (i = 0; i < CONF_THREADS_PER_CHILD (ROVM_CONF (r)); i++)
	{
	  my_info = (proc_info *) malloc (sizeof (proc_info));
	  if (my_info == NULL) 
	    {
	      rovm_log (NULL, ROVMLOG_ALERT, ROVMLOG_MARK, "malloc: out of memory");
	      clean_child_exit (ts->p, RCEXIT_CHILDFATAL);
            }
	  my_info->pid = my_child_num;
	  my_info->tid = i;
	  my_info->sd = 0;
          my_info->ts = ts;

	  rv = rc_thread_create (&threads[i], thread_attr, worker_thread, my_info, pchild);
	  if (rv != RC_SUCCESS)
	    {
	      rovm_log (NULL, ROVMLOG_ALERT, ROVMLOG_MARK, 
			"rc_thread_create: unable to create worker thread");
	      clean_child_exit (ts->p, RCEXIT_CHILDSICK);
            }
	  threads_created++;
        }

      /* Start the listener only when there are workers available */
      if (!listener_started && threads_created) 
	{
	  create_listener_thread (ts);
	  listener_started = 1;
	}

      if (threads_created == CONF_THREADS_PER_CHILD (ROVM_CONF (r))) 
	break;

      /* wait for previous generation to clean up an entry */
      rc_sleep (rc_time_from_sec (1));
      ++loops;

      if (loops % 120 == 0) 
	{ /* every couple of minutes */
	  if (prev_threads_created == threads_created) 
	    {
	      rovm_log (NULL, ROVMLOG_DEBUG, ROVMLOG_MARK,
			"child %d isn't taking over "
			"slots very quickly (%d of %d)",
			ts->my_pid, threads_created, CONF_THREADS_PER_CHILD (ROVM_CONF (r)));
	    }
	  prev_threads_created = threads_created;
	}
    }
  
  rc_thread_exit (thd, RC_SUCCESS);

  return NULL;  
}


static int 
check_signal (int signum)
{
  switch (signum) 
    {
    case SIGTERM:
    case SIGINT:
      return 1;
    }
  return 0;
}

/**
   (ټ fork   ) Children μ̸,   listener thread 
   worker thread  ϰ ȴ.

   @param r		ROVM ü
 */
void
child_main (r)
     struct rovm *r;
{
  rc_status_t rv;
  thread_starter *ts;
  rc_threadattr_t *thread_attr;
  rc_thread_t *start_thread_id, **threads;
  rc_pool_t *pchild;

  mp_create (&pchild, PROCREC_PCONF (ROVM_PROCESS (r)));

  threads = (rc_thread_t **) rc_calloc (1, sizeof (rc_thread_t *) * CONF_THREADS_PER_CHILD (ROVM_CONF (r)));
  if (!threads)
    {
      rovm_log (NULL, ROVMLOG_ERR, ROVMLOG_MARK, "malloc: out of memory");
      clean_child_exit (pchild, RCEXIT_CHILDFATAL);
    }
  
  rc_threadattr_create (&thread_attr, pchild);
  /* 0  PTHREAD_CREATE_JOINABLE  ǹѴ.  */
  rc_threadattr_detach_set (thread_attr, 0);

  ts = (thread_starter *) mp_alloc (pchild, sizeof (*ts));
  ts->threads = threads;
  ts->listener = NULL;
  ts->child_num_arg = -1;
  ts->threadattr = thread_attr;
  ts->p = pchild;
  ts->my_pid = getpid ();
  ts->arg = (void *) r;
  ts->listener_may_exit = 0;
  ts->workers_may_exit = 0;
  ts->resource_shortage = 0;
  ts->terminate_mode = ST_INIT;

  rv = rc_thread_create (&start_thread_id, thread_attr, start_threads, ts, pchild);
  if (rv != RC_SUCCESS) 
    {
      rovm_log (NULL, ROVMLOG_ERR, ROVMLOG_MARK,
		"rc_thread_create: unable to create worker thread");
      clean_child_exit (pchild, RCEXIT_CHILDSICK);
    }

  if (CONF_ONE_PROCESS (ROVM_CONF (r)))
    {
      /* Block until we get a terminating signal. */
      rc_signal_thread (check_signal);
      /* make sure the start thread has finished; signal_threads()
         and join_workers() depend on that  */
      /* XXX join_start_thread() won't be awakened if one of our
             threads encounters a critical error and attempts to
             shutdown this child  */
      join_start_thread (start_thread_id);
      signal_threads (ts, ST_UNGRACEFUL);
      /* A terminating signal was received. Now join each of the workers to clean them up.
         If the worker already exited, then the join frees their resources and returns.
         If the worker hasn't exited, then this blocks until they have (then cleans up).  */
      join_workers (ts, ts->listener, threads);
    }
  else
    {
      while (1)
        {
          rv = rc_mpm_pod_check (ROVM_POD (r));
          if (rv == RC_POD_NORESTART)
            {
              /* see if termination was triggered while we slept */
              switch (ts->terminate_mode) 
                {
                case ST_GRACEFUL:
                  rv = RC_POD_GRACEFUL;
                  break;
                case ST_UNGRACEFUL:
                  rv = RC_POD_RESTART;
                  break;
                }
            }
          if (rv == RC_POD_GRACEFUL || rv == RC_POD_RESTART)
            {
              /* make sure the start thread has finished;
               * signal_threads() and join_workers depend on that
               */
              join_start_thread (start_thread_id);
              signal_threads (ts, rv == RC_POD_GRACEFUL ? ST_GRACEFUL : ST_UNGRACEFUL);
              break;
            }
        }

      /* A terminating signal was received. Now join each of the
       * workers to clean them up.
       *   If the worker already exited, then the join frees
       *   their resources and returns.
       *   If the worker hasn't exited, then this blocks until
       *   they have (then cleans up).
       */
      join_workers (ts, ts->listener, threads);
    }

  rc_free (threads);
  
  clean_child_exit (pchild, ts->resource_shortage ? RCEXIT_CHILDSICK : 0);
}

/**
   MPM  main loop   ̾ Լ̴. .;    apache  
   fork  ؼ  ڽ process   ,  thread  ѹ  ̾µ,
    fork   ʱ , Լ   .    Ʈ
     Լ̴.

   @param r     ROVM ü
 */
int
mpm_main_loop (r)
     struct rovm *r;
{
  child_main (r);
  
  return 0;
}

/**
   MPM  entry .

   @param r     ROVM ü

   @return   0  ȯϸ, ׷  , -1  ȯѴ.
 */
int
mpm_run (r)
     struct rovm *r;
{
  rc_status_t rv;

  if (!CONF_ONE_PROCESS (ROVM_CONF (r)))
    {
      if ((rv = rc_mpm_pod_open (PROCREC_PCONF (ROVM_PROCESS (r)), &ROVM_POD (r), r)))
        {
          rovm_log (NULL, ROVMLOG_CRIT, ROVMLOG_MARK, "Could not open pipe-of-death.");
          return -1;
        }
    }

  set_signals (r);
  
  mpm_main_loop (r);

  return 0;
}
