/* rovm.c - ROVM Server  main .
   Copyright (C) 2006 Weongyo Jeong (weongyo@gmail.com)

This file is part of ROVM.

ROVM is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

ROVM is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.  */

#include <stdlib.h>

#include "types.h"
#include "mpool.h"
#include "tktree.h"
#include "clstree.h"
#include "sha1.h"
#include "listen.h"
#include "rovm.h"

#include "thread.h"
#include "thread_mutex.h"
#include "thread_cond.h"

#include "mpm_worker_fdqueue.h"
#include "mpm_worker_pod.h"
#include "mpm_worker.h"

#include "connection.h"
#include "request.h"
#include "proc_rc.h"

#include "log.h"
#include "common.h"
#include "ticket.h"

#include "gc.h"

/**
   ROVM  ߽ Լ.   Լ MPM  Ѵ.

   @param r	ROVM ü.
*/
int
rovm_main (r)
     struct rovm *r;
{
  gc_run (r);
  mpm_run (r);

  return 0;
}

/**
   Process     ü Ѵ.

   @param argc	Argument Count
   @param argv	Argument Value 
   @return	Ӱ  struct process_rec ü ȯѴ.
 */
static struct process_rec *
create_process_rec (argc, argv)
     int argc;
     const char * const *argv;
{
  struct process_rec *process;
  rc_pool_t *cntx;
  rc_status_t stat;
  
  stat = mp_create (&cntx, NULL);
  if (stat != RC_SUCCESS)
    {
      /* XXX From the time that we took away the NULL pool->malloc mapping
	     we have been unable to log here without segfaulting.  */
      rovm_log (NULL, ROVMLOG_ERR, ROVMLOG_MARK,
		"rc_pool_create() failed to create initial context");
      rc_terminate ();
      exit (1);
    }

  mp_tag (cntx, "process");
  
  process = mp_alloc (cntx, sizeof (struct process_rec));
  process->pool = cntx;
  
  mp_create (&process->pconf, process->pool);
  mp_tag (process->pconf, "pconf");
  process->argc = argc;
  process->argv = argv;
  process->short_name = rc_filepath_name_get (argv[0]);

  return process;
}

/**
   ROVM Server   ¸ ʱȭմϴ.

   @param r	ROVM ü
   @param p	޸ pool
 */
int
init_rovm_status (r, p)
     struct rovm *r;
     rc_pool_t *p;
{
  struct rovm_status *rs;

  rs = (struct rovm_status *) mp_alloc (p, sizeof (struct rovm_status));
  if (!rs)
    return -1;

  STATUS_SHUTDOWN_PENDING (rs) = 0;
  STATUS_RESTART_PENDING (rs) = 0;

  ROVM_STATUS (r) = rs;

  return 0;
}

/**
   ROVM  õ   ʱȭѴ.

   @param r	ROVM ü
   @param p	޸ pool
 */
int
init_rovm_config (r, p)
     struct rovm *r;
     rc_pool_t *p;
{
#define DEFAULT_LISTENBACKLOG 511
  struct rovm_config *c;

  c = (struct rovm_config *) mp_calloc (p, sizeof (struct rovm_config));
  if (!c)
    return -1;

  CONF_ONE_PROCESS (c) = 1;

  CONF_PORT (c) = DEFAULT_ROVM_LISTENING_PORT;
  CONF_THREADS_PER_CHILD (c) = 5;

  /* Listening    */
  CONF_SEND_BUFFER_SIZE (c) = 0;        /* ý default  ϱ ؼ 0  ؾ Ѵ.  */
  CONF_RECEIVE_BUFFER_SIZE (c) = 0;     /* ý default  ϱ ؼ 0  ؾ Ѵ.  */
  CONF_LISTENBACKLOG (c) = DEFAULT_LISTENBACKLOG;

  CONF_ERROR_LOG (c) = stderr;
  CONF_LOGLEVEL (c) = DEFAULT_LOGLEVEL;

  CONF_OPTION (c) |= OPTION_LOG_DEBUG_MESSAGES;

  /* Ticket ü ⺻  stack slot   ϴ 
     κԴϴ.  ⺻ DEFAULT_STACK_SLOT   Ǿ ֽϴ. */
  CONF_DEFAULT_STACK_SLOT (c) = DEFAULT_STACK_SLOT;

  /* Class ϵ root θ ϰ ˴ϴ.   ο 
     e://<hostname>[:<port>]<path> URL  <path>  Ͽ ϳ ý
     θ Ű ˴ϴ.

     (20050205)  ROVM Server  ,    ʱ 
     ڰ `NEW' Opcode  ׽Ʈϱ ؼ  κ  ϵ
     ȳ  ҰԴϴ.  */
  CONF_CLASSROOT (c) = "/usr/local/rovm/classroot";

  ROVM_CONF (r) = c;

  return 0;
}

/**
   ROVM  ʿ ʱȭ մϴ.

   @return	ʱȭ struct rovm *  ȯմϴ.
*/
struct rovm *
init_rovm (argc, argv)
     int argc;
     char *argv[];
{
  rc_status_t rv;
  struct rovm *r;
  struct process_rec *p;

  rc_initialize ();

  p = create_process_rec (argc, argv);

  r = (struct rovm *) mp_calloc (PROCREC_POOL (p), sizeof (struct rovm));
  if (!r)
    return NULL;

  ROVM_PROCESS (r) = p;

  rv = rc_thread_mutex_create (&ROVM_MUTEX (r), RC_THREAD_MUTEX_DEFAULT, PROCREC_POOL (p));
  if (rv != RC_SUCCESS)
    return NULL;
  
  /* Ticket TREE  ʱȭѴ.  */
  if (init_rovm_tktree (r))
    return NULL;
  /* Class TREE  ʱȭմϴ.  */
  if (init_rovm_clstree (r))
    return NULL;
  /* ROVM Configuration  κ Ѵ.  */
  if (init_rovm_config (r, PROCREC_POOL (p)))
    return NULL;
  /* ROVM Server    ʱȭմϴ.  */
  if (init_rovm_status (r, PROCREC_POOL (p)))
    return NULL;
  /* Listening Socket   ʱȭ ǽմϴ.  */
  if (init_rovm_listen (r, PROCREC_POOL (p)))
    return NULL;
  /* Garbage Collector   ʱȭ ǽմϴ.  */
  if (init_rovm_gc (r, PROCREC_POOL (p)))
    return NULL;

  return r;
}

/**
   ROVM  մϴ. 

   @param r	ROVM     ü.
 */
int
finish_rovm (r)
     struct rovm *r;
{
  rc_terminate ();

  return 0;
}

/**
   ROVM  main Լ.
 */
int
main (argc, argv)
     int argc;
     char *argv[];
{
  struct rovm *r;

  r = init_rovm (argc, argv);
  if (!r)
    return -1;

  rovm_main (r);

  finish_rovm (r);

  return 0;
}
