package org.archi.tools.excatj;

import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.apache.log4j.Logger;

public class TextOutput {
    private StringBuffer strBuf = new StringBuffer();
    private Logger logger;
    /** 
     * The resource bundle containing localizable message content.
     * This is loaded by TTY.main() at start-up
     */
    static ResourceBundle textResources;

    /** Our message formatter.  Allocated once, used many times */
    private static MessageFormat messageFormat;

    private Logger getLogger() {
    	return logger;
    }
    
    public TextOutput(String loggerName) {
    	logger = Logger.getLogger(loggerName);
    	
        textResources = ResourceBundle.getBundle("org.archi.tools.excatj.TTYResources",
        					Locale.getDefault());    	
    }
    
	private void error(Object message) {
		getLogger().error(message);
	}
	
	private void fatal(Object message) {
		getLogger().fatal(message);
	}
	

    /**
     * Fatal shutdown notification.  This is sent to System.err
     * instead of System.out
     */
    public void fatalError(String messageKey) {
        getLogger().fatal("");
        getLogger().fatal(format("Fatal error"));
        getLogger().fatal(format(messageKey));
        Env.shutdown();
    }

    public void printDevTrace(String line) {
    	getLogger().trace(line);
    }
    
    /**
     * "Format" a string by doing a simple key lookup.
     */
    public String format(String key) {
        return (textResources.getString(key));
    }    

    /**
     * Fetch and format a message with one string argument.
     * This is the most common usage.
     */
    public String format(String key, String argument) {
        return format(key, new Object [] {argument});
    }

    /**
     * Fetch a string by key lookup and format in the arguments.
     */
    public synchronized String format(String key, Object [] arguments) {
        if (messageFormat == null) {
            messageFormat = new MessageFormat (textResources.getString(key));
        } else {
            messageFormat.applyPattern (textResources.getString(key));
        }
        return (messageFormat.format (arguments));
    }

    /**
     * Print directly to System.out.  
     * Every rule has a few exceptions.
     * The exceptions to "must use the MessageOutput formatters" are:
     *     VMConnection.dumpStream()
     *     TTY.monitorCommand()
     *     TTY.TTY() (for the '!!' command only)
     *     Commands.java (multiple locations)
     * These are the only sites that should be calling this
     * method.
     */
    public void printDirectln(String line) {
    	getLogger().info(strBuf+line);
        strBuf = new StringBuffer(); 
    }
	public void printDirectln(Object message) {
		getLogger().info(strBuf+message.toString());
        strBuf = new StringBuffer();
	}
    public void printDirect(String line) {
    	strBuf.append(line);
    }
    public void printDirect(char c) {
        printDirectln(c);
    }

    /**
     * Print a newline.
     * Use this instead of '\n'
     */
    public void println() {
    	printDirectln("");
    }

    /**
     * Format and print a simple string.
     */
    public void print(String key) {
    	printDirect(format(key));
    }
    
    /**
     * Format and print a simple string.
     */
    public void println(String key) {
        printDirectln(format(key));
    }

    /**
     * Fetch, format and print a message with one string argument.
     * This is the most common usage.
     */
    public void print(String key, String argument) {
        printDirect(format(key, argument));
    }
    public void println(String key, String argument) {
        printDirectln(format(key, argument));
    }
    /**
     * Fetch, format and print a message with an arbitrary
     * number of message arguments.
     */
    public void println(String key, Object [] arguments) {
        printDirectln(format(key, arguments));
    }    

    /**
     * Print a newline, followed by the string.
     */
    public void lnprint(String key) {
        println();
        printDirect(textResources.getString(key));
    }

    public void lnprint(String key, String argument) {
        println();
        print(key, argument);
    }

    public void lnprint(String key, Object [] arguments) {
        println();
        println(key, arguments);
    }

    /**
     * Print an exception message with a stack trace.
     */
    public void printException(String key, Exception e) {
        if (key != null) {
            try {
                printDirectln(key);
            } catch (MissingResourceException mex) {
                printDirectln(key);
            }   
        }
        System.out.flush();
        e.printStackTrace();
    }    
}
