<?php
/**
 * 글 관련 라이브러리
 *
 * 글은 일반 포스트를 포함해서 게시물, 메모장, 일기장, 다이어리, 기타 등등 텍스트 정보를 말한다.
 * 글과 관련된 작업을 하기 위해서는 반드시 이 라이브러리를 사용해야한다.
 * 예를 들면, 지금은 그냥 글을 DB 에 바로 보관하지만, 차후 인덱싱을 하도록 변경이 될 수가 있다.
 * 이 경우, 인덱싱을 하는 글과 하지 않은 글들은 심각한 차이를 보일 수 있다. 이러한 일관성을 갖추기 위해서라도 반드시 글 관련 작업은 모두 이 라이블러리를 통해서 한다.
 *
 * @changed bbs.php 에서 post.php 로 변경되었다.
 * @note 모든 text 형 게시물(내용)을 담당하며, post 테이블과 연관된 작업을 한다.
 * @note post 테이블에는 글(포스트)의 분류가 있다. 이 분류가 카테고리이다. post.idx_category 에 해당 글(레코드)이 속해있는 category.idx 가 기록이된다.
 *
 * @note 모든 글 관련 작업을 할 때, 아이피, 날짜-시간 관련 작업은 내부적으로 자동 처리가된다. 따라서 직접 이런 값을 입력하지 않도록 한다.
 *
 * @version 20061118
 * @author thruthesky thruthesky@yahoo.co.kr
 * @package library
 */
/**
 * 데이터베이스 라이브러리 로드
 */
lib('db');

/**
 * 글을 데이터베이스에 저장한다.
 *
 * 글에는 쇼핑몰 아이템 관련 글이 있을 수 있고, 메모 글, 블로그 글, 게시판 글 등 다양하다.
 * 이 함수의 입력값에 글이 어떤 (카테고리)종류 인지 idx_category_type 에 기록을 해야한다.
 * @see BUILDGUIDE
 * 글 작성 시간, IP, user_agent 는 자동으로 입력된다.
 *
 * @param associative-array $post $post[idx_category] 는 필수 항목이다. 0 보다 크거나 같아야한다.
 * @return faultCode
 * @note post['idx_category'] (post.idx_category) 는 필수 항목이다.
 * <code>
 * 	$data[title] = 'abc';
 *	$data[description] = 'def';
 *	insertPost($data);
 * </code>
 *
 * @note 기본 값이 빠져 있으면 설정을 해서 업데이트한다.
 */
function insertPost($post)
{
	/**@since 2007/01/24 카테고리 번호는 0 이상의 값을 가진다.*/
	//if ( ! positive($post['idx_category']) ) return faultEx(wrongParam, "No category has been specified for insertPost");
	if ( $post['idx_category'] < 0 ) return faultEx(wrongParam, "No category has been specified for insertPost");
	if ( ! isset($post['dateTime']) ) $post['dateTime'] = dateTime();
	$post['dateTime_firstwrite'] = $post['dateTime'];
	if ( ! isset($post['ip']) ) $post['ip'] = $_SERVER['REMOTE_ADDR'];
	if ( ! isset($post['user_agent']) ) $post['user_agent'] = $_SERVER['HTTP_USER_AGENT'];
	
	return faultDb($GLOBALS['db']->insert('post', $post));
}



/**
 * post.idx 을 기반으로 글 내용을 업데이트한다.
 *
 * 입력 변수 $post 에는 idx 값이 들어있어야한다.
 * ip 는 갱신을 하고, dateTime 은 맨 처음 작성된 것으로 유지를 한다. 다만, dateTime_lastmodified 에 마지막에 기록된 dateTime 을 보관한다.
 *
 * 이 함수는 글 자체를 업데이트한다. 이 함수가 불려질 때마다, 글 정보가 기록된 시간, IP 등이 변경이된다.
 * 만약, 시간, IP 등의 값이 변경되지 않기를 원한다면,, updatePostField 를 이용하기 바란다.
 * 그리고 처음 작성된 시간이 필요하면, dateTime_firstwrite 값을 이용하도록 한다.
 * @see 개발자노트.정보의 수정
 * @param associative-array $post 카테고리 정보를 가지고 있는 연관 배열.
 * @return faultCode 성공시 ok, 실패시 에러코드
 * @note 만약 post.idx 값이 입력되지 않았으면 faultCode 가 리턴된다.
 *
 */
function updatePost($post)
{
	if ( empty($post['idx']) ) return faultEx(wrongIndex, "No post.idx");
	if ( ! isset($post['dateTime']) ) $post['dateTime'] = dateTime();
	if ( ! isset($post['ip']) ) $post['ip'] = $_SERVER['REMOTE_ADDR'];
	$post['user_agent']				= $_SERVER['HTTP_USER_AGENT'];
	global $db;
	
	$rc = $db->update('post', $post, array('idx'=>$post['idx']));
	return faultDb($rc);
}




/**
 * 글의 특정 항목을 수정한다.
 *
 * 입력되는 idx의 글 번호에 대해서 코멘트 수, 파일 수, 조회 수 등을 수정한다.
 * 입력 값으로는 레코드의 항목 이름과 숫자값이다.
 * 즉, 글의 레코드 항목에 숫자를 더하거나 뺀다.
 *
 * @param int $idx_parent 글 번호. 수정될 글의 레코드 번호이다. 즉, 부모글의 번호가 입력되어야한다.
 * @param string $field 레코드 항목
 * @param int $value 변경될 값(더하거나 빼기에서는 "field +(-)" 값 의 형태로 입력되어야한다.
 * @see updateParent
 * @return faultCode
 * @code updatePostfield($ui['idx'], "state", "'$ui[state]'");
 * @since 2007/02/09 변경하고자 하는 값이 문자열이면 홑 따옴표로 감싸야한다.
 */
function updatePostField($idx, $field, $value)
{
	if ( empty($idx) ) return faultEx(wrongParam, "post::updateParent wrong idx ($idx)");
	if ( empty($field) ) return faultEx(wrongParam, "post::updateParent wrong field ($field)");
	if ( empty($value) ) return faultEx(wrongParam, "post::updateParent wrong value ($value)");
	
	return faultDb( $GLOBALS['db']->query("UPDATE post SET $field =$value WHERE idx=$idx") );
}

/**
 * @see updatePostField
 */
function updatePostAttr($idx, $field, $value) { return updatePostField($idx, $field, $value); }



// ////////////////////////////////////////////////////////////////////////////
//
//
//
// ////////////////////////////////////////////////////////////////////////////
/**
 * 글 정보 읽기
 *
 *
 * @param int $idx 글 번호( post.idx 이다. )
 * @return associative-array mysql_fetch_assoc 의 결과값으로서 post.idx 의 레코드를 연관 배열로 리턴한다.
 *	empty 이면 값이 없는 것이다.
 */
function getPost($idx)
{
	global $db;
	
	$db->query("SELECT * FROM post WHERE idx=$idx");
	return $db->row();
}

/**
 * alias of getPost
 */
function post($idx) { return getPost($idx); }



/**
 * 글 삭제
 *
 * @param int $idx 글 번호 (post.idx)
 * @return faultCode
 */
function deletePost($idx)
{
	return faultDb($GLOBALS['db']->query("DELETE FROM post WHERE idx=$idx"));
}


/**
 * 여러개의 게시물을 리턴한다.
 *
 * @param int $idx_category 카테고리 번호이다.
 * @param int $limit 몇개의 글을 가져올지 제한한다.
 * @param associative-array 각 요소는 SQL 조건 구문에 사용될 필드 이름과 그 값을 가진다.
 * @return mixed(faultCode, array) 에러가 있을 경우 faultCode 를 리턴한다. 에러가 없을 경우 배열을 리턴한다. 에러 체크는 {@link faultCode} 로 한다.
 * <code>
 * $res = getPosts($ui['idx'], 100);
 * if ( faultCode($res) ) goBack("bbs::list->getPosts at " . __FILE__ . " line: " . __LINE__);
 * or
 * faultCode($res) and goBack("memo::index->getPosts at " . __FILE__ . " line: " . __LINE__);
 * </code>
 * 아래와 같이 getPosts 는 getPostCount 와 쌍을 이루어서 작업을 한다.
 * <code>
 * $memo_number = getPostCount(category_id, $user['idx']);
 * $memo_public = getPostCount(category_id, $user['idx'], array('r'=>GRADE_PUBLIC));
 * $res = getPosts(category_idx, $limit);
 * </code>
 * @changed 12/06/2006 LIMIT 을 하기 위해서 시작값과 갯수를 입력받는다.
 */
function getPosts($idx='', $limit=array(), $cond=array())
{
	global $db;
	
	if ( !is_numeric($idx) ) return faultEx(wrongParam, "wrong category index has submitted.");
	if ( !is_array($limit) ) return faultEx(wrongParam, "wrong number for extract has submitted");
	
	$cond['idx_category'] = $idx;
	$where = qWhere ( $cond );
	
	if ( ! empty($limit) ) {
		$qlimit = "LIMIT $limit[from], $limit[to]";
	}

	$q = "SELECT * FROM post $where $qlimit";
	//g($q);
	if ( ! $rc = $db->query($q,false) ) return faultDb($rc);
	
	
	$asAr = array();
	while ($row = $db->row())
	{
		$asAr[] = $row;
	}
	return $asAr;
}


/**
 * 글의 수를 리턴한다.
 *
 * 이 함수는 대부분의 글 수를 구하는 함수의 기본이 되는 함수이다.
 *
 * $id_category 가 empty 가 아니면, 이 분류의 총 글의 수를 리턴한다.
 * $idx_user 가 empty 가 아니면, $id_category 분류 내의 $idx_user 가 쓴 글의 총 수를 리턴한다.
 * 입력값이 모두 empty 이면 총 글의 수를 리턴한다.
 * 
 * @changed $id_category 변수는 기존에 아이디만 입력받았는데, 이제는 카테고리 번호도 입력을 받는다.
 * @param string $category 글의 분류 아이디 또는 번호이다. category.id 또는 category.idx 의 값이다.
 * @param int $idx_user 사용자 번호
 * @param associative-array $arCond 이 변수는 $id_category 와 $idx_user 이 외의 조건 비교가 필요할 경우 이 연관 배열에 항목=값의 쌍을 입력하면 SQL 검색 조건에 추가를 할 수 있다.
 * @return int 에러가 있을 경우 -1, 성공이면 0 이상의 값.
 * <code>
 * getPostCount( "bbs" );
 * getPostCount( "memo", 1 );
 * $memo_number = getPostCount(module_id, $user['idx'], array('r'=>0));
 * </code>
 *
 */
function getPostCount( $id_category='', $idx_user = 0, $arCond=array())
{
	$ar = array();
	
	if ( !empty( $id_category ) )
	{
		if ( is_numeric( $id_category ) )
			$ar['idx_category'] = $id_category;
		else
		{
			lib('category');
			$idx = getCategoryIndex($id_category);
			if ( empty($idx) ) return faultEx( wrongID, "post::getPostCount Wrong category id");
		$ar['idx_category'] = $idx;
		}
	}
	
	if ( !empty( $idx_user ) )
	{
		$ar['idx_user'] = $idx_user;
	}
	
		
	$where = qWhere( array_merge($ar, $arCond) );
	$q = "SELECT COUNT(*) FROM post $where";
	return $GLOBALS['db']->result($q);
}

/**
 * @see updatePostHit
 */
function updatePostCount($idx, $hit=1)
{
	return updatePostHit($idx, $hit);
}
/**
 * post.hits 의 값을 $hit 만큼 증가 시킨다.
 *
 * @todo mysql_affected_rows 를 통해서 제대로 수행되었는지 체크할 필요가 있다.
 */
function updatePostHit($idx, $hit=1)
{
	return updatePostField($idx, 'hits', "hits+$hit");
}







/**
 * 등급을 문자로 표현해서 리턴한다.
 *
 * @todo 이 함수는 default 라이브러리로 이동을 해야하지 않나?
 *
 * @todo 이 함수는 현제 문제가 있다. GRADE_PUBLIC_TEXT 에 해당하는 내용이 없다.
 * 해당 내용은 언어팩으로 설정을 한다.
 */
function textGrade($r)
{
	/*
	if ( $r == GRADE_PUBLIC ) return GRADE_PUBLIC_TEXT;
	else if ( $r <= GRADE_MEMBER ) return GRADE_MEMBER_TEXT;
	else if ( $r <= GRADE_ADMIN ) return GRADE_ADMIN_TEXT;
	else if ( $r <= GRADE_SUPER ) return GRADE_SUPER_TEXT;
	else if ( $r <= GRADE_ROOT ) return GRADE_ROOT_TEXT;
	else return GRADE_PRIVATE_TEXT;
	*/
	return "등급텍스트지원안됨";
}





/**
 * 입력된 정보가 가르키는 해당 카테고리의 총 레코드의 수를 리턴한다.
 *
 * 이 함수는 {@link getPostCount} 함수를 좀 더 간편하게 이름 지은 것이다.
 * @param string $id 숫자형 문자열이면 category.idx 를 기준으로 문자열이면 category.id 를 기준으로 post 테이블의 해당 정보의 수를 리턴한다.
 * @return int 정보의 갯수
 */
function number($id)
{
	/**
	if ( is_numeric($id) ) $cond = "idx=$id";
	else $cond = "id='$id'";
	return $GLOBALS['db']->result("SELECT COUNT(*) FROM post WHERE $cond");
	*/
	return getPostCount($id);
}










/**
 * 글 검색
 *
 * 이 함수는 모든 글에 대해서 검색을 한다. {@link getPosts}는 특정 영역의 글을 추출해서 리스트한다는 개념이지만, 이 함수는 가능한 모든 검색을 해서 그 결과를 리스트한다는 개념이다.
 * 이 함수는 Site Open API 의 동작을 보강해서 보다 많은 기능을 제공한다.
 * 검색할 카테고리가 empty 인 경우, 검색 구문 자체를 생성하지 않는다. 따라서 많은 경우에서 충분히 활용이 가능하다.
 * 모든 검색, 카운팅 등에서 이 함수를 이용하는 것은 올바른 선택이다.
 *
 * 이 함수는 {@link getPosts}와 비슷한 행동을 한다.
 * @param mixed $categories 검색할 카테고리의 아이디를 요소로 가지는 배열 변수이다. 만약 이 배열이 empty 이면 전체 카테고리를 검색한다. 숫자, 문자열이 가능하다. {@link qCategories}의 입력 형태를 참고한다.
 * @note 카테고리의 아이디가 숫자로만 구성이 되었으면 카테고리 번호로 인식을 한다.
 * @param associative-array $search 검색할 필드와 값을 가지는 연관 배열 변수이다.
 * search 연관 배열 변수에 검색을 할 필드가 'name' 이라면 $search[name] = '길동 명수' 와 같이 입력을 하고 $search[name cond] = 'OR' 와 같이 해서 검색 조건을 지정할 수 있다. 여기까지는 Site Open API 와 동일하다. cond 가 생략을 하면 LIKE 검색을 한다. name LIKE '%길동 명수%' 와 같이 검색 조건이 만들어진다. AND, OR 그리고 조건 생략, 외의 다른 조건을 추가 할 수 있다. 예를 들어 $search[name cond] = '<=' 와 같이 할 수 있다. 이같은 경우 name <= '길동 명수' 와 같이 지정된 조건이 그대로 사용된다.
 * @param associative-array $limit 검색의 결과를 제한할 값을 가지는 연관 배열 변수이다. 검색 제한 필드로는 날짜값과 LIMIT 구문이 사용된다.
 * 날짜 제한의 fromDate, toDate 는 기본 날짜 형식이어야한다. iso8601 형식이 아니다.
 * @param associative-array $orderby 검색 결과를 정렬할 값을 가지는 배열 변수이다. $orderby['dateTime'] => ASC 와 같은 값을 가질 수 있다.
 * @return mixed 에러가 발생했을 경우 faultCode 가 리턴된다. 에러가 없을 경우 데이터베이스 쿼리 결과가 배열로 저장되어 리턴된다. 각 배열의 요소는 하나의 post 테이블의 레코드를 나타내는 연관 배열이다. 만약 쿼리는 제대로 수행이 되었는데 결과가 없다면 empty 배열이 리턴된다.
 * @note $limit 값에는 fromDate, toDate 가 사용될 수 있으며, 날짜 형식은 시스템 날짜 형식이다.
 * <code>
 * <?
 * lib('post');
 * $categories[]					= 'talk';
 * $categories[]					= 'readme';
 * $search['r']						= GRADE_PUBLIC;
 * $search['r cond']			= '=';
 * $limit['fromNumber']		= 0;
 * $limit['toNumber']			= 10;
 * $orderby['dateTime']		= 'DESC';
 * $posts = searchPost($categories, $search, $limit, $orderby);
 * if ( faultCode($posts) ) javascriptAlert("검색에 문제가 있습니다.");
 * ?>
 * </code>
 * @param int &$count 총 레코드 수를 돌려받을 변수. 반드시 변수로 지정되어야한다.
 * 0 의 값이 입력되면 아무것도 안한다.
 * 1 의 값이 입력되면 이 변수에 LIMIT 제한 구문 없이 COUNT(*) 된 결과를 받는다. 즉 총 레코드 수도 구하고 검색 결과를 구하고자 할 때 이용될 수 있다.
 * 2 의 값이 입력되면, COUNT(*) 값만 구하고 그냥 리턴한다. 즉, 총 수만 구하고자 할 때에 사용된다.
 * 이 변수가 추가됨에 따라서 countPost 함수 단순히 이 함수의 재 지정 일 뿐이다.
 * @note PHP 버젼 4.x.x 대에서 &$count=null 과 같이 reference 하는 변수에 초기값을 지정할 수 없다.
 * 따라서 함수를 수정한다. $count 인자로 값을 돌려받을 수 없다. 즉, 2 의 경우만 사용이 가능하다.
 * @changed $count = 1 의 경우는 동작을 하지 않는다. 2 의 경우 레코드의 결과를 리턴한다. 직접 이 함수를 사용하지 말고 countPost 를 이용한다.
 * @since 2007/01/15 $search[exp cond] 에는 항목 사이들의 연관을 표현할 수 있다. AND, OR 등으로 사용이 가능하다. 생략시에는 AND 가 사용된다.
 * @see qFields
 * @since 2007/02/16 $search[exp extra] 를 통해서 직접 쿼리 조건 구문을 작성할 수 있다.
 * @see qFields
 * searchUser() 에서도 동일하다. 이것은 qFields() 를 사용하는 모든 함수는 동일하게 적용이된다.
 */
function searchPost($categories=array(), $search=array(), $limit=array(), $orderby=array(), $count=0)
{
	$where			= NULL;
	
	$idxs				= qCategories($categories);	// 카테고리들
	if ( faultCode($idxs) ) return faultEx($idxs, "post::searchPost couldn't make SQL query because of the wrong category.");
	$date				= qDateTime($limit);				// 변수 위치 주의
	$limit			= qLimit($limit);
	$order			= qOrder($orderby);
	$fields			= qFields($search);
	
	//g($fields);

	$cond = array();
	if ( !empty($idxs) )		$cond[] = $idxs;
	if ( !empty($date) )		$cond[] = $date;
	if ( !empty($fields) )	$cond[] = $fields;

	if ( !empty($cond) )
	{
		$where = "WHERE " . implode(" AND ", $cond);
	}

	global $db;
		
	if ( $count )
	{
		$q = "SELECT COUNT(*) FROM post $where";
		$rows = $db->result($q);
		if ( $count == 2 ) return $rows;
	}
	
	$q = "SELECT * FROM post $where $order $limit";
	//g($q);
	$db->query( $q );
	$asAr = array();
	while ($row = $db->row())
	{
		$asAr[] = $row;
	}
	return $asAr;
}

/**
 * 글의 수를 센다.
 *
 * 특정 글 영역(카테고리, 게시판)의 총 글 수를 세거나, 조건에 의한 글의 수를 얻기 위해서 사용될 수 있다.
 * {@link getPostCount}와 같은 역활을 하는 함수이다. 이 함수는 searchPost 함수와 같은 입력값을 가지며, 글의 내용이 아닌, 글의 수만 알고자 할 때에는 searchPost 함수 대신에 이 함수를 사용하면된다.
 * @see getPostCount
 * @param array $categories {@link searchPost} 참고
 * @param array $search {@link searchPost} 참고
 * @param array $limit 글을 수의를 세는데 limit 의 값으로는 날짜값이 들어갈 수 있다.
 * @return int
 * <code>
 *	$lastweek = mktime(0, 0, 0, date("m"), date("d")-7,   date("Y"));
 *	$fromDate = dateTime($lastweek);
 *	$lastweek_posts = countPost(array(), array(), array('fromDate'=>$fromDate));
 * </code>
 * @code <?=countPost($cat[idx])?>
 */
function countPost($categories=array(), $search=array(), $limit=array())
{
	return searchPost($categories, $search, $limit, array(), 2);
	/*
	$idxs				= qCategories($categories);
	$fields			= qFields($search);
	
	$cond = array();
	if ( !empty($idxs) )		$cond[] = $idxs;
	if ( !empty($fields) )	$cond[] = $fields;
	if ( !empty($cond) )
	{
		$where = "WHERE " . implode(" AND ", $cond);
	}
	
	$q = "SELECT COUNT(*) FROM post $where";
	return $GLOBALS['db']->result($q);
	*/
}



/**
 * 글 목록을 가져오는 함수.
 *
 * 이 함수의 역활은 {@link searchPost}나 {@link getPosts}와 동일하다. 다만 사용상 좀 더 쉽고 편하게 한 것이다. 이 함수는 내부적으로 {@link searchPost}를 사용한다. 입력 변수의 값과 출력 변수의 값을 좀 더 간단하게 하는데에 그 목적이 있다.
 * 내부적으로 searchPost 를 사용한다.
 * searchPost 는 데이터베이스 내용을 그대로 리턴하는데 반해 articles 는 내용을 가공해서 리턴한다.
 *
 * 부모글(답변글, 코멘트 등 제외)만 가져온다.
 * 읽기 권한이 있는 글만 가져온다. 따라서 이 함수를 사용할 때 카테고리를 지정 안해 주면, 개인 메모장, 블로그, 쇼핑몰 등의 정보를 모두 가져온다. 단, 일기 권한이 있는 공개 메모와 같은 정보만 가져온다.
 *
 * @note 카테고리에 구분없이 모든 글 정보를 가져온다.
 * @note 리턴되는 값의 url 항목에는 게시판 읽기 모드로 링크를 걸어준다. 다른 원하는 링크가 있으면 직접 만들어서 써야한다.
 * @note 리턴되는 값의 category 항목에는 카테고리 이름이 기록된다.
 * @note fname 항목에 성을 제외한 사용자 이름이 리턴된다. (한글 이름 기준)
 * @since 2006/12/14 글 목록을 가져오는 함수이다. 답변글(코멘트)는 가져 오지 않는다.
 * @since 2007/01/11 자신의 권한과 글의 접근 권한에 따라서 자료를 가져온다. 즉, 권한이 없는 글은 보여주지 않는다.
 * @important @note 리턴하는 $post['url'] 의 값은 상대 경로이다. 즉, 외부로 값을 전달할 때에는 절대 경로를 만들어야한다.
 * 예를 들면, urlHompage.'/'.$post['url'] 과 같이 사용할 수 있다.
 * @since 2007/01/21 리턴값 $post[url] 과 $post[url_read] 과 동일한 값을 가진다.
 *
 * @important @note $category 에 is_numeric() 으로 들어오면 카테고리 타입으로 인식을 한다.
 *	즉, idx_category_type 으로 비교를 한다. 따라서 필요한 경우 글 영역의 아이디를 입력해야한다.
 * @param mixed $category 글의 정보 영역.
 *	string 으로 값이 들어오면 하나의 글 정보 영역 (예 free),
 *	array 로 값이 들어오면 여러개의 글 정보 영역,
 *	is_numeric() 의 값으로 들어오면 카테고리 타입으로 인식해서 그 영역의 글을 검색한다.
 * @since 2007/01/26 입력 값이 searchPost 와 동일 해졌다. 물론 기존의 방식대로 호출이 가능하다.
 *	searchPost 방식대로 입력을 할 경우, 추가 정보를 얻는 효과가 있다.
 * @since mixed $rows 숫자로 입력이되면, 간단하게 해당 수 만큼 결과를 리턴한다. 배열로 입력이 되면, searchPost 와 동일하게 입력을 받는다.
 * 배열일 경우, $kvs 이다.
 */
function articles($category=array(), $rows=10, $limit=array(), $orderby=array())
{
	if ( is_array($rows) )
	{
		$kvs = $rows;
	}
	else
	{
		$limit['toNumber']		= $rows;
		$orderby['dateTime']	= 'DESC';
		$orderby['idx']				= 'DESC';
		$kvs['idx_parent']	= 0;
		$kvs['idx_parent cond'] = '=';
	}

	// 자신의 등급에 따라 글을 검색
	if ( login() ) $kvs['r'] = $GLOBALS['user']['grade'];
	else $kvs['r'] = '0';
	$kvs['r cond'] = "<=";
	
	
	//
	if ( empty($category) ) $ar = array();
	else if ( is_numeric($category) ) { $ar = array(); $kvs['idx_category_type'] = $category; $kvs['idx_category_type cond'] = '='; }
	else if ( is_string($category) ) $ar[] = $category;
	else $ar = $category;

	
	
	$as = searchPost($ar, $kvs, $limit, $orderby);
	if ( faultCode($as) ) return $as;	// 에러의 경우 faultCode 를 그대로 리턴


	$posts = array();
	foreach( $as as $post )
	{
		$posts[] = postinfo($post);
	}
	return $posts;
}
/**
 * 글 레크드에 정보를 추가한다.
 *
 * @see articles
 */
function postinfo(&$post)
{
	/**
	 * @since 2007/02/19 짧은 주소로 표기를 한다.
	 */
	//$post['url']						= "?cate=bbs&mode=read&idx=$post[idx]";
	$post['url']						= "?$post[idx]";
	$post['url_read']				= $post['url'];
	lib('category');
	$post['category']				= getCategoryName($post['idx_category']);
	$post['dt']							= dt($post['dateTime']);
	$post['fname']					= firstName($post['name']);
	if ( empty($post['title']) ) $post['title'] = $GLOBALS['lang']['emptyTitle'];
	else $post['title'] = $post['title'];

	return $post;
}

/**
 * 처음 글쓴 순서대로글을 리스트하는 것을 간편하게 사용하는 함수
 *
 * @note 복잡한 구문은 그냥 articles 나 searchPost 함수를 사용한다.
 */
function articles_orderby_firstwrite($category=array(), $rows=10)
{
	$limit['toNumber']		= $rows;
	$orderby['dateTime_firstwrite']	= 'DESC';
	$orderby['idx']				= 'DESC';
	$kvs['idx_parent']	= 0;
	$kvs['idx_parent cond'] = '=';
		
	return articles($category, $kvs, $limit, $orderby);
}



/**
 * 등록된 글이 자신의 것인지 체크를 한다.
 *
 * 입력값이 글 정보를 가지는 연관 배열 변수일 경우, DB 액세스를 하지 않는다.
 * @param mixed $mix numeric 타입이면 글 번호, array 타입이면 글 정보가 입력된다.
 *	이 값이 0 이면, 전역 변수의 $ui['idx'] 변수를 사용한다.
 * @param global associative-array $user 사용자 정보
 * @return boolean 자기 글이면 true, 아니면 false
 * @note post.idx_user 와 user.idx 를 가지고 비교를 한다,
 * 따라서 로그인 하지 않았거나, post.idx_user 의 값이 없고, 비밀번호가 지정된 글의 경우는 무조건 자신의 글이 아니다.
 * 자신의 글이 아니면, 비밀번호를 입력하게 하는 예제:
 * <code>
 * <? if ( ! myPost() ) { ?>
 * <div class=name>비밀번호</div><div><input type=password name=password></div>
 * <?}?>
 * </code>
 */
function myPost($mix=0)
{
	if ( ! login() ) return false;

	global $user;
	if ( $mix == 0 ) $mix = $GLOBALS['ui']['idx'];
	
	if ( is_numeric($mix) )
	{
		$post = getPost($mix);
		return $user['idx'] == $post['idx_user'];
	}
	else if ( is_array($mix) )
	{
		return $user['idx'] == $mix['idx_user'];
	}
	else
	{
		return false;
	}	
}

/**
 * return records of children. //답변글(들)을 리턴한다.
 *
 * @param int $idx 글 번호. idx_parent=$idx 인 정보를 추출한다.
 * @return array 2 차원 배열. searchPost 와 동일하다.
 */
function getReplies($idx)
{
	return searchPost( array(), array('idx_parent'=>$idx, 'idx_parent cond'=>'='), array(), array('dateTime'=>'ASC') );
}
/**
 * @see getReplies
 * @since 2007/01/25 $idx 가 NULL 이면, 모든 답변글을 $number 만큼 뽑는다. 이때, getReplies 와는 달리, 전처리를 한다.
 *	예를 들면, url 에 링크를 걸어주는 등의 처리를 한다.
 * <code>
 *	replies(NULL, 5);
 * </code>
 */
function replies($idx=NULL, $number=10) {
	if ( $idx )
		return getReplies($idx);
	else
	{
		$ar = searchPost(array(), array('idx_parent'=>'0', 'idx_parent cond'=>'<>'), array('toNumber'=>$number), array('dateTime'=>'DESC'));
		$posts = array();
		foreach($ar as $post)
		{
			$post['url'] = "?cate=bbs&mode=read&idx=$post[idx_parent]";
			$posts[] = $post;
		}
		return $posts;
	}
}

/**
 * 글의 정보를 수정한다.
 *
 * 입력되는 idx의 글 번호는 부모글이라 단정을 하고, 해당 글에 대해서 코멘트 수, 파일 수 등을 수정한다.
 * 입력 값으로는 레코드의 항목 이름과 숫자값이다.
 * 즉, 글의 레코드 항목에 숫자를 더하거나 뺀다.
 *
 * @param int $idx_parent 글 번호. 수정될 글의 레코드 번호이다. 즉, 부모글의 번호가 입력되어야한다.
 * @param string $field 레코드 항목
 * @param int $cnt 변경될 값(더하거나 뺄 값)
 * @return faultCode
 */
function updateParent($idx_parent, $field, $cnt)
{
	return updatePostField($idx_parent, $field, "$field + $cnt");
}



// ////////////////////////////////////////////////////////////////////////////
//
// 메세지 ( 쪽지, 내부 폼 메일 등 ) 관련 루틴
//
// ////////////////////////////////////////////////////////////////////////////
/**
 * 새 메세지 개수를 리턴
 *
 * @since 2007/03/10 user.idx 입력값이 없으면 현제 사용자의 idx (글로벌 변수)를 사용한다.
 * @param int $idx user.idx
 * @return int
 *
 */
function newMessage($idx=NULL)
{
	if ( $idx === NULL ) $idx = $GLOBALS['user']['idx'];
	return countPost(array(), array('idx_user'=>$idx, 'idx_user cond'=>'=', 'idx_category_type'=>CATE_MESSAGE, 'idx_category_type cond'=>'=', 'hits'=>0, 'hits cond'=>'='));
}
/**
 * 총 메세지 개수를 리턴
 */
function countMessage($idx=NULL)
{
	if ( $idx === NULL ) $idx = $GLOBALS['user']['idx'];
	return countPost(array(), array('idx_user'=>$idx, 'idx_user cond'=>'=', 'idx_category_type'=>CATE_MESSAGE, 'idx_category_type cond'=>'='));
}


/**
 * 사용자의 총 메세지 개수, 새 메세지 개수를 조정한다.
 *
 * user.message_number, user.message_new 의 값을 조정한다.
 *
 * @param int $idx user.idx
 * @return faultcode
 */
function checkMessage($idx)
{
	$update['idx'] = $idx;
	$update['message_new'] = newMessage($update['idx']);
	$update['message_number'] = countMessage($update['idx']);
	return updateUser($update);
}

/**
 * 메세지(쪽지)를 전송한다.
 *
 * @note 보내는 사람이 회원 로그인을 하지 않아도 메세지를 전송할 수 있다.
 * @param associative-array $data
 *	$data['idx_user']			받는 사람 회원 번호
 *	$data['idx_parent']		보내는 사람 회원 번호
 *	$data['description']	내용
 * @return faultCode just same as insertPost
 */
function sendMessage($data)
{
	$data['idx_category']				= 0;
	$data['idx_category_type']	= CATE_MESSAGE;
	$data['r']									= GRADE_PRIVATE;
	if ( $fc = insertPost($data) ) return $fc;
	
	return checkMessage($data['idx_user']);
}

function deleteMessage($idx) { return deletePost($idx); }



/**
 * 사용자 정보에 대한 글을 쓴다.
 *
 * 예를 들면, 신규 가입한 사용자에 대해서 가입 축하메세지를 남기는 경우 등에 이용될 수 있다.
 *
 *
 * 입력 값은 idx_user 받는 사람, idx_parent 보내는 사람, description 내용이다.
 * 자세한 정보는 빌드가이드를 참고한다.
 */
function writeUserComment($data)
{
	$data['idx_category']				= 0;
	$data['idx_category_type']	= CATE_USER_INFO;
	return insertPost($data);
}



/**
 *
 * @param int $idx user.idx_user 의 값으로 이 사용자에 대한 코멘트를 전체 추출한다.
 */
function userComment($idx)
{
	
	$kvs['idx_category_type']				= CATE_USER_INFO;
	$kvs['idx_category_type cond']	= '=';
	$kvs['idx_user']								= $idx;
	$kvs['idx_user cond']						= '=';
	
	return articles(array(), $kvs, array(), array('dateTime'=>'DESC'));
}
?>
