<?php
/**
 * 플러그인 처리를 한다.
 *
 * @author ky <thruthesky@yahoo.co.kr>
 * @license http://jangnans.com JNAGNAN
 * @see BUILDGUIDE
 * @package library
 *
 * @note $GLOBALS['plugin_area'] 에는 각 위치마다 사용될 플러그인이 기록되어 있다.
 * @note $GLOBALS['areaname'] 에는 위치 이름이 들어있다.
 * @note $GLOBALS['plugindir'] 에는 플러그인 디렉토리 이름이 들어간다.
 * @note $GLOBALS['url_config'] 에는 각 플러그인의 설정 화면으로 갈 수 있는 URL 경로
 * @since 2007/03/17 HTTP 변수 area_name 이 areaname 으로 변경. 관련된 모든 소스의 변경.
 * @note 플러그인과 관련된 문서는 V2 파일을 참고한다.
 */


/**
 * 스킨을 처리(출력,실행)할 때, 플러그인을 사용해도 되는 상황이면 true 를 리턴한다.
 *
 *
 * @param string $name 플러그인 이름
 * @return bool 
 */
function checkPlugin($name)
{
	return true;
}

/**
 * 플러그인을 처리하고 그 결과(정보)를 표현
 *
 * 스킨으로 부터 이 함수가 호출된다.
 * @see BUILDGUIDE#plugin
 *
 * @return string 만약 플러그인 데이터를 제대로 출력을 할 상황이 아니면, NULL 을 리턴한다.
 * @since 2007/03/23 플러그인의 기본 값을 받아서 사용할 수 있도록 한다.
 * @param string $name 위치 (이름)
 * @param associative-array $cfg_default 기본 설정 값
 */
function plugin($name, $cfg_default = NULL)
{
	/** 플러그인을 인클루드 해도 되는가? */
	if ( ! checkPlugin($name) ) return NULL;
	
	/** PHP 특성에 따른 변수명 처리 */
	$name = str_replace(".", "_", $name);
	
	
	/** 플러그인 배치표. 한번만 실행되게 한다. */
	if ( ! isset($GLOBALS['plugin_area']) )
	{
		$GLOBALS['plugin_area'] = @config(plugin_areaname_file());
	}
	
	/** 위치의 설정 파일을 읽는다. */
	$cfg = config(cfgfile($name));

	/** 현제 위치(영역이름)에 해당하는 플러그인을 찾는다. */
	
	// 위치 설정에 현제 위치 이름이 있는가?
	if ( isset($GLOBALS['plugin_area'][$name]) )
	{
	}
	// 없다면, 현제 위치의 초기값이 지정되어 있으면 그 설정을 사용하도록 한다.
	else if ( $cfg_default )
	{
		$cfg_default['dir'] = $cfg_default['plugin'];
		$cfg_default['areaname'] = $name;
		$re = config(cfgfile($name), $cfg_default); /** @todo error check with $re */
		
		$GLOBALS['plugin_area'][$name] = $cfg_default['dir'];
		$re = config(plugin_areaname_file(), $GLOBALS['plugin_area']); /** @todo error check with $re */
	}
	
	$dir = NULL;
	if ( isset($GLOBALS['plugin_area'][$name]) )
		$dir = $GLOBALS['plugin_area'][$name];

		
	/** 전역 변수로 기록 */
	$GLOBALS['plugindir'] = $dir;					// 전역 변수로 기록
	$GLOBALS['areaname'] = $name;
	$GLOBALS['url_config'] = "?cate=plugin&mode=plugin&dir=$dir&areaname=$name&action=config";
	
	
	
	
	/** 현제 위치에 플러그인이 지정되어있는가? */
	if ( $dir )
	{
		include dirRoot . "/plugin/$dir/$dir.php";
	}
	/** 현제 위치에 플러그인이 지정되어 있지 않다면, 즉 설정이 되어 있지 않다면, */
	else
	{
		return plugin_set();
	}

}



/**
 * 캐쉬 루틴 시작
 *
 * 플러그인에 대한 캐쉬를 한다.
 * 스킨에 대해서 캐쉬를 하는 lib/cache.php 의 skin_cache_start 와 비슷하다.
 * 전반적으로 lib/cache.php 와 동일하게 동작한다.
 *
 * @param associative-array $cfg 위치에 대한 설정 값
 *	$cfg['cache'] 에 분단위로 캐쉬 유효 기간이 기록되어있어야한다.
 *
 * @return bool 캐쉬 데이터를 출력한 경우, true, 아니면 false 를 리턴한다.
 *
 * plugin_cache_start 에서 ob_start 를 호출하고, plugin_cache_end 에서 버퍼링을 종료하므로
 * 이 두개 함수는 쌍으로 사용되어야한다.
 *
 * @note 주의: 캐쉬 설정을 0 으로 한다면, 이 함수 자체가 실행되지 않을 것이다. 즉 캐쉬와는 전혀 상관없이
 * 실시간으로 정보가 보여질 것이다.
 * 그리고 다시 캐쉬를 지정한다면, 기존의 캐쉬 파일이 존재하면 그것을 보여줄 것이다.
 * 즉, 설정을 수정해도 변경되지 않는 수가 있으니 조심해야한다.
 * 이 문제는 캐쉬를 사용하지 않을 때, 캐쉬 파일을 삭제함으로 해결이 가능하다.
 */
function plugin_cache_start($cfg)
{
	global $pc_file, $pc_filepath, $pc_file_last, $areaname;


	
	if ( !isset($cfg['cache']) || empty($cfg['cache']) ) return false; // 캐쉬를 하지 않는다면, 그냥 리턴
	
	$minute = $cfg['cache'];


	$pc_file_last = $minute;
	$back = debug_backtrace();
	$pc_file = $areaname;
	$pc_filepath = tmp_repository . "/cache/plugin.$pc_file.php";
	
	$data = @readfileEx($pc_filepath);


	if ( $data )
	{
		$time = substr($data, 4, 10);
		$expire = $pc_file_last * 60 + $time;
		
		
		/** * 유효 시간 경과하지 않은 경우, 캐쉬 내용을 출력하고 true 리턴 */
		if ( $expire > time() )
		{
			echo $data;
			return true;
		}
	}
	
	/**
	 * 캐쉬 파일이 없거나 유효시간이 경과한 경우,
	 */
	ob_start();
	return false;
}

/**
 *
 * @param associative-array $cfg same as plugin_cache_start
 * @return bool 성공적으로 캐쉬 데이터를 저장했으면 true, 아니면 false
 */
function plugin_cache_end($cfg)
{
	if ( !isset($cfg['cache']) || empty($cfg['cache']) ) return false; // 캐쉬를 하지 않는다면, 그냥 리턴
	
	$content = ob_get_flush();
	
	
	$b = "<!--".time()." $GLOBALS[pc_file] cached at ".date("y/m/d h:i:s")." and will be expire on ".date("y/m/d h:i:s", time()+$GLOBALS['pc_file_last']*60)."-->\r\n";
	$e = "\r\n<!--end of cache $GLOBALS[pc_file]-->\r\n";
	$data = $b . $content . $e;
	return writeStringEx($GLOBALS['pc_filepath'], $data);
}




/**
 * 플러그인의 설정에서 캐쉬할 유효기간의 선택을 쉽게하기 위해서 HTML option 태그들을 리턴한다.
 *
 * @param string $value 선택될 값
 * @return string
 * @code <?=@plugin_cache_options($cfg['cache'])?> 와 같이 undefined variable 에러를 피한다.
 */
function plugin_cache_options($value=NULL)
{

	$v = "v$value";
	$$v = " selected";

return <<<EOH
				<option value="0"		$v0>캐쉬 안함</option>
				<option value="1"		$v1>1 분</option>
				<option value="3"		$v3>3 분</option>
				<option value="5"		$v5>5 분</option>
				<option value="10"	$v10>10 분</option>
				<option value="20"	$v20>20 분</option>
				<option value="30"	$v30>30 분</option>
				<option value="60"	$v60>60 (1시간)</option>
				<option value="120"	$v120>120 (2시간)</option>
				<option value="600"	$v600>600 (10시간)</option>
				<option value="1440"$v1440>1440 (하루)</option>
				<option value="2880"$v2880>2880 (이틀)</option>
				<option value="5760"$v5760>5760 (4 일)</option>
EOH;
}


/**
 * 각 스킨에서 위치별 플러그인 이름을 가지고 있는 파일의 경로를 리턴한다.
 *
 * plugin_areaname_file 은 스킨에서 위치-플러그인 두개의 쌍을 구성하는 정보를 가진다.
 */
function plugin_areaname_file()
{
	return module_repository."/plugin.{$GLOBALS['system']['skin']}.php";
}


/**
 * 각 (플러그인) 위치 설정 파일 경로를 리턴한다.
 *
 * 위치 이름을 입력 받아서 그 위치에 설정된 (플러그인의 정보가 저장된) 파일 이름을 리턴한다.
 * 반드시 이 함수를 통해서 각 위치의 설정 파일 경로명을 얻어야한다.
 *
 * @global string $areaname 위치 이름
 * @return string
 * @since 2007/03/22 $areaname 을 입력 받도록 해서 옵션 처리
 * @param string $areaname NULL 이면 $GLOBALS['areanme'] 을 사용
 */
function cfgfile($areaname=NULL)
{
	if ( $areaname == NULL && isset($GLOBALS['areaname'])) $areaname = $GLOBALS['areaname'];
	if ( $areaname == NULL && isset($GLOBALS['ui']['areaname'])) $areaname = $GLOBALS['ui']['areaname'];
	if ( $areaname == NULL ) alert("plugin.php::cfgfile something wrong with areaname");

	return plugin_areaconfig_file($areaname);
}
// alias of cfgfile()
function plugin_areaconfig_file($areaname)
{
	return module_repository . "/plugin.cfg.{$GLOBALS['system']['skin']}.$areaname.php";
}

/**
 * 여러 플러그인의 설정을 가지는(저장하고 있는) 파일의 경로를 리턴.
 *
 * 관리자 모드에서 플러그인 설정을 저장해놓고 다시 활용할 수 있도록 한다.
 */
function plugin_config_file()
{
	return module_repository . "/plugin.config.list.php";
}



/**
 * 숨김 상태이면, 관리 표시를 한다.
 */
function plugin_hide()
{
	plugin_display_admin_mark($GLOBALS['url_config'], $GLOBALS['areaname']);
}

/**
 * 현제 위치에 플러그인이 지정되지 않았을 때, 이 함수가 실행된다.
 *
 * 관리 표시를 나타내 주며, 관리 표시를 클릭했을 때, 플러그인을 선택할 수 있는 창을 먼저 띄운다.
 *
 * 즉, 위치에 설정이 안되어 있을 경우, 설정을 할 수 있도록 해 준다.
 */
function plugin_set()
{
	global $areaname;
	$url_config = "?cate=plugin&mode=set_plugin&areaname=$areaname";
	plugin_display_admin_mark($url_config, $areaname);
}

/**
 * 관리 표시
 *
 * @since 2007/03/23 관리자 일 경우만 표현을 한다.
 * @return string 올바른 표혀능ㄹ 하지 못할 경우, NULL 리턴.
 */
function plugin_display_admin_mark($url, $areaname=NULL)
{
	if ( !admin('plugin') ) return NULL;
	if ( isset($GLOBALS['plugin_print_admin_mark']) && $GLOBALS['plugin_print_admin_mark'] )
	{
		echo "
			<script src=\"./etc/res/javascript/plugin_library.js\"></script>
			<script>plugin_display_config_link(\"$url\",'$areaname')</script>
		";
	}
}
?>
