/**
 * ZipEntryData : Zip 엔트리 정보 클래스
 */
package net.kldp.jzip;

import java.io.File;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.util.Date;

import org.apache.tools.zip.ZipEntry;

/**
 * @author jeongseungwon
 * 
 */
public class ZipEntryData {
	/**
	 * Zip 엔트리의 크기를 문자열로 반환함
	 * 
	 * @param size
	 *            Zip 엔트리의 크기
	 * @return 크기를 표현하는 문자열
	 */
	public static String getSizeString(long size) {
		final int KB = 1024;
		final int MB = 1024 * KB;
		final int GB = 1024 * MB;

		double newSize = size;
		String sizeString = "";
		NumberFormat nf = NumberFormat.getInstance();
		nf.setMaximumFractionDigits(2);

		if (size >= GB) {
			newSize /= GB;
			sizeString = nf.format(newSize) + " GB";
		} else if (size >= MB) {
			newSize /= MB;
			sizeString = nf.format(newSize) + " MB";
		} else if (size >= KB) {
			newSize /= KB;
			sizeString = nf.format(newSize) + " KB";
		} else {
			sizeString = size + " 바이트";
		}

		return sizeString;
	}

	/**
	 * 시간을 문자열로 표현하는 메소드
	 * 
	 * @param time
	 *            문자열로 표현할 시간
	 * @return 시간을 표현하는 문자열
	 */
	public static String getTimeString(long time) {
		String timeString = null;

		if (time != -1) {
			Date date = new Date(time);
			DateFormat df = DateFormat.getDateTimeInstance(DateFormat.LONG,
					DateFormat.LONG);
			timeString = df.format(date);
		} else {
			timeString = "알 수 없음";
		}

		return timeString;
	}

	private ZipEntry zipEntry;

	/**
	 * ZipEntryData의 생성자
	 * 
	 * @param entry
	 *            Zip 엔트리
	 */
	public ZipEntryData(ZipEntry entry) {
		setZipEntry(entry);
	}

	/**
	 * 경로가 포함된 Zip 엔트리의 이름을 반환하는 메소드
	 * 
	 * @return 경로가 포함된 Zip 엔트리의 이름
	 */
	public String getEntryName() {
		return zipEntry.getName();
	}

	/**
	 * Zip 엔트리의 파일명을 반환하는 메소드
	 * 
	 * @return Zip 엔트리의 파일명
	 */
	public String getName() {
		File file = new File(zipEntry.getName());

		return file.getName();
	}

	/**
	 * Zip 엔트리의 경로를 반환하는 메소드
	 * 
	 * @return Zip 엔트리의 경로
	 */
	public String getPath() {
		File file = new File(zipEntry.getName());
		String parent = file.getParent();

		if (parent != null) {
			return "/" + parent;
		} else {
			return "/";
		}
	}

	/**
	 * Zip 엔트리의 크기를 반환하는 메소드
	 * 
	 * @return Zip 엔트리의 크기
	 */
	public long getSize() {
		return zipEntry.getSize();
	}

	/**
	 * 테이블에 표시하기 위해, Zip 엔트리의 정보를 텍스트로 반환하는 메소드
	 * 
	 * @return Zip 엔트리의 정보
	 */
	public String[] getStrings() {
		// 이름
		String name = getName();

		// 크기
		String size = getSizeString(getSize());

		// 형식
		String type = getType();

		// 바뀐 시간
		String time = getTimeString(getTime());

		// 위치
		String path = getPath();

		return new String[] { name, size, type, time, path };
	}

	/**
	 * Zip 엔트리의 바뀐 시간을 반환하는 메소드
	 * 
	 * @return Zip 엔트리의 바뀐 시간
	 */
	public long getTime() {
		return zipEntry.getTime();
	}

	/**
	 * Zip 엔트리의 형식을 문자열로 반환하는 메소드
	 * 
	 * @return 디렉토리이면 "디렉토리", 파일이면 "파일"
	 */
	public String getType() {
		String type = null;

		if (zipEntry.isDirectory()) {
			type = "디렉토리";
		} else {
			type = "파일";
		}

		return type;
	}

	/**
	 * Zip 엔트리를 반환하는 메소드
	 * 
	 * @return Zip 엔트리
	 */
	public ZipEntry getZipEntry() {
		return zipEntry;
	}

	/**
	 * Zip 엔트리를 설정하는 메소드
	 * 
	 * @param zipEntry
	 *            Zip 엔트리
	 */
	public void setZipEntry(ZipEntry zipEntry) {
		this.zipEntry = zipEntry;
	}
}
