/**
 * JZip 0.6
 * 
 * 이 프로그램은 Zip 파일 포맷을 지원하는 압축 프로그램입니다.
 * 이 프로그램은 SWT(Standard Widget Toolkit)와 Apache Ant를 사용합니다.
 * 
 * 홈페이지 : http://jzip.kldp.net 
 * 저작권 : GNU General Public License
 * 개발자 : 정승원(jeongseungwon@hanmail.net)
 */
package net.kldp.jzip;

import java.io.File;
import java.io.IOException;
import java.text.DateFormat;
import java.text.NumberFormat;

import org.apache.tools.zip.ZipFile;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSource;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.DropTargetListener;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;

/**
 * JZip 프로그램의 메인 클래스
 * 
 * @author jeongseungwon
 * 
 */
public class JZip {

	/**
	 * {@link DropTargetListener}를 구현하는 클래스
	 * 
	 * @author jeongsw
	 * 
	 */
	private final class DropListener implements DropTargetListener {
		/**
		 * 열 수 있는 파일인지 확인하는 메소드
		 * 
		 * @param fileName
		 *            파일명
		 * @return 열 수 있는 파일인지 여부
		 */
		private boolean canOpen(String fileName) {
			if (fileName == null) {
				return false;
			}

			File file = new File(fileName);

			if (!file.exists()) {
				return false;
			} else if (file.isDirectory()) {
				return false;
			} else if (!file.canRead()) {
				return false;
			} else {
				return true;
			}
		}

		public void dragEnter(DropTargetEvent event) {

		}

		public void dragLeave(DropTargetEvent event) {

		}

		public void dragOperationChanged(DropTargetEvent event) {

		}

		public void dragOver(DropTargetEvent event) {

		}

		public void drop(DropTargetEvent event) {
			if (event.data == null) {
				event.detail = DND.DROP_NONE;

				return;
			}

			String[] fileNames = (String[]) event.data;

			if (zip == null) {
				// 현재 열려있는 압축 파일이 없는 경우

				if (fileNames.length == 1 && fileNames[0] != null) {
					// 파일이 하나인 경우

					// 압축 파일 열기
					open(fileNames[0]);
				} else {
					// 파일이 여러 개인 경우

					event.detail = DND.DROP_NONE;
				}
			} else {
				// 열려있는 압축 파일이 있는 경우

				if (fileNames.length != 1) {
					// 파일이 여러 개인 경우

					if (zip.canWrite()) {
						// Zip 파일에 대한 쓰기 권한이 있는 경우

						// 파일 더하기
						zip.addFile(sShell, fileNames, tmpDir);

						updateTable();
					} else {
						// Zip 파일에 대한 쓰기 권한이 없는 경우

						MessageBox messageBox = new MessageBox(sShell, SWT.OK
								| SWT.ICON_ERROR);
						messageBox.setText("파일 또는 디렉토리 더하기 실패!");
						messageBox.setMessage(zip.getFilePath()
								+ " 파일에 대한 쓰기 권한이 없습니다.");
						messageBox.open();

						event.detail = DND.DROP_NONE;
					}
				} else {
					// 파일이 하나인 경우

					if (canOpen(fileNames[0])) {
						// 열 수 있는 파일인 경우

						// 동작 선택 대화상자
						ActionSelectDialog selectDialog = new ActionSelectDialog();
						final int select = selectDialog.open(sShell,
								fileNames[0]);

						switch (select) {
						case ActionSelectDialog.CANCEL:
							event.detail = DND.DROP_NONE;

							break;

						case ActionSelectDialog.ADD:
							if (zip.canWrite()) {
								zip.addFile(sShell, fileNames, tmpDir);

								updateTable();
							} else {
								MessageBox messageBox = new MessageBox(sShell,
										SWT.OK | SWT.ICON_ERROR);
								messageBox.setText("파일 또는 디렉토리 더하기 실패!");
								messageBox.setMessage(zip.getFilePath()
										+ " 파일에 대한 쓰기 권한이 없습니다.");
								messageBox.open();

								event.detail = DND.DROP_NONE;
							}

							break;

						case ActionSelectDialog.OPEN:
							open(fileNames[0]);

							break;
						}
					} else {
						// 열 수 없는 파일인 경우

						if (zip.canWrite()) {
							zip.addFile(sShell, fileNames, tmpDir);

							updateTable();
						} else {
							MessageBox messageBox = new MessageBox(sShell,
									SWT.OK | SWT.ICON_ERROR);
							messageBox.setText("파일 또는 디렉토리 더하기 실패!");
							messageBox.setMessage(zip.getFilePath()
									+ " 파일에 대한 쓰기 권한이 없습니다.");
							messageBox.open();

							event.detail = DND.DROP_NONE;
						}
					}
				}
			}
		}

		public void dropAccept(DropTargetEvent event) {

		}
	}

	/**
	 * 디렉토리를 삭제하는 메소드 (모든 하위 디렉토리와 파일까지 삭제함)
	 * 
	 * @param dirFile
	 *            삭제할 디렉토리 {@link File}
	 */
	public static void deleteDir(File dirFile) {
		File[] files = dirFile.listFiles();

		for (File file : files) {
			if (file.isDirectory()) {
				// 디렉토리인 경우
				deleteDir(file);
			} else {
				// 파일인 경우
				file.delete();
			}
		}

		dirFile.delete();
	}

	/**
	 * JZip 프로그램의 main 메소드
	 * 
	 * @param args
	 *            Zip 파일 이름
	 */
	public static void main(String[] args) {
		/*
		 * Before this is run, be sure to set up the launch configuration
		 * (Arguments->VM Arguments) for the correct SWT library path in order
		 * to run with the SWT dlls. The dlls are located in the SWT plugin jar.
		 * For example, on Windows the Eclipse SWT 3.1 plugin jar is:
		 * installation_directory\plugins\org.eclipse.swt.win32_3.1.0.jar
		 */
		Display display = Display.getDefault();
		JZip thisClass = new JZip();
		thisClass.createSShell();
		// DnD 기능 설정
		thisClass.setDnd();
		// 문맥 메뉴 설정
		thisClass.setContextMenu();

		thisClass.sShell.open();

		// 압축 파일 열기
		if (args.length == 1 && args[0] != null) {
			thisClass.open(args[0]);
		}

		while (!thisClass.sShell.isDisposed()) {
			if (!display.readAndDispatch())
				display.sleep();
		}
		display.dispose();
	}

	private Shell sShell = null; // @jve:decl-index=0:visual-constraint="10,10"

	private String jzip;

	private Menu menuBar = null;

	private Menu submenuFile = null;

	private Menu submenuEdit = null;

	private Menu submenuView = null;

	private Menu submenuHelp = null;

	private Menu submenuAlignment = null;

	private Menu submenuFormat = null;

	private String defaultPath;

	private Zip zip = null; // @jve:decl-index=0:

	private Table table = null;

	private MenuItem radioFile;

	private MenuItem radioDir;

	private MenuItem pushRefresh;

	private MenuItem pushClose;

	private MenuItem radioLong;

	private MenuItem radioShort;

	private MenuItem pushProperty;

	private MenuItem radioUtf8;

	private MenuItem pushSelectAll;

	private MenuItem pushDeselectAll;

	private MenuItem pushAddFile;

	private MenuItem pushAddDir;

	private MenuItem pushRename;

	private MenuItem pushDel;

	private MenuItem radioMs949;

	private MenuItem pushSave;

	private MenuItem pushExtract;

	private TableColumn tableColumnPath;

	private MenuItem radioMid;

	private MenuItem radioName;

	private MenuItem radioSize;

	private MenuItem radioType;

	private MenuItem radioTime;

	private MenuItem radioPath;

	private MenuItem checkReverse;

	private TableColumn tableColumnName;

	private TableColumn tableColumnSize;

	private TableColumn tableColumnType;

	private TableColumn tableColumnTime;

	private ToolBar toolBar = null;

	private Composite composite = null;

	private Button buttonUp = null;

	private Button buttonTop = null;

	private Label compositeSeparator = null;

	private Label labelPath = null;

	private Text text = null;

	private Label statusSeparator = null;

	private Label statusLine = null;

	private ToolItem toolItemExtract;

	private File tmpDir;

	private ToolItem toolItemAddFile;

	private ToolItem toolItemAddDir;

	private MenuItem contextPushView;

	private MenuItem contextPushOpen;

	private MenuItem contextPushExtract;

	private MenuItem contextPushRename;

	private MenuItem contextPushDel;

	private MenuItem contextPushOpenDir;

	/**
	 * {@link JZip} 클래스의 생성자
	 */
	public JZip() {
		jzip = "JZip 0.6";

		defaultPath = System.getProperty("user.home");

		tmpDir = new File(System.getProperty("java.io.tmpdir"), "JZip");
	}

	/**
	 * 압축 파일에 디렉토리를 더하는 메소드
	 */
	private void addDir() {
		if (!zip.canWrite()) {
			// Zip 파일에 대한 쓰기 권한이 없는 경우

			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("디렉토리 더하기 실패");
			messageBox.setMessage(zip.getFilePath() + " 파일에 대한 쓰기 권한이 없습니다.");
			messageBox.open();

			return;
		}

		// 디렉토리 선택 대화상자
		DirectoryDialog directoryDialog = new DirectoryDialog(sShell, SWT.OPEN);
		directoryDialog.setText("압축 파일에 더할 디렉토리를 선택하세요.");
		directoryDialog.setMessage("압축 파일에 더할 디렉토리를 선택하세요.\n"
				+ "모든 하위 디렉토리와 디렉토리 안의 파일들도 다 더합니다.");
		directoryDialog.setFilterPath(defaultPath);
		final String directoryName = directoryDialog.open();

		if (directoryName != null) {
			addDir(directoryName);
		}
	}

	/**
	 * 압축 파일에 디렉토리를 더하는 메소드
	 * 
	 * @param directoryName
	 *            더할 디렉토리명
	 */
	private void addDir(String directoryName) {
		// 더할 디렉토리 File
		File directory = new File(directoryName);

		if (!directory.exists()) {
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("디렉토리 더하기 실패!");
			messageBox.setMessage(directory.getPath() + " 디렉토리가 존재하지 않습니다.");
			messageBox.open();

			return;
		}

		if (!directory.isDirectory()) {
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("디렉토리 더하기 실패!");
			messageBox
					.setMessage(directory.getPath() + " 파일은 디렉토리가 아니라 파일입니다.");
			messageBox.open();

			return;
		}

		if (!directory.canRead()) {
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("디렉토리 더하기 실패!");
			messageBox.setMessage(directory.getPath()
					+ " 디렉토리에 대한 읽기 권한이 없습니다.");
			messageBox.open();

			return;
		}

		if (!statusLine.isDisposed()) {
			statusLine.setText("압축 파일에 디렉토리를 더하는 중입니다.");
			statusLine.update();
		}

		zip.addDir(sShell, directory, tmpDir);

		updateTable();
	}

	/**
	 * 압축 파일에 파일을 더하는 메소드
	 */
	private void addFile() {
		if (!zip.canWrite()) {
			// 쓰기 권한이 없는 경우
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("파일 더하기 실패");
			messageBox.setMessage(zip.getFilePath() + " 파일에 대한 쓰기 권한이 없습니다.");
			messageBox.open();

			return;
		}

		// 파일 선택 대화상자
		FileDialog dialog = new FileDialog(sShell, SWT.OPEN | SWT.MULTI);
		dialog.setText("압축 파일에 더할 파일을 선택하세요.");
		dialog.setFilterPath(defaultPath);

		if (dialog.open() != null) {
			final String parentName = dialog.getFilterPath();
			String[] fileNames = dialog.getFileNames();

			String[] filePaths = new String[fileNames.length];

			for (int i = 0; i < filePaths.length; i++) {
				File file = new File(parentName, fileNames[i]);
				filePaths[i] = file.getAbsolutePath();
			}

			if (!statusLine.isDisposed()) {
				statusLine.setText("압축 파일에 파일을 더하는 중입니다.");
				statusLine.update();
			}

			zip.addFile(sShell, filePaths, tmpDir);

			updateTable();
		}
	}

	/**
	 * 상태 표시줄 구분자를 추가하는 메소드
	 */
	private void addStatusLine() {
		GridData gridData11 = new GridData();
		gridData11.horizontalAlignment = GridData.FILL;
		gridData11.verticalAlignment = GridData.CENTER;

		statusLine = new Label(sShell, SWT.NONE);
		statusLine.setText("");
		statusLine.setLayoutData(gridData11);

		updateStatusLine();
	}

	/**
	 * 상태 표시줄을 추가하는 메소드
	 */
	private void addStatusSeparator() {
		GridData gridData2 = new GridData();
		gridData2.horizontalAlignment = GridData.FILL;
		gridData2.verticalAlignment = GridData.CENTER;

		statusSeparator = new Label(sShell, SWT.SEPARATOR | SWT.HORIZONTAL);
		statusSeparator.setLayoutData(gridData2);
	}

	/**
	 * 테이블에 위치 컬럼을 추가하는 메소드
	 */
	private void addTablecolumnPath() {
		tableColumnPath = new TableColumn(table, SWT.NONE);
		tableColumnPath.setWidth(200);
		tableColumnPath.setText("위치");
		tableColumnPath
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSortColumn() == tableColumnPath
								&& table.getSortDirection() == SWT.DOWN) {
							radioName.setSelection(false);
							radioSize.setSelection(false);
							radioType.setSelection(false);
							radioTime.setSelection(false);
							radioPath.setSelection(true);

							checkReverse.setSelection(true);

							updateTable();

							table.setSortColumn(tableColumnPath);
							table.setSortDirection(SWT.UP);
						} else {
							radioName.setSelection(false);
							radioSize.setSelection(false);
							radioType.setSelection(false);
							radioTime.setSelection(false);
							radioPath.setSelection(true);

							checkReverse.setSelection(false);

							updateTable();

							table.setSortColumn(tableColumnPath);
							table.setSortDirection(SWT.DOWN);
						}
					}
				});
		tableColumnPath.setToolTipText("위치순으로 정렬합니다.");
	}

	/**
	 * 현재 열려있는 압축 파일을 닫는 메소드
	 */
	private void close() {
		zip = null;

		updateTable();

		sShell.setText(jzip);
	}

	/**
	 * This method initializes composite
	 * 
	 */
	private void createComposite() {
		GridData gridData5 = new GridData();
		gridData5.horizontalAlignment = GridData.FILL;
		gridData5.verticalAlignment = GridData.CENTER;
		GridData gridData3 = new GridData();
		gridData3.widthHint = 200;
		GridData gridData1 = new GridData();
		gridData1.heightHint = 0;
		gridData1.verticalAlignment = GridData.FILL;
		gridData1.horizontalAlignment = GridData.BEGINNING;
		GridLayout gridLayout1 = new GridLayout();
		gridLayout1.numColumns = 5;
		composite = new Composite(sShell, SWT.NONE);
		composite.setLayout(gridLayout1);
		composite.setLayoutData(gridData5);
		buttonUp = new Button(composite, SWT.NONE);
		buttonUp.setText("위로");
		buttonUp.setToolTipText("상위 디렉토리로 이동합니다.");
		buttonUp.setEnabled(false);
		buttonUp
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						zip.goToParent();

						updateTable();
					}
				});
		buttonTop = new Button(composite, SWT.NONE);
		buttonTop.setText("맨 위로");
		buttonTop.setToolTipText("최상위 디렉토리로 이동합니다.");
		buttonTop.setEnabled(false);
		buttonTop
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						zip.goToTop();

						updateTable();
					}
				});
		compositeSeparator = new Label(composite, SWT.SEPARATOR);
		compositeSeparator.setEnabled(false);
		compositeSeparator.setLayoutData(gridData1);
		labelPath = new Label(composite, SWT.NONE);
		labelPath.setText("위치 : ");
		labelPath.setEnabled(false);
		text = new Text(composite, SWT.BORDER);
		text.setToolTipText("압축 파일 내의 현재 위치");
		text.setEnabled(false);
		text.setLayoutData(gridData3);
		text
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						zip.setPath(text.getText().trim());

						updateTable();
					}
				});
		text.addFocusListener(new org.eclipse.swt.events.FocusAdapter() {
			public void focusGained(org.eclipse.swt.events.FocusEvent e) {
				pushDel.setAccelerator(0);
			}

			public void focusLost(org.eclipse.swt.events.FocusEvent e) {
				pushDel.setAccelerator(SWT.DEL);
			}
		});
	}

	/**
	 * 새로운 압축 파일을 생성하는 메소드
	 */
	private void createNew() {
		// 새로운 압축 파일 선택 대화상자
		FileDialog fileDialog = new FileDialog(sShell, SWT.SAVE);
		fileDialog.setText("새로 만들 압축 파일을 선택하세요.");
		fileDialog.setFilterPath(defaultPath);
		fileDialog.setFilterExtensions(new String[] { "*.zip", "*.*" });
		fileDialog.setFilterNames(new String[] { "Zip 파일 (*.zip)",
				"모든 파일 (*.*)" });
		String fileName = fileDialog.open();

		if (fileName != null) {
			if (createNew(fileName)) {
				// 인코딩 설정
				radioMs949.setSelection(true);
				radioUtf8.setSelection(false);

				open(fileName);
			}
		}
	}

	/**
	 * 새로운 압축 파일을 생성하는 메소드
	 * 
	 * @param fileName
	 *            압축 파일명
	 * @return 압축 성공 여부
	 */
	private boolean createNew(String fileName) {
		final File file = getSaveFile(fileName);

		if (file == null) {
			// 압축 파일 저장 실패
			return false;
		}

		Zip.createNew(file);

		// 압축 파일 저장 성공
		return true;
	}

	/**
	 * This method initializes sShell
	 */
	private void createSShell() {
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		GridLayout gridLayout = new GridLayout();
		gridLayout.horizontalSpacing = 0;
		gridLayout.verticalSpacing = 0;
		gridLayout.marginWidth = 0;
		gridLayout.marginHeight = 0;
		gridLayout.numColumns = 1;
		sShell = new Shell();
		sShell.setText(jzip);
		sShell.setLayout(gridLayout);
		createToolBar();
		createComposite();
		sShell.setSize(new Point(713, 429));
		table = new Table(sShell, SWT.MULTI | SWT.VIRTUAL);
		table.setHeaderVisible(true);
		table.setVisible(false);
		table.setLayoutData(gridData);
		table.setLinesVisible(true);
		addStatusSeparator();
		addStatusLine();
		tableColumnName = new TableColumn(table, SWT.NONE);
		table
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					@Override
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						final TableItem item = (TableItem) e.item;
						final int index = table.indexOf(item);

						if (zip.isDirecotry(index)) {
							// 디렉토리인 경우
							if (radioDir.getSelection()) {
								// 디렉토리로 보기인 경우
								zip.openDir(index);

								updateTable();
							}
						} else {
							// 파일인 경우
							zip.openFile(index, tmpDir);
						}
					}

					@Override
					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						updateMenu();

						updateStatusLine();
					}
				});
		table.addListener(SWT.SetData, new Listener() {
			public void handleEvent(Event event) {
				TableItem item = (TableItem) event.item;
				int index = table.indexOf(item);

				int dateFormat = DateFormat.MEDIUM;
				if (radioShort.getSelection()) {
					dateFormat = DateFormat.SHORT;
				} else if (radioLong.getSelection()) {
					dateFormat = DateFormat.LONG;
				}

				item.setText(zip.getStrings(index, dateFormat));
			}
		});
		tableColumnName.setWidth(260);
		tableColumnName.setText("이름");
		tableColumnName
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSortColumn() == tableColumnName
								&& table.getSortDirection() == SWT.DOWN) {
							radioName.setSelection(true);
							radioSize.setSelection(false);
							radioType.setSelection(false);
							radioTime.setSelection(false);
							radioPath.setSelection(false);

							checkReverse.setSelection(true);

							updateTable();

							table.setSortColumn(tableColumnName);
							table.setSortDirection(SWT.UP);
						} else {
							radioName.setSelection(true);
							radioSize.setSelection(false);
							radioType.setSelection(false);
							radioTime.setSelection(false);
							radioPath.setSelection(false);

							checkReverse.setSelection(false);

							updateTable();

							table.setSortColumn(tableColumnName);
							table.setSortDirection(SWT.DOWN);
						}
					}
				});
		tableColumnName.setToolTipText("이름순으로 정렬합니다.");
		tableColumnSize = new TableColumn(table, SWT.RIGHT);
		tableColumnSize.setWidth(80);
		tableColumnSize.setText("크기");
		tableColumnSize
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSortColumn() == tableColumnSize
								&& table.getSortDirection() == SWT.DOWN) {
							radioName.setSelection(false);
							radioSize.setSelection(true);
							radioType.setSelection(false);
							radioTime.setSelection(false);
							radioPath.setSelection(false);

							checkReverse.setSelection(true);

							updateTable();

							table.setSortColumn(tableColumnSize);
							table.setSortDirection(SWT.UP);
						} else {
							radioName.setSelection(false);
							radioSize.setSelection(true);
							radioType.setSelection(false);
							radioTime.setSelection(false);
							radioPath.setSelection(false);

							checkReverse.setSelection(false);

							updateTable();

							table.setSortColumn(tableColumnSize);
							table.setSortDirection(SWT.DOWN);
						}
					}
				});
		tableColumnSize.setToolTipText("크기순으로 정렬합니다.");
		tableColumnType = new TableColumn(table, SWT.RIGHT);
		tableColumnType.setWidth(80);
		tableColumnType.setText("형식");
		tableColumnType
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSortColumn() == tableColumnType
								&& table.getSortDirection() == SWT.DOWN) {
							radioName.setSelection(false);
							radioSize.setSelection(false);
							radioType.setSelection(true);
							radioTime.setSelection(false);
							radioPath.setSelection(false);

							checkReverse.setSelection(true);

							updateTable();

							table.setSortColumn(tableColumnType);
							table.setSortDirection(SWT.UP);
						} else {
							radioName.setSelection(false);
							radioSize.setSelection(false);
							radioType.setSelection(true);
							radioTime.setSelection(false);
							radioPath.setSelection(false);

							checkReverse.setSelection(false);

							updateTable();

							table.setSortColumn(tableColumnType);
							table.setSortDirection(SWT.DOWN);
						}
					}
				});
		tableColumnType.setToolTipText("형식순으로 정렬합니다.");
		tableColumnTime = new TableColumn(table, SWT.NONE);
		tableColumnTime.setWidth(200);
		tableColumnTime.setText("바뀐 시간");
		tableColumnTime
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSortColumn() == tableColumnTime
								&& table.getSortDirection() == SWT.DOWN) {
							radioName.setSelection(false);
							radioSize.setSelection(false);
							radioType.setSelection(false);
							radioTime.setSelection(true);
							radioPath.setSelection(false);

							checkReverse.setSelection(true);

							updateTable();

							table.setSortColumn(tableColumnTime);
							table.setSortDirection(SWT.UP);
						} else {
							radioName.setSelection(false);
							radioSize.setSelection(false);
							radioType.setSelection(false);
							radioTime.setSelection(true);
							radioPath.setSelection(false);

							checkReverse.setSelection(false);

							updateTable();

							table.setSortColumn(tableColumnTime);
							table.setSortDirection(SWT.DOWN);
						}
					}
				});
		tableColumnTime.setToolTipText("바뀐 시간순으로 정렬합니다.");
		addTablecolumnPath();
		menuBar = new Menu(sShell, SWT.BAR);
		MenuItem submenuItemFile = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemFile.setText("압축 파일(&A)");
		MenuItem submenuItemEdit = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemEdit.setText("편집(&E)");
		MenuItem submenuItemView = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemView.setText("보기(&V)");
		MenuItem submenuItemHelp = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemHelp.setText("도움말(&H)");
		submenuHelp = new Menu(submenuItemHelp);
		submenuHelp.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				updateStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		MenuItem pushHomePage = new MenuItem(submenuHelp, SWT.PUSH);
		pushHomePage.setText("홈페이지 방문(&V)");
		MenuItem pushKldp = new MenuItem(submenuHelp, SWT.PUSH);
		pushKldp.setText("&KLDP 프로젝트 페이지 방문");
		pushKldp.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("KLDP.net 프로젝트 페이지를 방문합니다.");
			}
		});
		pushKldp
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// KLDP.net 프로젝트 페이지 방문
						Program.launch("http://kldp.net/projects/jzip");
					}

					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}
				});
		pushHomePage
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// JZip 홈페이지 방문
						Program.launch("http://jzip.kldp.net/");
					}
				});
		pushHomePage.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("프로그램 홈페이지를 방문합니다.");
				}
			}
		});
		@SuppressWarnings("unused")
		MenuItem separator8 = new MenuItem(submenuHelp, SWT.SEPARATOR);
		MenuItem pushAbout = new MenuItem(submenuHelp, SWT.PUSH);
		pushAbout.setText("JZip 정보(&A)...");
		pushAbout
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// JZip 정보
						MessageBox messageBox = new MessageBox(sShell, SWT.OK
								| SWT.ICON_INFORMATION);
						messageBox.setText("JZip 정보");

						StringBuilder message = new StringBuilder(50);
						message.append(jzip);
						message
								.append("\n\n이 프로그램은 Zip 파일 포맷을 지원하는 압축 프로그램입니다.");
						message.append("\n\n저작권 : GNU General Public License");
						message.append("\n홈페이지 : http://jzip.kldp.net");
						message
								.append("\n개발자 : 정승원(jeongseungwon@hanmail.net)");

						messageBox.setMessage(message.toString());
						messageBox.open();
					}
				});
		pushAbout.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("프로그램 정보를 보여줍니다.");
				}
			}
		});
		submenuItemHelp.setMenu(submenuHelp);
		submenuView = new Menu(submenuItemView);
		submenuView.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				updateStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		pushRefresh = new MenuItem(submenuView, SWT.PUSH);
		pushRefresh.setText("새로 고침(&R)\tF5");
		pushRefresh.setEnabled(false);
		pushRefresh.setAccelerator(SWT.F5);
		pushRefresh
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						open(zip.getFilePath());
					}
				});
		pushRefresh.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("현재 열려있는 압축 파일을 다시 불러옵니다.");
				}
			}
		});
		@SuppressWarnings("unused")
		MenuItem separator3 = new MenuItem(submenuView, SWT.SEPARATOR);
		radioMs949 = new MenuItem(submenuView, SWT.RADIO);
		radioMs949.setText("&MS949 (윈도우)");
		radioMs949.setSelection(true);
		radioMs949
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && radioMs949.getSelection()) {
							open(zip.getFilePath());
						}
					}
				});
		radioMs949.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					if (!statusLine.isDisposed()) {
						statusLine.setText("인코딩을 MS949(윈도우)로 변경합니다.");
					}
				}
			}
		});
		radioUtf8 = new MenuItem(submenuView, SWT.RADIO);
		radioUtf8.setText("&UTF8 (리눅스)");
		radioUtf8
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && radioUtf8.getSelection()) {
							open(zip.getFilePath());
						}
					}
				});
		radioUtf8.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("인코딩을 UTF8(리눅스)로 변경합니다.");
				}
			}
		});
		@SuppressWarnings("unused")
		MenuItem separator4 = new MenuItem(submenuView, SWT.SEPARATOR);
		MenuItem checkToolBar = new MenuItem(submenuView, SWT.CHECK);
		checkToolBar.setText("도구 모음(&T)");
		checkToolBar.setSelection(true);
		checkToolBar
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						MenuItem checkToolBar = (MenuItem) e.widget;

						if (checkToolBar.getSelection()) {
							// 도구 모음 보이기
							if (toolBar.isDisposed()) {
								createToolBar();

								updateMenu();

								toolBar.moveAbove(null);
								sShell.layout();
							}
						} else {
							// 도구 모음 숨기기
							if (!toolBar.isDisposed()) {
								toolBar.dispose();

								sShell.layout();
							}
						}
					}
				});
		checkToolBar.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("도구 모음을 보이거나 감춥니다.");
				}
			}
		});
		MenuItem checkStatusLine = new MenuItem(submenuView, SWT.CHECK);
		checkStatusLine.setText("상태 표시줄(&S)");
		checkStatusLine.setSelection(true);
		checkStatusLine
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						MenuItem checkStatusLine = (MenuItem) e.widget;

						if (checkStatusLine.getSelection()) {
							// 상태 표시줄 보이기
							if (statusSeparator.isDisposed()) {
								addStatusSeparator();
							}

							if (statusLine.isDisposed()) {
								addStatusLine();
							}

							sShell.layout();
						} else {
							// 상태 표시줄 숨기기
							if (!statusSeparator.isDisposed()) {
								statusSeparator.dispose();
							}

							if (!statusLine.isDisposed()) {
								statusLine.dispose();
							}

							sShell.layout();
						}
					}
				});
		checkStatusLine
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						if (!statusLine.isDisposed()) {
							statusLine.setText("상태 표시줄을 보이거나 감춥니다.");
						}
					}
				});
		@SuppressWarnings("unused")
		MenuItem separator5 = new MenuItem(submenuView, SWT.SEPARATOR);
		MenuItem submenuItemAlignment = new MenuItem(submenuView, SWT.CASCADE);
		submenuItemAlignment.setText("항목 정렬(&A)");
		MenuItem submenuItemFormat = new MenuItem(submenuView, SWT.CASCADE);
		submenuItemFormat.setText("바뀐 시간 출력 형식(&F)");
		@SuppressWarnings("unused")
		MenuItem separator6 = new MenuItem(submenuView, SWT.SEPARATOR);
		radioFile = new MenuItem(submenuView, SWT.RADIO);
		radioFile.setText("모든 파일 보기(&A)\tCtrl+1");
		radioFile.setAccelerator(SWT.CTRL | '1');
		radioFile
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (radioFile.getSelection()) {
							if (!composite.isDisposed()) {
								composite.dispose();

								sShell.layout();
							}

							if (tableColumnPath.isDisposed()) {
								addTablecolumnPath();
							}

							if (zip != null) {
								open(zip.getFilePath());
							}
						}
					}
				});
		radioFile.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("압축 파일 내의 모든 항목을 표시합니다.");
				}
			}
		});
		radioDir = new MenuItem(submenuView, SWT.RADIO);
		radioDir.setText("디렉토리로 보기(&D)\tCtrl+2");
		radioDir.setSelection(true);
		radioDir.setAccelerator(SWT.CTRL | '2');
		radioDir
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (radioDir.getSelection()) {
							if (composite.isDisposed()) {
								createComposite();

								composite.moveAbove(table);
								sShell.layout();
							}

							if (!tableColumnPath.isDisposed()) {
								tableColumnPath.dispose();
							}

							if (zip != null) {
								open(zip.getFilePath());
							}
						}
					}
				});
		radioDir.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("압축 파일의 내용을 디렉토리 형태로 보여줍니다.");
				}
			}
		});
		submenuFormat = new Menu(submenuItemFormat);
		radioShort = new MenuItem(submenuFormat, SWT.RADIO);
		radioShort.setText("간략하게(&S)");
		radioShort
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && radioShort.getSelection()) {
							updateTable();
						}
					}
				});
		radioShort.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("바뀐 시간을 간략하게 출력합니다.");
				}
			}
		});
		radioMid = new MenuItem(submenuFormat, SWT.RADIO);
		radioMid.setText("보통(&M)");
		radioMid.setSelection(true);
		radioMid
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && radioMid.getSelection()) {
							updateTable();
						}
					}
				});
		radioMid.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("바뀐 시간의 출력 형식을 기본 값으로 변경합니다.");
				}
			}
		});
		radioLong = new MenuItem(submenuFormat, SWT.RADIO);
		radioLong.setText("자세하게(&L)");
		radioLong
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && radioLong.getSelection()) {
							updateTable();
						}
					}
				});
		radioLong.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("바뀐 시간을 자세하게 출력합니다.");
				}
			}
		});
		submenuItemFormat.setMenu(submenuFormat);
		submenuItemFormat
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						if (!statusLine.isDisposed()) {
							statusLine.setText("바뀐 시간 출력 형식을 변경합니다.");
						}
					}
				});
		submenuAlignment = new Menu(submenuItemAlignment);
		radioName = new MenuItem(submenuAlignment, SWT.RADIO);
		radioName.setText("이름으로 정렬(&N)");
		radioName.setSelection(true);
		radioName
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && radioName.getSelection()) {
							updateTable();

							table.setSortColumn(tableColumnName);

							if (checkReverse.getSelection()) {
								table.setSortDirection(SWT.UP);
							} else {
								table.setSortDirection(SWT.DOWN);
							}
						}
					}
				});
		radioName.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("이름으로 정렬합니다.");
				}
			}
		});
		radioSize = new MenuItem(submenuAlignment, SWT.RADIO);
		radioSize.setText("크기로 정렬(&S)");
		radioSize
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && radioSize.getSelection()) {
							updateTable();

							table.setSortColumn(tableColumnSize);

							if (checkReverse.getSelection()) {
								table.setSortDirection(SWT.UP);
							} else {
								table.setSortDirection(SWT.DOWN);
							}
						}
					}
				});
		radioSize.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("크기로 정렬합니다.");
				}
			}
		});
		radioType = new MenuItem(submenuAlignment, SWT.RADIO);
		radioType.setText("형식으로 정렬(&T)");
		radioType
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && radioType.getSelection()) {
							updateTable();

							table.setSortColumn(tableColumnType);

							if (checkReverse.getSelection()) {
								table.setSortDirection(SWT.UP);
							} else {
								table.setSortDirection(SWT.DOWN);
							}
						}
					}
				});
		radioType.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("형식으로 정렬합니다.");
				}
			}
		});
		radioTime = new MenuItem(submenuAlignment, SWT.RADIO);
		radioTime.setText("바뀐 시간으로 정렬(&D)");
		radioTime
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && radioTime.getSelection()) {
							updateTable();

							table.setSortColumn(tableColumnTime);

							if (checkReverse.getSelection()) {
								table.setSortDirection(SWT.UP);
							} else {
								table.setSortDirection(SWT.DOWN);
							}
						}
					}
				});
		radioTime.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("바뀐 시간으로 정렬합니다.");
				}
			}
		});
		radioPath = new MenuItem(submenuAlignment, SWT.RADIO);
		radioPath.setText("위치로 정렬(&P)");
		radioPath
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && radioPath.getSelection()) {
							if (!tableColumnPath.isDisposed()) {
								updateTable();

								table.setSortColumn(tableColumnPath);

								if (checkReverse.getSelection()) {
									table.setSortDirection(SWT.UP);
								} else {
									table.setSortDirection(SWT.DOWN);
								}
							}
						}
					}
				});
		radioPath.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("위치로 정렬합니다.");
				}
			}
		});
		@SuppressWarnings("unused")
		MenuItem separator7 = new MenuItem(submenuAlignment, SWT.SEPARATOR);
		checkReverse = new MenuItem(submenuAlignment, SWT.CHECK);
		checkReverse.setText("역순으로 정렬(&R)");
		checkReverse
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null) {
							updateTable();

							final TableColumn sortColumn = table
									.getSortColumn();
							if (sortColumn == null) {
								table.setSortColumn(tableColumnName);
							} else if (sortColumn == tableColumnName) {
								table.setSortColumn(tableColumnName);
							} else if (sortColumn == tableColumnSize) {
								table.setSortColumn(tableColumnSize);
							} else if (sortColumn == tableColumnType) {
								table.setSortColumn(tableColumnType);
							} else if (sortColumn == tableColumnTime) {
								table.setSortColumn(tableColumnTime);
							} else if (sortColumn == tableColumnPath) {
								table.setSortColumn(tableColumnPath);
							} else {
								table.setSortColumn(tableColumnName);
							}

							if (checkReverse.getSelection()) {
								table.setSortDirection(SWT.UP);
							} else {
								table.setSortDirection(SWT.DOWN);
							}
						}
					}
				});
		checkReverse.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("거꾸로 정렬합니다.");
				}
			}
		});
		submenuItemAlignment.setMenu(submenuAlignment);
		submenuItemAlignment
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						if (!statusLine.isDisposed()) {
							statusLine.setText("항목을 정렬합니다.");
						}
					}
				});
		submenuItemView.setMenu(submenuView);
		submenuEdit = new Menu(submenuItemEdit);
		submenuEdit.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				updateStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		pushAddFile = new MenuItem(submenuEdit, SWT.PUSH);
		pushAddFile.setText("파일 더하기(&F)...");
		pushAddFile.setEnabled(false);
		pushAddFile
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						addFile();
					}
				});
		pushAddFile.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("압축 파일에 새로운 파일을 더합니다.");
				}
			}
		});
		pushAddDir = new MenuItem(submenuEdit, SWT.PUSH);
		pushAddDir.setText("디렉토리 더하기(&D)...");
		pushAddDir.setEnabled(false);
		pushAddDir
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						addDir();
					}
				});
		pushAddDir.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("압축 파일에 새로운 디렉토리를 더합니다.");
				}
			}
		});
		@SuppressWarnings("unused")
		MenuItem separator1 = new MenuItem(submenuEdit, SWT.SEPARATOR);
		pushRename = new MenuItem(submenuEdit, SWT.PUSH);
		pushRename.setText("이름 바꾸기(&R)...\tF2");
		pushRename.setAccelerator(SWT.F2);
		pushRename.setEnabled(false);
		pushRename
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						rename();
					}
				});
		pushRename.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("선택된 항목의 이름을 변경합니다.");
				}
			}
		});
		pushDel = new MenuItem(submenuEdit, SWT.PUSH);
		pushDel.setText("지우기(&D)...\tDel");
		pushDel.setAccelerator(SWT.DEL);
		pushDel.setEnabled(false);
		pushDel
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						delete();
					}
				});
		pushDel.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("선택된 항목을 삭제합니다.");
				}
			}
		});
		@SuppressWarnings("unused")
		MenuItem separator2 = new MenuItem(submenuEdit, SWT.SEPARATOR);
		pushSelectAll = new MenuItem(submenuEdit, SWT.PUSH);
		pushSelectAll.setText("모두 선택(&A)\tCtrl+A");
		pushSelectAll.setEnabled(false);
		pushSelectAll.setAccelerator(SWT.CTRL | 'A');
		pushSelectAll
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						table.selectAll();

						updateMenu();

						updateStatusLine();
					}
				});
		pushSelectAll.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("모든 항목을 선택합니다.");
				}
			}
		});
		pushDeselectAll = new MenuItem(submenuEdit, SWT.PUSH);
		pushDeselectAll.setText("모두 선택 해제(&U)");
		pushDeselectAll.setEnabled(false);
		pushDeselectAll
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						table.deselectAll();

						updateMenu();

						updateStatusLine();
					}
				});
		pushDeselectAll
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						if (!statusLine.isDisposed()) {
							statusLine.setText("모든 항목을 선택 해제합니다.");
						}
					}
				});
		submenuItemEdit.setMenu(submenuEdit);
		submenuFile = new Menu(submenuItemFile);
		submenuFile.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				updateStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		MenuItem pushNew = new MenuItem(submenuFile, SWT.PUSH);
		pushNew.setText("새로 만들기(&N)...\tCtrl+N");
		pushNew.setAccelerator(SWT.CTRL | 'N');
		pushNew
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						createNew();
					}
				});
		pushNew.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("새로운 압축 파일을 생성합니다.");
				}
			}
		});
		MenuItem pushOpen = new MenuItem(submenuFile, SWT.PUSH);
		pushOpen.setText("열기(&O)...\tCtrl+O");
		pushOpen.setAccelerator(SWT.CTRL | 'O');
		pushOpen
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						open();
					}
				});
		pushOpen.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("기존 압축 파일을 엽니다.");
				}
			}
		});
		pushSave = new MenuItem(submenuFile, SWT.PUSH);
		pushSave.setText("다른 이름으로 저장(&S)...\tCtrl+S");
		pushSave.setAccelerator(SWT.CTRL | 'S');
		pushSave.setEnabled(false);
		pushSave
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						saveAs();
					}
				});
		pushSave.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("선택된 항목을 다른 이름으로 저장합니다.");
				}
			}
		});
		pushExtract = new MenuItem(submenuFile, SWT.PUSH);
		pushExtract.setText("압축 풀기(&E)...\tCtrl+E");
		pushExtract.setAccelerator(SWT.CTRL | 'E');
		pushExtract.setEnabled(false);
		pushExtract
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						extract();
					}
				});
		pushExtract.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("선택된 항목을 압축 해제합니다.");
				}
			}
		});
		pushProperty = new MenuItem(submenuFile, SWT.PUSH);
		pushProperty.setText("속성(&P)...");
		pushProperty.setEnabled(false);
		pushProperty
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						int dateFormat = DateFormat.MEDIUM;
						if (radioLong.getSelection()) {
							dateFormat = DateFormat.SHORT;
						} else if (radioShort.getSelection()) {
							dateFormat = DateFormat.LONG;
						}

						NumberFormat nf = NumberFormat.getInstance();
						nf.setMaximumFractionDigits(2);

						long length = zip.getLength();
						long originalLength = zip.getOriginalLength();
						double ratio = (double) originalLength / length;

						StringBuilder message = new StringBuilder(70);
						message.append("파일 이름 : " + zip.getFileName());
						message.append("\n파일 경로 : " + zip.getFileParentPath());
						message.append("\n바뀐 시간 : "
								+ Zip.getTimeString(zip.lastModified(),
										dateFormat));
						message
								.append("\n압축 크기 : "
										+ Zip.getSizeString(length));
						message.append("\n실제 크기 : "
								+ Zip.getSizeString(originalLength));
						message.append("\n압축 정도 : " + nf.format(ratio));
						message.append("\n항목 개수 : " + zip.getOriginalSize());

						MessageBox property = new MessageBox(sShell, SWT.OK
								| SWT.ICON_INFORMATION);
						property.setText(zip.getFileName() + " 속성");
						property.setMessage(message.toString());
						property.open();
					}
				});
		pushProperty.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("압축 파일의 속성을 보여줍니다.");
				}
			}
		});
		pushClose = new MenuItem(submenuFile, SWT.PUSH);
		pushClose.setText("닫기(&C)");
		pushClose.setEnabled(false);
		pushClose
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 압축 파일 닫기
						close();
					}
				});
		pushClose.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("압축 파일을 닫습니다.");
				}
			}
		});
		@SuppressWarnings("unused")
		MenuItem separator = new MenuItem(submenuFile, SWT.SEPARATOR);
		MenuItem pushQuit = new MenuItem(submenuFile, SWT.PUSH);
		pushQuit.setText("프로그램 종료(&Q)\tCtrl+Q");
		pushQuit.setAccelerator(SWT.CTRL | 'Q');
		pushQuit
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 임시 파일 삭제
						if (tmpDir.exists()) {
							deleteDir(tmpDir);
						}

						// 프로그램 종료
						System.exit(0);
					}
				});
		pushQuit.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("프로그램을 종료합니다.");
				}
			}
		});
		submenuItemFile.setMenu(submenuFile);
		sShell.setMenuBar(menuBar);
		sShell.addShellListener(new org.eclipse.swt.events.ShellAdapter() {
			public void shellClosed(org.eclipse.swt.events.ShellEvent e) {
				if (tmpDir.exists()) {
					deleteDir(tmpDir);
				}
			}
		});
		sShell.addKeyListener(new org.eclipse.swt.events.KeyAdapter() {
			public void keyReleased(org.eclipse.swt.events.KeyEvent e) {
				
			}
		});
	}

	/**
	 * This method initializes toolBar
	 * 
	 */
	private void createToolBar() {
		GridData gridData4 = new GridData();
		gridData4.horizontalAlignment = GridData.FILL;
		gridData4.verticalAlignment = GridData.CENTER;
		toolBar = new ToolBar(sShell, SWT.NONE);
		toolBar.setLayoutData(gridData4);
		ToolItem toolItemNew = new ToolItem(toolBar, SWT.PUSH);
		toolItemNew.setText("새로 만들기");
		toolItemNew.setToolTipText("새로운 압축 파일을 생성합니다.");
		toolItemNew
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						createNew();
					}
				});
		ToolItem toolItemOpen = new ToolItem(toolBar, SWT.PUSH);
		toolItemOpen.setText("열기");
		toolItemOpen.setToolTipText("기존 압축 파일을 엽니다.");
		toolItemOpen
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						open();
					}
				});
		toolItemExtract = new ToolItem(toolBar, SWT.PUSH);
		toolItemExtract.setText("풀기");
		toolItemExtract.setToolTipText("압축을 해제합니다.");
		toolItemExtract.setEnabled(false);
		@SuppressWarnings("unused")
		ToolItem toolItemSeparator = new ToolItem(toolBar, SWT.SEPARATOR);
		toolItemAddFile = new ToolItem(toolBar, SWT.PUSH);
		toolItemAddFile.setText("파일 더하기");
		toolItemAddFile.setEnabled(false);
		toolItemAddFile.setToolTipText("압축 파일에 파일을 더합니다.");
		toolItemAddFile
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						addFile();
					}
				});
		toolItemAddDir = new ToolItem(toolBar, SWT.PUSH);
		toolItemAddDir.setText("디렉토리 더하기");
		toolItemAddDir.setEnabled(false);
		toolItemAddDir.setToolTipText("압축 파일에 디렉토리를 더합니다.");
		toolItemAddDir
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						addDir();
					}
				});
		toolItemExtract
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						extract();
					}
				});
	}

	/**
	 * 항목을 삭제하는 메소드
	 */
	private void delete() {
		if (!zip.canWrite()) {
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("지우기 실패!");
			messageBox.setMessage(zip.getFilePath() + " 파일에 대한 쓰기 권한이 없습니다.");
			messageBox.open();

			return;
		}

		MessageBox messageBox = new MessageBox(sShell, SWT.YES | SWT.NO
				| SWT.ICON_QUESTION);
		messageBox.setText("정말로 삭제하시겠습니까?");
		messageBox
				.setMessage("선택된 항목을 정말로 삭제하시겠습니까?\n\n한 번 삭제하면, 다시 되돌릴 수 없습니다.");

		if (messageBox.open() == SWT.YES) {
			final int[] indices = table.getSelectionIndices();

			if (!statusLine.isDisposed()) {
				statusLine.setText("항목을 삭제하는 중입니다.");
				statusLine.update();
			}

			zip.delete(sShell, indices, tmpDir);

			updateTable();
		}
	}

	/**
	 * 압축 파일을 푸는 메소드 압축 파일이 열려있어야만 함
	 */
	private void extract() {
		DirectoryDialog directoryDialog = new DirectoryDialog(sShell, SWT.SAVE);
		directoryDialog.setText("압축을 풀 디렉토리를 선택하세요.");
		directoryDialog.setMessage("압축을 풀 디렉토리를 선택하세요.");
		directoryDialog.setFilterPath(defaultPath);
		String directoryName = directoryDialog.open();

		if (directoryName != null) {
			extract(directoryName);
		}
	}

	/**
	 * 압축 파일을 푸는 메소드 압축 파일이 열려있어야 함
	 * 
	 * @param directoryName
	 *            압축을 풀 디렉토리명
	 */
	private void extract(String directoryName) {
		final File directory = new File(directoryName);

		if (!directory.exists()) {
			// 디렉토리가 존재하지 않는 경우
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("압축 풀기 실패!");
			messageBox.setMessage(directory.getPath() + " 디렉토리가 존재하지 않습니다.");
			messageBox.open();

			return;
		}

		if (!directory.isDirectory()) {
			// 디렉토리가 아니라 파일인 경우
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("압축 풀기 실패!");
			messageBox
					.setMessage(directory.getPath() + " 파일을 디렉토리가 아니라 파일입니다.");
			messageBox.open();

			return;
		}

		if (!directory.canWrite()) {
			// 디렉토리에 쓰기 권한이 없는 경우
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("압축 풀기 실패!");
			messageBox.setMessage(directory.getPath()
					+ " 디렉토리에 대한 쓰기 권한이 없습니다.");
			messageBox.open();

			return;
		}

		if (!statusLine.isDisposed()) {
			statusLine.setText("압축을 해제하는 중입니다.");
			statusLine.update();
		}

		if (table.getSelectionCount() >= 1) {
			// 선택된 항목이 있는 경우
			
			zip.extract(sShell, directory, table.getSelectionIndices());
		} else {
			// 선택된 항목이 없는 경우
			
			zip.extractAll(sShell, directory);
		}

		updateStatusLine();
	}

	/**
	 * 저장 가능한 파일인지 확인하는 메소드
	 * 
	 * @param fileName
	 *            확인할 파일명
	 * @return 저장 가능한 {@link File}, 저장 가능하지 않다면 null
	 */
	private File getSaveFile(String fileName) {
		File file = new File(fileName); // 저장할 압축 파일
		File parent = file.getParentFile(); // 저장할 압축파일에 대한 부모 디렉토리

		if (parent.exists()) {
			// 부모 디렉토리가 존재하는 경우
			
			if (!parent.canWrite()) {
				MessageBox messageBox = new MessageBox(sShell, SWT.OK
						| SWT.ICON_ERROR);
				messageBox.setText("압축 파일 생성 실패!");
				messageBox.setMessage(parent.getAbsolutePath()
						+ " 디렉토리에 대한 쓰기 권한이 없습니다.");
				messageBox.open();

				return null;
			}
		} else {
			// 부모 디렉토리가 존재하지 않는 경우
			
			if (!parent.mkdirs()) {
				MessageBox messageBox = new MessageBox(sShell, SWT.OK
						| SWT.ICON_ERROR);
				messageBox.setText("압축 파일 생성 실패!");
				messageBox.setMessage(parent.getAbsolutePath()
						+ " 디렉토리 생성에 실패했습니다.");
				messageBox.open();

				return null;
			}
		}

		if (file.exists()) {
			// 같은 이름을 가진 파일이 이미 존재하는 경우
			
			MessageBox messageBox = new MessageBox(sShell, SWT.YES | SWT.NO
					| SWT.ICON_QUESTION);
			messageBox.setText("파일을 덮어쓸까요?");
			messageBox.setMessage(fileName + " 파일이 이미 존재합니다.\n\n이 파일을 덮어쓸까요?");

			if (messageBox.open() == SWT.YES) {
				if (!file.canWrite()) {
					MessageBox messageBox1 = new MessageBox(sShell, SWT.OK
							| SWT.ICON_ERROR);
					messageBox1.setText("압축 파일 생성 실패!");
					messageBox1.setMessage(fileName + " 파일에 대한 쓰기 권한이 없습니다.");
					messageBox1.open();

					return null;
				}
			} else {
				MessageBox messageBox1 = new MessageBox(sShell, SWT.OK
						| SWT.ICON_INFORMATION);
				messageBox1.setText("압축 파일 저장 취소!");
				messageBox1.setMessage("압축 파일 저장이 취소되었습니다.");
				messageBox1.open();

				return null;
			}
		}

		return file;
	}

	/**
	 * 압축 파일을 불러오는 메소드
	 */
	private void open() {
		FileDialog fileDialog = new FileDialog(sShell, SWT.OPEN);
		fileDialog.setText("불러올 압축 파일을 선택하세요.");
		fileDialog
				.setFilterExtensions(new String[] { "*.zip", "*.jar", "*.*" });
		fileDialog.setFilterNames(new String[] { "Zip 파일", "Jar 파일", "압축 파일" });
		fileDialog.setFilterPath(defaultPath);
		String fileName = fileDialog.open();

		if (fileName != null) {
			open(fileName);
		}
	}

	/**
	 * Zip 파일을 불러오는 메소드
	 * 
	 * @param fileName
	 *            불러올 파일명(null이면 안됨)
	 */
	private void open(String fileName) {
		File file = new File(fileName);

		if (!file.exists()) {
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("압축 파일 열기 실패!");
			messageBox.setMessage(file.getPath() + " 파일이 존재하지 않습니다.");
			messageBox.open();

			return;
		}

		if (file.isDirectory()) {
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("압축 파일 열기 실패!");
			messageBox.setMessage(file.getPath() + " 디렉토리는 파일이 아니라 디렉토리입니다.");
			messageBox.open();

			return;
		}

		if (!file.canRead()) {
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("압축 파일 열기 실패!");
			messageBox.setMessage(file.getPath() + " 파일에 대한 읽기 권한이 없습니다.");
			messageBox.open();

			return;
		}

		// 인코딩
		String encoding = "MS949";
		if (radioUtf8.getSelection()) {
			encoding = "UTF8";
		}

		ZipFile zipFile = null;
		try {
			zipFile = new ZipFile(file, encoding);
		} catch (IOException e) {
			// Zip 파일이 아닌 경우
			
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("압축 파일 열기 실패!");
			messageBox.setMessage(file.getPath()
					+ " 파일을 열지 못했습니다.\nZip 파일이 아닌 것 같습니다.");
			messageBox.open();

			return;
		}

		// 디렉토리로 보기 여부
		boolean dir = true;
		if (radioFile.getSelection()) {
			dir = false;
		}

		zip = new Zip(file, zipFile, dir);

		if (dir) {
			// 디렉토리로 보기인 경우
			
			if (!tableColumnPath.isDisposed()) {
				tableColumnPath.dispose();
			}
		} else {
			// 모든 파일 보기인 경우
			
			if (tableColumnPath.isDisposed()) {
				addTablecolumnPath();
			}
		}

		updateTable();

		sShell.setText(zip.getFileName() + " - " + jzip);
		
		table.setFocus();
	}

	/**
	 * 항목의 이름을 변경하는 메소드
	 */
	private void rename() {
		if (!zip.canWrite()) {
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("이름 바꾸기 실패!");
			messageBox.setMessage(zip.getFilePath() + " 파일에 대한 쓰기 권한이 없습니다.");
			messageBox.open();

			return;
		}

		final int index = table.getSelectionIndex();
		final String oldName = zip.getEntryName(index);

		RenameDialog renameDialog = new RenameDialog(oldName);
		final String newName = renameDialog.open(sShell);

		if (newName == null) {
			// 이름 바꾸기 취소
			return;
		}

		if (!newName.equals(oldName) && !newName.isEmpty()) {
			if (!statusLine.isDisposed()) {
				statusLine.setText("항목의 이름을 바꾸는 중입니다.");
				statusLine.update();
			}

			zip.rename(sShell, index, newName, tmpDir);

			updateTable();
		}
	}

	/**
	 * 현재 열린 압축 파일을 다른 이름으로 저장하는 메소드
	 */
	private void saveAs() {
		// 압축 파일 선택 대화상자
		FileDialog fileDialog = new FileDialog(sShell, SWT.SAVE);
		fileDialog.setText("저장할 압축 파일을 선택하세요.");
		fileDialog.setFilterPath(defaultPath);
		fileDialog.setFilterExtensions(new String[] { "*.zip", "*.*" });
		fileDialog.setFilterNames(new String[] { "Zip 파일 (*.zip)",
				"모든 파일 (*.*)" });
		String fileName = fileDialog.open();

		if (fileName != null) {
			if (saveAs(fileName)) {
				// 압축 파일 저장 완료 대화상자
				MessageBox messageBox = new MessageBox(sShell, SWT.YES | SWT.NO
						| SWT.ICON_INFORMATION);
				messageBox.setText("압축 완료!");
				final String message = fileName
						+ " 파일 생성이 완료되었습니다.\n\n새로 생성된 파일을 불러올까요?";
				messageBox.setMessage(message);

				if (messageBox.open() == SWT.YES) {
					// 인코딩 설정 변경
					radioUtf8.setSelection(false);
					radioMs949.setSelection(true);

					// 압축 파일 열기
					open(fileName);
				}
			}
		}
	}

	/**
	 * 압축 파일을 저장하는 메소드
	 * 
	 * @param fileName
	 *            저장할 파일명
	 * @return 압축 성공 여부
	 */
	private boolean saveAs(String fileName) {
		File file = getSaveFile(fileName);

		if (file == null) {
			// 압축 파일 저장 실패
			return false;
		}

		if (!statusLine.isDisposed()) {
			statusLine.setText("압축하는 중입니다.");
			statusLine.update();
		}

		if (table.getSelectionCount() >= 1) {
			// 선택된 항목이 있는 경우
			
			zip.save(sShell, file, table.getSelectionIndices());
		} else {
			// 선택된 항목이 없는 경우
			
			zip.saveAll(sShell, file);
		}

		updateStatusLine();

		// 압축 파일 저장 성공
		return true;
	}

	/**
	 * 테이블에 문맥 메뉴를 추가하는 메소드
	 */
	private void setContextMenu() {
		Menu contextMenu = new Menu(table);
		contextMenu.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				updateStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		contextPushOpenDir = new MenuItem(contextMenu, SWT.PUSH);
		contextPushOpenDir.setText("디렉토리 열기");
		contextPushOpenDir.setEnabled(false);
		contextPushOpenDir
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						zip.openDir(table.getSelectionIndex());

						updateTable();
					}
				});
		contextPushOpenDir
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						if (!statusLine.isDisposed()) {
							statusLine.setText("선택된 디렉토리로 이동합니다.");
						}
					}
				});
		@SuppressWarnings("unused")
		MenuItem separator11 = new MenuItem(contextMenu, SWT.SEPARATOR);
		contextPushView = new MenuItem(contextMenu, SWT.PUSH);
		contextPushView.setText("파일 보기(&V)");
		contextPushView.setEnabled(false);
		contextPushView
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						zip.openFile(table.getSelectionIndex(), tmpDir);
					}
				});
		contextPushView
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						if (!statusLine.isDisposed()) {
							statusLine.setText("선택된 파일을 엽니다.");
						}
					}
				});
		contextPushOpen = new MenuItem(contextMenu, SWT.PUSH);
		contextPushOpen.setText("파일을 열 프로그램(&O)...");
		contextPushOpen.setEnabled(false);
		contextPushOpen
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						zip.openWith(sShell, table.getSelectionIndex(), tmpDir);
					}
				});
		contextPushOpen
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						if (!statusLine.isDisposed()) {
							statusLine.setText("선택된 파일을 열 프로그램을 선택합니다.");
						}
					}
				});
		@SuppressWarnings("unused")
		MenuItem separator9 = new MenuItem(contextMenu, SWT.SEPARATOR);
		contextPushExtract = new MenuItem(contextMenu, SWT.PUSH);
		contextPushExtract.setText("풀기(&E)...");
		contextPushExtract.setEnabled(false);
		contextPushExtract
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						extract();
					}
				});
		contextPushExtract
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						if (!statusLine.isDisposed()) {
							statusLine.setText("선택된 항목을 압축 해제합니다.");
						}
					}
				});
		@SuppressWarnings("unused")
		MenuItem separator10 = new MenuItem(contextMenu, SWT.SEPARATOR);
		contextPushRename = new MenuItem(contextMenu, SWT.PUSH);
		contextPushRename.setText("이름 바꾸기(&R)...");
		contextPushRename.setEnabled(false);
		contextPushRename
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						rename();
					}
				});
		contextPushRename
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						if (!statusLine.isDisposed()) {
							statusLine.setText("선택된 항목의 이름을 변경합니다.");
						}
					}
				});
		contextPushDel = new MenuItem(contextMenu, SWT.PUSH);
		contextPushDel.setText("지우기(&D)...");
		contextPushDel.setEnabled(false);
		contextPushDel
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						delete();
					}
				});

		contextPushDel.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				if (!statusLine.isDisposed()) {
					statusLine.setText("선택된 항목을 삭제합니다.");
				}
			}
		});
		table.setMenu(contextMenu);
	}

	/**
	 * DnD 기능을 설정하는 메소드
	 */
	private void setDnd() {
		final DropTarget target = new DropTarget(sShell, DND.DROP_MOVE
				| DND.DROP_COPY | DND.DROP_LINK);
		target.setTransfer(new Transfer[] { FileTransfer.getInstance() });
		final DropListener dropListener = new DropListener();
		target.addDropListener(dropListener);

		DragSource source = new DragSource(table, DND.DROP_MOVE | DND.DROP_COPY
				| DND.DROP_LINK);
		source.setTransfer(new Transfer[] { FileTransfer.getInstance() });
		source.addDragListener(new DragSourceListener() {

			public void dragFinished(DragSourceEvent event) {
				target.addDropListener(dropListener);
			}

			public void dragSetData(DragSourceEvent event) {
				int[] indices = table.getSelectionIndices();
				String[] filePaths = new String[indices.length];
				for (int i = 0; i < filePaths.length; i++) {
					filePaths[i] = zip.extract(indices[i], tmpDir);
				}
				event.data = filePaths;
			}

			public void dragStart(DragSourceEvent event) {
				target.removeDropListener(dropListener);
			}

		});

	}

	/**
	 * 메뉴를 갱신하는 메소드
	 */
	private void updateMenu() {
		if (zip == null) {
			// Zip 파일이 열려있지 않은 경우
			
			// 메뉴 막대
			// Zip 파일
			pushSave.setEnabled(false);
			pushExtract.setEnabled(false);
			pushProperty.setEnabled(false);
			pushClose.setEnabled(false);
			
			// 편집
			pushAddFile.setEnabled(false);
			pushAddDir.setEnabled(false);
			pushRename.setEnabled(false);
			pushDel.setEnabled(false);
			pushSelectAll.setEnabled(false);
			pushDeselectAll.setEnabled(false);
			
			// 보기
			pushRefresh.setEnabled(false);

			// 도구 모음
			if (!toolBar.isDisposed()) {
				toolItemExtract.setEnabled(false);
				toolItemAddFile.setEnabled(false);
				toolItemAddDir.setEnabled(false);
			}

			// 디렉토리로 보기 메뉴
			if (!composite.isDisposed()) {
				buttonUp.setEnabled(false);
				buttonTop.setEnabled(false);
				compositeSeparator.setEnabled(false);
				labelPath.setEnabled(false);
				text.setText("");
				text.setEnabled(false);
			}

			// 문맥 메뉴
			contextPushOpenDir.setEnabled(false);
			contextPushView.setEnabled(false);
			contextPushOpen.setEnabled(false);
			contextPushExtract.setEnabled(false);
			contextPushRename.setEnabled(false);
			contextPushDel.setEnabled(false);
		} else {
			// Zip 파일이 열려있는 경우
			
			// 항목 개수
			final int itemCount = table.getItemCount();
			
			// 선택된 항목 개수
			final int selectionCount = table.getSelectionCount();

			if (itemCount >= 1) {
				// 항목이 하나 이상 존재하는 경우
				
				pushSave.setEnabled(true);
				pushExtract.setEnabled(true);
				contextPushExtract.setEnabled(true);

				if (!toolBar.isDisposed()) {
					toolItemExtract.setEnabled(true);
				}

				if (selectionCount == itemCount) {
					// 모두 선택된 경우
					
					pushSelectAll.setEnabled(false);
				} else {
					pushSelectAll.setEnabled(true);
				}
			} else {
				// 항목이 하나도 없는 경우
				
				pushSave.setEnabled(false);
				pushExtract.setEnabled(false);
				contextPushExtract.setEnabled(false);

				if (!toolBar.isDisposed()) {
					toolItemExtract.setEnabled(false);
				}

				pushSelectAll.setEnabled(false);
			}

			if (selectionCount >= 1) {
				// 하나 이상 선택된 경우
				
				pushDeselectAll.setEnabled(true);
				pushDel.setEnabled(true);
				contextPushDel.setEnabled(true);
			} else {
				// 하나도 선택되지 않은 경우
				
				pushDeselectAll.setEnabled(false);
				pushDel.setEnabled(false);
				contextPushDel.setEnabled(false);
			}

			pushProperty.setEnabled(true);
			pushClose.setEnabled(true);
			pushAddFile.setEnabled(true);
			pushAddDir.setEnabled(true);

			if (!toolBar.isDisposed()) {
				toolItemAddFile.setEnabled(true);
				toolItemAddDir.setEnabled(true);
			}

			if (selectionCount == 1) {
				// 선택된 항목이 하나인 경우
				
				pushRename.setEnabled(true);
				contextPushRename.setEnabled(true);

				if (zip.isDirecotry(table.getSelectionIndex())) {
					// 선택된 항목이 디렉토리인 경우

					if (radioDir.getSelection()) {
						contextPushOpenDir.setEnabled(true);
					} else {
						contextPushOpenDir.setEnabled(false);
					}
					
					contextPushView.setEnabled(false);
					contextPushOpen.setEnabled(false);
				} else {
					// 선택된 항목이 파일인 경우
					
					contextPushOpenDir.setEnabled(false);
					contextPushView.setEnabled(true);
					contextPushOpen.setEnabled(true);
				}
			} else {
				pushRename.setEnabled(false);
				contextPushRename.setEnabled(false);
				contextPushOpenDir.setEnabled(false);
				contextPushView.setEnabled(false);
				contextPushOpen.setEnabled(false);
			}

			pushRefresh.setEnabled(true);

			if (!composite.isDisposed()) {
				final String path = zip.getPath();
				if (path.equals("/")) {
					buttonUp.setEnabled(false);
					buttonTop.setEnabled(false);
				} else {
					buttonUp.setEnabled(true);
					buttonTop.setEnabled(true);
				}

				compositeSeparator.setEnabled(true);
				labelPath.setEnabled(true);
				text.setText(path);
				text.setEnabled(true);
			}
		}
	}

	/**
	 * 상태 표시줄을 갱신하는 메소드
	 */
	private void updateStatusLine() {
		if (statusLine.isDisposed()) {
			return;
		}

		if (zip == null) {
			// Zip 파일이 열려 있지 않은 경우
			
			statusLine.setText("");
		} else {
			// Zip 파일이 열려 있는 경우
			
			String text = "총 " + zip.getSize() + " 항목 ("
					+ Zip.getSizeString(zip.getOriginalLength()) + ")";

			final int count = table.getSelectionCount();

			if (count >= 1) {
				// 선택된 항목이 있는 경우
				
				long totalSize = 0;
				int[] indices = table.getSelectionIndices();
				for (int index : indices) {
					totalSize += zip.getEntrySize(index);
				}

				text += " 중 " + count + " 항목 선택됨 ("
						+ Zip.getSizeString(totalSize) + ")";
			}

			statusLine.setText(text);
		}
	}

	/**
	 * 테이블을 갱신하는 메소드
	 */
	private void updateTable() {
		if (zip == null) {
			// 현재 열려있는 압축 파일이 없는 경우
			
			table.removeAll();
			table.setVisible(false);
		} else {
			// 현재 열려있는 압축 파일이 있는 경우
			
			final boolean reverse = checkReverse.getSelection();

			if (radioName.getSelection()) {
				zip.sortByName(reverse);
			} else if (radioSize.getSelection()) {
				zip.sortBySize(reverse);
			} else if (radioType.getSelection()) {
				zip.sortByType(reverse);
			} else if (radioTime.getSelection()) {
				zip.sortByTime(reverse);
			} else if (radioPath.getSelection()
					&& !tableColumnPath.isDisposed()) {
				zip.sortByPath(reverse);
			} else {
				zip.sortByName(reverse);

				radioName.setSelection(true);
				radioSize.setSelection(false);
				radioType.setSelection(false);
				radioTime.setSelection(false);
				radioPath.setSelection(false);
			}

			table.setItemCount(zip.getSize());
			table.clearAll();
			table.setSortColumn(null);
			table.deselectAll();
			table.setVisible(true);
		}

		updateMenu();

		updateStatusLine();
	}
}
