package net.kldp.j2ee.kupload;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.util.ArrayList;
import java.util.Random;

/**
 * 업로드 한 파일을 파일 시스템에 임시 저장한후 ByteBuffer로 제공한다.
 */
class TempMappedByteBuffer {
	private int size;
	private File file;
	private FileInputStream fin;
	private FileChannel channel;
	private ByteBuffer buffer;
	private static ArrayList<File> deleteFile = new ArrayList<File>();

	/**
	 * 바이트 버버를 생성한다.
	 * 
	 * @param in
	 * @param size
	 * @throws IOException
	 * @throws HttpServletUploadException
	 */
	public TempMappedByteBuffer(InputStream in, int size) throws IOException, HttpServletUploadException {
		System.out.println("Create TempMappedByteBuffer");
		this.size = size;
		file = saveTempFile(in);
		fin = new FileInputStream(file);
		channel = fin.getChannel();
		buffer = channel.map(MapMode.READ_ONLY, 0, size);
	}

	/**
	 * 임시 파일에 저장한다.
	 * 
	 * @param in
	 * @return File
	 * @throws IOException
	 * @throws HttpServletUploadException
	 */
	public File saveTempFile(InputStream in) throws IOException, HttpServletUploadException {
		String prefix = String.valueOf(System.currentTimeMillis());
		String suffix = String.valueOf(new Random().nextInt());
		file = File.createTempFile(prefix, suffix);
		file.deleteOnExit();
		OutputStream out = new BufferedOutputStream(new FileOutputStream(file), 8192 * 4);
		byte[] b = new byte[8192];
		int read;
		try {
			while ((read = in.read(b)) != -1) {
				out.write(b, 0, read);
			}
		} catch (Exception e) {
			throw new HttpServletUploadException("unable to upload.");
		}
		out.close();
		return file;
	}

	/**
	 * 생성된 바이트 버퍼를 리턴한다.
	 * 
	 * @return buffer
	 */
	public ByteBuffer getBuffer() {
		return buffer;
	}

	/**
	 * 크기를 리턴한다. 이 값은 request.getContentLength()와 같다.
	 * 
	 * @return size
	 */
	public int getSize() {
		return size;
	}

	/**
	 * ByteBuffer가 가비지 컬렉터에 수집되기 전까지는 file이 삭제되지 않는다. 따라서 file을 deleteFile에 넣고,
	 * 이전에 넣은 file을 삭제 시도한다.
	 */
	@Override
	protected synchronized void finalize() throws Throwable {
		super.finalize();
		buffer = ByteBuffer.allocate(1);
		buffer = null;
		channel.close();
		channel = null;
		fin.close();
		for (File f : deleteFile)
			if (f.delete()) {
				deleteFile.remove(f);
				System.out.println(f.getAbsolutePath() + "deleted.");
			}
		deleteFile.add(0, file);
		file = null;
	}
}