#ifndef PROCESS_H
#define PROCESS_H

#ifndef __CORESERVICES__
#include <CoreServices/CoreServices.h>
#endif

#include <sys/types.h>

#include <AvailabilityMacros.h>

#if PRAGMA_ONCE
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*Result Codes

The table below lists the most common result codes returned by the Process Manager.
Result Code	Value	Description
procNotFound	-600	
No eligible process with specified process serial number.
memFragErr	-601	
Not enough room to launch application with special requirements.
appModeErr	-602	
Addressing mode is 32-bit, but application is not 32-bit clean.
protocolErr	-603	
app made module calls in improper order
hardwareConfigErr	-604	
Hardware configuration not supported.
appMemFullErr	-605	
Partition size specified in SIZE resource is not big enough for launch.
appIsDaemon	-606	
Application runs in background only.
wrongApplicationPlatform	-875	
The application could not launch because the required platform is not available.
appVersionTooOld	-876	
The application's creator and version are incompatible with the current version of Mac OS.
notAppropriateForClassic	-877	
This application will not (or should not) run in Classic.*/
enum {
procNotFound	=-600	,
memFragErr	=-601	,
appModeErr	=-602	,
protocolErr	=-603	,
hardwareConfigErr=	-604	,
appMemFullErr	=-605	,
appIsDaemon	=-606	,
wrongApplicationPlatform=	-875	,
appVersionTooOld	=-876	,
notAppropriateForClassic	=-877	
};

/*AppParameters

Defines the first high-level event sent to a newly-launched application.

struct AppParameters {
   struct {
      UInt16 what;
      UInt32 message;
      UInt32 when;
      Point where;
      UInt16 modifiers;
   } theMsgEvent;
   unsigned long eventRefCon
   unsigned long messageLength
};
typedef struct AppParameters AppParameters;
typedef AppParameters * AppParametersPtr;
Discussion

The application parameters structure is used in the launchAppParameters field of the launch parameter block, LaunchParamBlockRec , whose address is passed to the LaunchApplication function.
Availability
Available in Mac OS X v10.0 and later.*/
struct AppParameters {
   struct {
      UInt16 what;
      UInt32 message;
      UInt32 when;
      Point where;
      UInt16 modifiers;
   } theMsgEvent;
   unsigned long eventRefCon;
   unsigned long messageLength;
};
typedef struct AppParameters AppParameters;
typedef AppParameters * AppParametersPtr;

/*Launch Options

Specifies the valid launch options in the launchControlFlags field of the launch parameter block.

typedef unsigned short LaunchFlags;
enum {
   launchContinue                = 0x4000,
   launchNoFileFlags             = 0x0800,
   launchUseMinimum              = 0x0400,
   launchDontSwitch              = 0x0200,
   launchAllow24Bit              = 0x0100,
   launchInhibitDaemon           = 0x0080
};
Constants
launchContinue

Set this flag if you want your application to continue after the specified application is launched. If you do not set this flag, LaunchApplication terminates your application after launching the specified application, even if the launch fails.

Available in Mac OS X v10.0 and later.
launchNoFileFlags

Set this flag if you want the LaunchApplication function to ignore any value specified in the launchFileFlags field. If you set the launchNoFileFlags flag, the LaunchApplication function extracts the Finder flags from the application file for you. If you want to supply the file flags, clear the launchNoFileFlags flag and specify the Finder flags in the launchFileFlags field of the launch parameter block.

Available in Mac OS X v10.0 and later.
launchUseMinimum

Clear this flag if you want the LaunchApplication function to attempt to launch the application in the preferred size (as specified in the application’s 'SIZE' resource). If you set the launchUseMinimum flag, the LaunchApplication function attempts to launch the application using the largest available size greater than or equal to the minimum size but less than the preferred size. If the LaunchApplication function returns the result code memFullErr or memFragErr, the application cannot be launched under the current memory conditions.

Available in Mac OS X v10.0 and later.
launchDontSwitch

Set this flag if you do not want the launched application brought to the front. If you set this flag, the launched application runs in the background until the user brings the application to the front—for example, by clicking in one of the application’s windows. Note that most applications expect to be launched in the foreground. If you clear the launchDontSwitch flag, the launched application is brought to the front, and your application is sent to the background.

Available in Mac OS X v10.0 and later.
launchAllow24Bit

Available in Mac OS X v10.0 and later.
launchInhibitDaemon

Set this flag if you do not want LaunchApplication to launch a background-only application. (A background-only application has the onlyBackground flag set in its 'SIZE' resource.)

Available in Mac OS X v10.0 and later.
Discussion

For more information, see LaunchApplication and LaunchParamBlockRec.*/
typedef unsigned short LaunchFlags;
enum {
   launchContinue                = 0x4000,
   launchNoFileFlags             = 0x0800,
   launchUseMinimum              = 0x0400,
   launchDontSwitch              = 0x0200,
   launchAllow24Bit              = 0x0100,
   launchInhibitDaemon           = 0x0080
};

/*LaunchParamBlockRec

Defines the required parameters when launching an application.

struct LaunchParamBlockRec {
   unsigned long reserved1;
   unsigned short reserved2;
   unsigned short launchBlockID;
   unsigned long launchEPBLength;
   unsigned short launchFileFlags;
   LaunchFlags launchControlFlags;
   FSSpecPtr launchAppSpec;
   ProcessSerialNumber launchProcessSN;
   unsigned long launchPreferredSize;
   unsigned long launchMinimumSize;
   unsigned long launchAvailableSize;
   AppParametersPtr launchAppParameters;
};
typedef struct LaunchParamBlockRec LaunchParamBlockRec;
typedef LaunchParamBlockRec * LaunchPBPtr;
Fields
reserved1

Reserved.
reserved2

Reserved.
launchBlockID

A value that indicates whether you are using the fields following it in the launch parameter block. Specify the constant extendedBlock if you use the fields that follow it.
launchEPBLength

The length of the fields following this field in the launch parameter block. Use the constant extendedBlockLen to specify this value.
launchFileFlags

The Finder flags for the application file. Set the launchNoFileFlags constant in the launchControlFlags field if you want the LaunchApplication function to extract the Finder flags from the application file and to set the launchFileFlags field for you.
launchControlFlags

See “Launch Options” for a complete description of these flags.
launchAppSpec

A pointer to a file specification structure that gives the location of the application file to launch. 
launchProcessSN

The process serial number returned to your application if the launch is successful. You can use this process serial number in other Process Manager functions to refer to the launched application. 
launchPreferredSize

The preferred partition size for the launched application as specified in the launched application’s 'SIZE' resource. LaunchApplication sets this field to 0 if an error occurred or if the application is already open.
launchMinimumSize

The minimum partition size for the launched application as specified in the launched application’s 'SIZE' resource. LaunchApplication sets this field to 0 if an error occurred or if the application is already open.
launchAvailableSize

The maximum partition size that is available for allocation. This value is returned to your application only if the memFullErr result code is returned. If the application launch fails because of insufficient memory, you can use this value to determine if there is enough memory available to launch in the minimum size.
launchAppParameters

The first high-level event to send to the launched application. If you set this field to NULL, LaunchApplication creates and sends the Open Application Apple event to the launched application. 
Discussion

You specify a launch parameter block as a parameter to the LaunchApplication function.
Availability
Available in Mac OS X v10.0 and later.*/
struct LaunchParamBlockRec {
   unsigned long reserved1;
   unsigned short reserved2;
   unsigned short launchBlockID;
   unsigned long launchEPBLength;
   unsigned short launchFileFlags;
   LaunchFlags launchControlFlags;
   FSSpecPtr launchAppSpec;
   ProcessSerialNumber launchProcessSN;
   unsigned long launchPreferredSize;
   unsigned long launchMinimumSize;
   unsigned long launchAvailableSize;
   AppParametersPtr launchAppParameters;
};
typedef struct LaunchParamBlockRec LaunchParamBlockRec;
typedef LaunchParamBlockRec * LaunchPBPtr;


/*Extension Launch Codes

Specifies the values used when launching extensions.

enum {
   extendedBlock = 0x4C43,
   extendedBlockLen = sizeof(LaunchParamBlockRec) - 12
};*/
enum {
   extendedBlock = 0x4C43,
   extendedBlockLen = sizeof(LaunchParamBlockRec) - 12
};


/*Process Identification Constants

Specifies constants used instead of a process serial number to identify a process.

enum {
   kNoProcess = 0,
   kSystemProcess = 1,
   kCurrentProcess = 2
};
Constants
kNoProcess

Identifies a process that doesn’t exist.

Available in Mac OS X v10.0 and later.
kSystemProcess

Identifies a process that belongs to the Operating System.

Available in Mac OS X v10.0 and later.
kCurrentProcess

Identifies the current process.

Available in Mac OS X v10.0 and later.
Discussion

If you want to use these constants to specify a process, you must populate a process serial number structure (ProcessSerialNumber), passing 0 in the highLongOfPSN field and the appropriate constant (such as kCurrentProcess) in the lowLongOfPSN. For example, to bring the current process forward, you can use the following code:  ProcessSerialNumber psn = { 0, kCurrentProcess };
 SetFrontProcess( &psn );*/
enum {
   kNoProcess = 0,
   kSystemProcess = 1,
   kCurrentProcess = 2
};


/*ProcessInfoRec

Defines the structure of a process information record.

struct ProcessInfoRec {
   unsigned long processInfoLength;
   StringPtr processName;
   ProcessSerialNumber processNumber;
   unsigned long processType;
   OSType processSignature;
   unsigned long processMode;
   Ptr processLocation;
   unsigned long processSize;
   unsigned long processFreeMem;
   ProcessSerialNumber processLauncher;
   unsigned long processLaunchDate;
   unsigned long processActiveTime;
   FSSpecPtr processAppSpec;
};
typedef struct ProcessInfoRec ProcessInfoRec;
typedef ProcessInfoRec * ProcessInfoRecPtr;
Fields
processInfoLength

The number of bytes in the process information structure. For compatibility, you should specify the length of the structure in this field.
processName

The name of the application. This field contains the name of the application as designated by the user at the time the application was opened. For example, for foreground applications, the processName field contains the name as it appears in the Application menu. You must specify NULL in the processName field if you do not want the application name returned. Otherwise, you should allocate at least 32 bytes of storage for the string pointed to by the processName field. Note that the processName field specifies the name of the application, whereas the processAppSpec field specifies the location of the file. 
processNumber

The process serial number. 
processType

The file type of the application, generally 'APPL' for applications and 'appe' for background-only applications launched at startup.
processSignature

The signature (or creator) of the file containing the application.
processMode

Process mode flags. These flags indicate whether the process is an application or desk accessory. For applications, this field also returns information specified in the application’s 'SIZE' resource. This information is returned as flags.

On Mac OS X, some flags in processMode will not be set as they were on Mac OS 9, even for Classic applications. Mac OS X doesn't support applications which can't be sent into the background, so modeCanBackground will always be set. Similarly, Mac OS X applications will always have mode32BitCompatible and modeHighLevelEventAware set
processLocation

The beginning address of the application partition.
processSize

The number of bytes in the application partition (including the heap, stack, and A5 world).
processFreeMem

The number of free bytes in the application heap.
processLauncher

The process serial number of the process that launched the application or desk accessory. If the original launcher of the process is no longer open, the lowLongOfPSN field of the process serial number structure contains the constant kNoProcess.
processLaunchDate

The value of the Ticks global variable at the time that the process was launched.
processActiveTime

The accumulated time, in ticks, during which the process has used the CPU, including both foreground and background processing time.
processAppSpec

The address of a file specification structure that stores the location of the file containing the application or 'DRVR' resource. You should specify NULL in the processAppSpec field if you do not want the FSSpec structure of the file returned. 
Discussion

A process information record is returned by the GetProcessInformation function.
Availability
Available in Mac OS X v10.0 and later.*/
struct ProcessInfoRec {
   unsigned long processInfoLength;
   StringPtr processName;
   ProcessSerialNumber processNumber;
   unsigned long processType;
   OSType processSignature;
   unsigned long processMode;
   Ptr processLocation;
   unsigned long processSize;
   unsigned long processFreeMem;
   ProcessSerialNumber processLauncher;
   unsigned long processLaunchDate;
   unsigned long processActiveTime;
   FSSpecPtr processAppSpec;
};
typedef struct ProcessInfoRec ProcessInfoRec;
typedef ProcessInfoRec * ProcessInfoRecPtr;

/*GetFrontProcess

Gets the process serial number of the front process.

OSErr GetFrontProcess (
   ProcessSerialNumber * PSN
);
Parameters
PSN

On return, a pointer to the process serial number of the process running in the foreground. 
Return Value

A result code. See Process Manager Result Codes. If no process is running in the foreground, returns procNotFound.
Discussion

You can use this function to determine if your process or some other process is in the foreground. You can use the process serial number returned in the PSN parameter in other Process Manager functions. 
Availability
Available in CarbonLib 1.0 and later when running Mac OS 8.1 or later.
Available in Mac OS X version 10.0 and later.
Declared In
Processes.h*/
OSErr GetFrontProcess (
   ProcessSerialNumber * PSN
);

/*GetProcessInformation

Get information about a specific process.

Modified in Carbon

OSErr GetProcessInformation (
   const ProcessSerialNumber * PSN,
   ProcessInfoRec * info
);
Parameters
PSN

A pointer to a valid process serial number. You can pass a process serial number structure containing the constant kCurrentProcess to get information about the current process. See ProcessSerialNumber for more information.
info

On return, a pointer to a structure containing information about the specified process. 
Return Value

A result code. See Process Manager Result Codes.
Discussion

The information returned in the info parameter includes the applications name as it appears in the Application menu, the type and signature of the application, the address of the application partition, the number of bytes in the application partition, the number of free bytes in the application heap, the application that launched the application, the time at which the application was launched, and the location of the application file.

The GetProcessInformation function also returns information about the applications 'SIZE' resource and indicates whether the process is an application or a desk accessory.

You need to specify values for the processInfoLength, processName, and processAppSpec fields of the process information structure. Specify the length of the process information structure in the processInfoLength field. If you do not want information returned in the processName and processAppSpec fields, specify NULL for these fields. Otherwise, allocate at least 32 bytes of storage for the string pointed to by the processName field and, in the processAppSpec field, specify a pointer to an FSSpec structure.

The processName field may not be what you expect, especially if an application has a localized name. The processName field, if not NULL, on return will contain the filename part of the executable file of the application. If you want the localized, user-displayable name for an application, call CopyProcessName.
Special Considerations

Do not call GetProcessInformation at interrupt time.
Availability
Modified in Carbon. Available in CarbonLib 1.0 and later when running Mac OS 8.1 or later.
Available in Mac OS X version 10.0 and later.
Carbon Porting Notes

The processActiveTime field of the returned ProcessInfoRec structure is not used in Carbon on Mac OS X. In Mac OS 8 and 9, the accumulated amount of CPU time used by the process is returned in this field. In Mac OS X, the value returned in this field is always 0.

On Mac OS X, some flags in processMode will not be set as they were on Mac OS 9, even for Classic applications. Mac OS X doesn't support applications which can't be sent into the background, so modeCanBackground will always be set. Similarly, Mac OS X applications will always have mode32BitCompatible and modeHighLevelEventAware set.

Other fields in the ProcessInfoRec structure (such as processSize and processFreeMem) may also no longer be applicable.

In most cases, you should consider using ProcessInformationCopyDictionary if you are running only on Mac OS X. 
Declared In
Processes.h */

OSErr GetProcessInformation (
   const ProcessSerialNumber * PSN,
   ProcessInfoRec * info
);


/*GetCurrentProcess

Gets information about the current process, if any.

OSErr GetCurrentProcess (
   ProcessSerialNumber * PSN
);
Parameters
PSN

On output, a pointer to the process serial number of the current process, that is, the one currently accessing the CPU. This application can be running in either the foreground or the background. 
Return Value

A result code. See Process Manager Result Codes.
Discussion

Applications can use this function to find their own process serial number. Drivers can use this function to find the process serial number of the current process. You can use the returned process serial number in other Process Manager functions. 

This function is named MacGetCurrentProcess on non Macintosh platforms and GetCurrentProcess on Macintosh computers. However, even Macintosh code can use the MacGetCurrentProcess name because a macro exists that automatically maps that call to GetCurrentProcess.
Availability
Available in CarbonLib 1.0 and later when running Mac OS 8.1 or later.
Available in Mac OS X version 10.0 and later.
Declared In
Processes.h*/
OSErr GetCurrentProcess (
   ProcessSerialNumber * PSN
);

/*SetFrontProcess

Moves a process to the foreground.

OSErr SetFrontProcess (
   const ProcessSerialNumber * PSN
);
Parameters
PSN

A pointer to a valid process serial number. You can also pass a process serial number structure containing the constant kCurrentProcess to refer to the current process. See ProcessSerialNumber for more information.
Return Value

A result code. See Process Manager Result Codes.
Discussion

The SetFrontProcess function moves the specified process to the foreground immediately.

If the specified process serial number is invalid or if the specified process is a background-only application, SetFrontProcess returns a nonzero result code and does not change the current foreground process.
Special Considerations

Do not call SetFrontProcess at interrupt time.
Availability
Available in CarbonLib 1.0 and later when running Mac OS 8.1 or later.
Available in Mac OS X version 10.0 and later.
Declared In
Processes.h*/
OSErr SetFrontProcess (
   const ProcessSerialNumber * PSN
);

/*GetNextProcess

Gets information about the next process, if any, in the Process Managers internal list of open processes.

OSErr GetNextProcess (
   ProcessSerialNumber * PSN
);
Parameters
PSN

On input, a pointer to the process serial number of a process. This number should be a valid process serial number returned from LaunchApplication, GetFrontProcess, or GetCurrentProcess, or a process serial number structure containing kNoProcess. For details about this structure, see ProcessSerialNumber. On return, a pointer to the process serial number of the next process, or else kNoProcess. 
Return Value

A result code. See Process Manager Result Codes.
Discussion

The Process Manager maintains a list of all open processes. You can derive this list by using repetitive calls to GetNextProcess. Begin generating the list by calling GetNextProcess and specifying the constant kNoProcess in the PSN parameter. You can then use the returned process serial number to get the process serial number of the next process. Note that the order of the list of processes is internal to the Process Manager. When the end of the list is reached, GetNextProcess returns the constant kNoProcess in the PSN parameter and the result code procNotFound.

You can use the returned process serial number in other Process Manager functions. You can also use this process serial number to specify a target application when your application sends a high-level event. 
Availability
Available in CarbonLib 1.0 and later when running Mac OS 8.1 or later.
Available in Mac OS X version 10.0 and later.
Declared In  Processes.h*/
OSErr GetNextProcess (
   ProcessSerialNumber * PSN
);

/*ExitToShell

Terminates an application.

void ExitToShell (void);
Discussion

In general, you need to call ExitToShell only if you want your application to terminate without reaching the end of its main function.

The ExitToShell function terminates the calling process. The Process Manager removes your application from the list of open processes and performs any other necessary cleanup operations. In particular, all memory in your application partition and any temporary memory still allocated to your application is released. If necessary, the Application Died Apple event is sent to the process that launched your application.
Availability
Available in CarbonLib 1.0 and later when running Mac OS 8.1 or later.
Available in Mac OS X version 10.0 and later.
Declared In
Processes.h*/
void ExitToShell (void);

/*GetProcessBundleLocation

Retrieves the file system location of the application bundle (or executable file) associated with a process.

OSStatus GetProcessBundleLocation (
   const ProcessSerialNumber * PSN,
   FSRef * location
);
Parameters
PSN

A pointer to a valid process serial number. See ProcessSerialNumber for more information.
location
Return Value

A result code. See “Process Manager Result Codes”.
Availability
Available in CarbonLib 1.5 and later.
Available in Mac OS X version 10.0 and later.
Declared In
Processes.h*/
OSStatus GetProcessBundleLocation (
   const ProcessSerialNumber * PSN,
   FSRef * location
);

/*GetProcessForPID

Obtains the process serial number from a Unix PID. 

OSStatus GetProcessForPID(
   pid_t                  pid,
   ProcessSerialNumber *  psn);
Parameters
pid

The Unix process ID (PID).
psn

On return, psn points to the process serial number.
Return Value

A result code. See “Process Manager Result Codes”.
Discussion

Note that this call does not make sense for Classic applications, since they all share a single UNIX process ID.
Availability
Available in Mac OS X version 10.0 and later in ApplicationServices.framework.
Declared In
Processes.h*/
OSStatus GetProcessForPID(
   pid_t                  pid,
   ProcessSerialNumber *  psn);

/*LaunchApplication

Launches an application.

OSErr LaunchApplication (
   LaunchPBPtr LaunchParams
);
Parameters
LaunchParams

A pointer to a LaunchParamBlockRec specifying information about the application to launch. 
Return Value

A result code. See “Process Manager Result Codes”.
Discussion

The LaunchApplication function launches the application from the specified file and returns the process serial number, preferred partition size, and minimum partition size if the application is successfully launched.

Note that if you launch another application without terminating your application, the launched application is not actually executed until you make a subsequent call to WaitNextEvent or EventAvail.

Set the launchContinue flag in the launchControlFlags field of the launch parameter block if you want your application to continue after the specified application is launched. If you do not set this flag, LaunchApplication terminates your application after launching the specified application, even if the launch fails. 
Availability
Available in CarbonLib 1.0 and later when running Mac OS 8.1 or later.
Available in Mac OS X version 10.0 and later.
Declared In
Processes.h*/
OSErr LaunchApplication (
   LaunchPBPtr LaunchParams
);

#ifdef __cplusplus
}
#endif

#endif