#ifndef MULTIPROCESSING_H
#define MULTIPROCESSING_H

#ifndef MACTYPES_H
#include <CarbonCore/MacTypes.h>
#endif



#include <HYS/CarbonCore/Multiprocessing.h>
#include <AvailabilityMacros.h>

#if PRAGMA_ONCE
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*MPTaskID

Represents a task ID. 

typedef struct OpaqueMPTaskID * MPTaskID;
Discussion

You obtain a task ID by calling the function MPCreateTask. 
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct OpaqueMPTaskID * MPTaskID;

/*Task Creation Options

Specify optional actions when calling the MPCreateTask function.

enum {
   kMPCreateTaskTakesAllExceptionsMask = 2,
   kMPCreateTaskNotDebuggableMask = 4,
   kMPCreateTaskValidOptionsMask = 6
};
typedef OptionBits MPTaskOptions;
Constants
Discussion
Version Notes

Introduced with Multiprocessing Services 2.1.*/
enum {
   kMPCreateTaskTakesAllExceptionsMask = 2,
   kMPCreateTaskNotDebuggableMask = 4,
   kMPCreateTaskValidOptionsMask = 6
};
typedef OptionBits MPTaskOptions;

/*MPQueueID

Represents a queue ID, which Multiprocessing Services uses to manipulate message queues.

typedef struct OpaqueMPQueueID * MPQueueID;
Discussion

You obtain a queue ID by calling the function MPCreateQueue. 
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct OpaqueMPQueueID * MPQueueID;



/*Timer Duration Constants

Specify the maximum time a task should wait for an event to occur. 

enum {
   kDurationImmediate = 0,
   kDurationForever = 0x7FFFFFFF,
   kDurationMillisecond = 1,
   kDurationMicrosecond = -1
};
Constants
kDurationImmediate

The task times out immediately, whether or not the event has occurred. If the event occurred, the return status is noErr. If the event did not occur, the return status is kMPTimeoutErr (assuming no other errors occurred). 

Available in Mac OS X v10.0 and later.
kDurationForever

The task waits forever. The blocking call waits until either the event occurs, or until the object being waited upon (such as a message queue) is deleted.

Available in Mac OS X v10.0 and later.
kDurationMillisecond

The task waits one millisecond before timing out. 

Available in Mac OS X v10.0 and later.
kDurationMicrosecond

The task waits one microsecond before timing out. 

Available in Mac OS X v10.0 and later.
Discussion

You can use these constants in conjunction with other values to indicate specific wait intervals. For example, to wait 1 second, you can pass kDurationMillisecond * 1000. 
Version Notes

Introduced with Multiprocessing Services 2.0.*/
enum {
   kDurationImmediate = 0,
   kDurationForever = 0x7FFFFFFF,
   kDurationMillisecond = 1,
   kDurationMicrosecond = -1
};

/*MPSemaphoreID

Represents a semaphore ID, which Multiprocessing Services uses to manipulate semaphores.

typedef struct OpaqueMPSemaphoreID * MPSemaphoreID;
Discussion

You obtain a semaphore ID by calling the function MPCreateSemaphore. 
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct OpaqueMPSemaphoreID * MPSemaphoreID;

/*MPSemaphoreCount

Represents a semaphore count. 

typedef ItemCount MPSemaphoreCount;
Discussion
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef ItemCount MPSemaphoreCount;

/*MPProcessID

Represents a process ID. 

typedef struct OpaqueMPProcessID * MPProcessID;
Discussion

Note that this process ID is identical to the process ID (or context ID) handled by the Code Fragment Manager. 
Version Notes

Introduced with Multiprocessing Services 2.0. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct OpaqueMPProcessID * MPProcessID;



//function
/*MPCreateSemaphore

Creates a semaphore.

OSStatus MPCreateSemaphore (
   MPSemaphoreCount maximumValue,
   MPSemaphoreCount initialValue,
   MPSemaphoreID * semaphore
);
Parameters
maximumValue

The maximum allowed value of the semaphore. 
initialValue

The initial value of the semaphore.
semaphore

On return, semaphore contains the ID of the newly–created semaphore. 
Return Value

A result code. See "Multiprocessing Services Result Codes".
Discussion

If you want to create a binary semaphore, you can call the macro MPCreateBinarySemaphore (MPSemaphoreID *semaphore) instead, which simply calls MPCreateSemaphore with both maximumValue and initialValue set to 1.

Also see the function MPDeleteSemaphore. 
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Available in CarbonLib 1.0 and later when MultiprocessingLib 1.0 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
Multiprocessing.h*/
OSStatus MPCreateSemaphore (
   MPSemaphoreCount maximumValue,
   MPSemaphoreCount initialValue,
   MPSemaphoreID * semaphore
);

/*MPSignalSemaphore

Signals a semaphore. 

OSStatus MPSignalSemaphore (
   MPSemaphoreID semaphore
);
Parameters
semaphore

The ID of the semaphore you want to signal. 
Return Value

A result code. See "Multiprocessing Services Result Codes". If the value of the semaphore was already at the maximum, MPSignalSemaphore returns kInsufficientResourcesErr.
Discussion

If tasks are waiting on the semaphore, the oldest (first queued) task is unblocked so that the corresponding MPWaitOnSemaphore call for that task completes. Otherwise, if the value of the semaphore is not already equal to its maximum value, it is incremented by one.

Note that you can call this function from an interrupt handler. 
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Available in CarbonLib 1.0 and later when MultiprocessingLib 1.0 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
Multiprocessing.h*/
OSStatus MPSignalSemaphore (
   MPSemaphoreID semaphore
);

/*MPWaitOnSemaphore

Waits on a semaphore

OSStatus MPWaitOnSemaphore (
   MPSemaphoreID semaphore,
   Duration timeout
);
Parameters
semaphore

The ID of the semaphore you want to wait on. 
timeout

The maximum time the function should wait before timing out. See Timer Duration Constants for a list of constants you can use to specify the wait interval. 
Return Value

A result code. See "Multiprocessing Services Result Codes".
Discussion

If the value of the semaphore is greater than zero, the value is decremented and the function returns with noErr. Otherwise, the task is blocked awaiting a signal until the specified timeout is exceeded.

If you call this function from a cooperative task, you should specify only kDurationImmediate for the timeout length; other waits will cause the task to block. 

Also see the function MPSignalSemaphore. 
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Available in CarbonLib 1.0 and later when MultiprocessingLib 1.0 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
Multiprocessing.h*/
OSStatus MPWaitOnSemaphore (
   MPSemaphoreID semaphore,
   Duration timeout
);

/*MPDeleteSemaphore

Removes a semaphore. 

OSStatus MPDeleteSemaphore (
   MPSemaphoreID semaphore
);
Parameters
semaphore

The ID of the semaphore you want to remove. 
Return Value

A result code. See "Multiprocessing Services Result Codes".
Discussion

Calling this function unblocks all tasks waiting on the semaphore and the tasks’ respective MPWaitOnSemaphore calls will return with the result code kMPDeletedErr.

Also see the function MPCreateSemaphore. 
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Available in CarbonLib 1.0 and later when MultiprocessingLib 1.0 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
Multiprocessing.h*/

OSStatus MPDeleteSemaphore (
   MPSemaphoreID semaphore
);

/*_MPIsFullyInitialized

Indicates whether Multiprocessing Services is available for use.

Boolean _MPIsFullyInitialized ();
Parameters
Return Value
Discussion

Checking for the availability of the MP API is rather ugly. This is a historical problem, caused by the original implementation letting itself get prepared when it really wasn't usable and complicated by some important clients then depending on weak linking to "work". (And further complicated by CFM not supporting "deferred" imports, which is how many programmers think weak imports work.)

The end result is that the MP API library may get prepared by CFM but be totally unusable. This means that if you import from the MP API library, you cannot simply check for a resolved import to decide if MP services are available. Worse, if you explicitly prepare the MP API library you cannot assume that a noErr result from GetSharedLibrary means that MP services are available.
If you import from the MP API library you must do the following:

Use the MPLibraryIsLoaded macro (or equivalent code in languages other than C) to tell if the MP API services are available. It is not sufficient to simply check that an imported symbol is resolved as is commonly done for other libraries. The macro expands to the expression: ( ( (UInt32)_MPIsFullyInitialized != (UInt32)kUnresolvedCFragSymbolAddress  )  &&
              ( _MPIsFullyInitialized () ) )

This code checks if the imported symbol _MPIsFullyInitialized is resolved and, if resolved, calls it. Both parts must succeed for the MP API services to be available.
If you explicitly prepare the MP API library you must use code similar to the following example to tell if the MP API services are available. It is not sufficient to depend on just a noErr result from GetSharedLibrary. OSErr err;
Boolean mpIsAvailable = false;
CFragConnectionID connID = kInvalidID;
MPIsFullyInitializedProc    mpIsFullyInitialized    = NULL;
 
err = GetSharedLibrary  ( "\pMPLibrary", kCompiledCFragArch,                             kReferenceCFrag, &connID, NULL,  NULL );
 
if ( err == noErr ) {
err = FindSymbol ( connID, "\p_MPIsFullyInitialized",
                    (Ptr *) &mpIsFullyInitialized, NULL );
}
 
if ( err == noErr ) {mpIsAvailable = (* mpIsFullyInitialized) ();
    }

Availability
Available in CarbonLib 1.0 and later when MultiprocessingLib 1.0 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
Multiprocessing.h*/
Boolean _MPIsFullyInitialized ();

#define MPLibraryIsLoaded \
( ( (UInt32)_MPIsFullyInitialized != (UInt32)kUnresolvedCFragSymbolAddress  )  && \
              ( _MPIsFullyInitialized () ) )

/*MPCreateTask

Creates a preemptive task.

Modified

OSStatus MPCreateTask (
   TaskProc entryPoint,
   void * parameter,
   ByteCount stackSize,
   MPQueueID notifyQueue,
   void * terminationParameter1,
   void * terminationParameter2,
   MPTaskOptions options,
   MPTaskID * task
);
Parameters
entryPoint

A pointer to the task function. The task function should take a single 32-bit parameter and return a value of type OSStatus. 
parameter

The parameter to pass to the task function. 
stackSize

The size of the stack assigned to the task. Note that you should be careful not to exceed the bounds of the stack, since stack overflows may not be detected. Specifying zero for the size will result in a default stack size of 4KB. 

Note that in Mac OS X prior to version 10.1, this parameter is ignored, and all stacks have the default size of 512 KB. Versions 10.1 and later do not have this limitation.

notifyQueue

The ID of the message queue to which the system will send a message when the task terminates. You specify the first 64-bits of the message in the parameters terminationParameter1 and terminationParameter2 respectively. The last 32-bits contain the result code of the task function. 
terminationParameter1

A 32-bit value that is sent to the message queue specified by the parameter notifyQueue when the task terminates.
terminationParameter2

A 32-bit value that is sent to the message queue specified by the parameter notifyQueue when the task terminates.
options

Optional attributes of the preemptive task. See Task Creation Options for a list of possible values. 
task

On return, task points to the ID of the newly created task. 
Return Value

A result code. See "Multiprocessing Services Result Codes". If MPCreateTask could not create the task because some critical resource was not available, the function returns kMPInsufficientResourcesErr. Usually this is due to lack of memory to allocate the internal data structures associated with the task or the stack. The function also returns kMPInsufficientResourcesErr if any reserved option bits are set.
Discussion

Tasks are created in the unblocked state, ready for execution. A task can terminate in the following ways:
By returning from its entry point 
By calling MPExit
When specified as the target of an MPTerminateTask call 
If a hardware-detected exception or programming exception occurs and no exception handler is installed 
If the application calls ExitToShell 

Task resources (its stack, active timers, internal structures related to the task, and so on) are reclaimed by the system when the task terminates. The task's address space is inherited from the process address space. All existing tasks are terminated when the owning process terminates.

To set the relative processor weight to be assigned to a task, use the function MPSetTaskWeight.

See also the function MPTerminateTask. 
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Modified in Carbon. Available in CarbonLib 1.0 and later when MultiprocessingLib 1.0 or later is present.
Available in Mac OS X 10.0 and later.
Carbon Porting Notes

In Mac OS X prior to version 10.1, the stackSize parameter is ignored, and all stacks have the default size of 512 KB. Versions 10.1 and later do not have this limitation. 
Declared In
Multiprocessing.h*/
OSStatus MPCreateTask (
   TaskProc entryPoint,
   void * parameter,
   ByteCount stackSize,
   MPQueueID notifyQueue,
   void * terminationParameter1,
   void * terminationParameter2,
   MPTaskOptions options,
   MPTaskID * task
);

/*MPTerminateTask

Terminates an existing task.

OSStatus MPTerminateTask (
   MPTaskID task,
   OSStatus terminationStatus
);
Parameters
task

The ID of the task you wish to terminate. 
terminationStatus

A 32-bit value of type OSStatus indicating termination status. This value is sent to the termination status message queue you specified in MPCreateTask in place of the task function’s result code. 
Return Value

A result code. See "Multiprocessing Services Result Codes". If the task to be terminated is already in the process of termination, MPTerminateTask returns kMPInsufficientResourcesErr. You do not need to take any additional action if this occurs.
Discussion

You should be very careful when calling MPTerminateTask. As defined, this call will asynchronously and abruptly terminate a task, potentially leaving whatever structures or resources it was operating upon in an indeterminate state. Mac OS X exacerbates this problem, as MP tasks can use many more system services that are not expecting client threads to asynchronously terminate, and these services do not take the rather complicated steps necessary to protect against, or recover from, such a situation.

However, there are situations in which calling MPTerminateTask is useful and relatively safe. One such situation is when your application or service is quitting and you know that the task you wish to terminate is waiting on an MP synchonization construct (queue, event, semaphore or critical region). While you could do this more cleanly by waking the task and causing it to exit on its own, doing so may not always be practical.

For example, suppose you have several service tasks performing background processing for your application. These service tasks wait on a queue, onto which the applicaton places requests for processing. When the task is done with a request, it notifies another queue, which the application polls. Since the main application task is placing items on the shared queue, and receiving notifications when the requests are done, it can track whether or not there are outstanding requests being processed. If all outstanding requests have, in fact, been processed, it is relatively safe to terminate a task (or all tasks) waiting on the request queue.

You should not assume that the task has completed termination when this call returns; the proper way to synchronize with task termination is to wait on the termination queue (specified in MPCreateTask ) until a message appears. Because task termination is a multistage activity, it is possible for a preemptive task to attempt to terminate a task that is already undergoing termination. In such cases, MPTerminateTask returns kMPInsufficientResourcesErr.

Note that Multiprocessing Services resources (event groups, queues, semaphores, and critical regions) owned by a preemptive task are not released when that task terminates. If a task has a critical region locked when it terminates, the critical region remains in the locked state. Multiprocessing Services resources no longer needed should be explicitly deleted by the task that handles the termination message. All Multiprocessing Services resources created by tasks are released when their owning process (that is, the host application) terminates. 
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Available in CarbonLib 1.0 and later when MultiprocessingLib 1.0 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
Multiprocessing.h*/
OSStatus MPTerminateTask (
   MPTaskID task,
   OSStatus terminationStatus
);


/*MPTaskIsPreemptive

Determines whether a task is preemptively scheduled. 

Boolean MPTaskIsPreemptive (
   MPTaskID taskID
);
Parameters
taskID

The task you want to check. Pass kMPNoID or kInvalidID if you want to specify the current task. 
Return Value

If true, the task is preemptively scheduled. If false, the task is cooperatively scheduled. 
Discussion

If you have code that may be called from either cooperative or preemptive tasks, that code can call MPTaskIsPreemptive if its actions should differ depending on its execution environment. 

Note that you can call this function from an interrupt handler. 
Version Notes

Introduced with Multiprocessing Services 2.0. 
Availability
Available in CarbonLib 1.0 and later when MultiprocessingLib 2.0 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
Multiprocessing.h*/

Boolean MPTaskIsPreemptive (
   MPTaskID taskID
);

/*MPCurrentTaskID

Obtains the task ID of the currently-executing preemptive task

MPTaskID MPCurrentTaskID ();
Parameters
Return Value

The task ID of the current preemptive task. See the description of the MPTaskID data type.
Discussion

Returns the ID of the current preemptive task. If called from a cooperative task, this function returns an ID which is different than the ID of any preemptive task. Nonpreemptive processes may or may not have different task IDs for each application; future implementations of this API may behave differently in this regard.

Note that you can call this function from an interrupt handler. 
Version Notes

Introduced with Multiprocessing Services 1.0. 
Availability
Available in CarbonLib 1.0 and later when MultiprocessingLib 1.0 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
Multiprocessing.h*/
MPTaskID MPCurrentTaskID ();

#ifdef __cplusplus
}
#endif

#endif