#ifndef CGDATACONSUMER_H
#define CGDATACONSUMER_H

#include <CoreGraphics/CGBase.h>
#include <CoreFoundation/CFURL.h>
#include <stddef.h>

/*CGDataConsumerReleaseInfoCallback

Defines a callback function that releases any private data or resources associated with the data consumer.

void (*CGDataConsumerReleaseInfoCallback) (
   void *info
);
   

If you name your function MyConsumerReleaseInfo, you would declare it like this:

void MyConsumerReleaseInfo (
   void *info
);
Parameters
info

A generic pointer to private data shared among your callback functions. This is the same pointer you supplied to CGDataConsumerCreate.
Discussion

When Quartz frees a data consumer that has an associated release function, the release function is called.

For information on how to associate your callback function with a data consumer, see CGDataConsumerCreate and CGDataConsumerCallbacks.*/
typedef void (*CGDataConsumerReleaseInfoCallback) (
   void *info
);

/*CGDataConsumerPutBytesCallback

Defines a pointer to a callback function that copies data from a Quartz-supplied buffer into a data consumer.

size_t (*CGDataConsumerPutBytesCallback) (
   void *info,
   const void *buffer,
   size_t count
);
   

If you name your function MyConsumerPutBytes, you would declare it like this:

size_t MyConsumerPutBytes (
   void *info,
   const void *buffer,
   size_t count
);
Parameters
info

A generic pointer to private data shared among your callback functions. This is the pointer supplied to CGDataConsumerCreate.
buffer

The Quartz-supplied buffer from which you copy the specified number of bytes.
count

The number of bytes to copy.
Return Value

The number of bytes copied. If no more data can be written to the consumer, you should return 0.
Discussion

When Quartz is ready to send data to the consumer, your function is called. It should copy the specified number of bytes from buffer into some resource under your control—for example, a file.

For information on how to associate your callback function with a data consumer, see CGDataConsumerCreate and CGDataConsumerCallbacks.*/
typedef size_t (*CGDataConsumerPutBytesCallback) (
   void *info,
   const void *buffer,
   size_t count
);


/*CGDataConsumerRef

Defines an opaque type that handles the storage of data supplied by Quartz functions.

typedef struct CGDataConsumer *CGDataConsumerRef;
Discussion

Some Quartz routines supply blocks of data to your program. Rather than writing to a raw memory buffer, data consumer objects of type CGDataConsumerRef provide a destination for data from Quartz functions. You use the function CGDataConsumerCreate to create a data consumer that uses callback functions. You use function CGDataConsumerCreateWithURL to create a data consumer that sends the data to a URL. 

In Mac OS X version 10.2 and later, CGDataConsumerRef is derived from CFTypeRef and inherits the properties that all Core Foundation types have in common. For more information, see CFType Reference.
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct CGDataConsumer *CGDataConsumerRef;

/*CGDataConsumerCallbacks

Defines a structure that contains pointers to callback functions that manage the copying of data for a data consumer.

struct CGDataConsumerCallbacks {
   CGDataConsumerPutBytesCallback putBytes;
   CGDataConsumerReleaseInfoCallback releaseConsumer;
};
typedef struct CGDataConsumerCallbacks CGDataConsumerCallbacks;
Fields
putBytes

A pointer to a function that copies data to the consumer. For more information, see CGDataConsumerPutBytesCallback.
releaseConsumer

A pointer to a function that handles clean-up for the data consumer, or NULL. For more information, see CGDataConsumerReleaseInfoCallback
Discussion

The functions specified by the CGDataConsumerCallbacks structure are responsible for copying data that Quartz sends to your consumer and for handling the consumer’s basic memory management. You supply a CGDataConsumerCallbacks structure to the function CGDataConsumerCreate to create a data consumer. 
Availability
Available in Mac OS X v10.0 and later.*/
struct CGDataConsumerCallbacks {
   CGDataConsumerPutBytesCallback putBytes;
   CGDataConsumerReleaseInfoCallback releaseConsumer;
};
typedef struct CGDataConsumerCallbacks CGDataConsumerCallbacks;


//functions 

/*CGDataConsumerRelease

Decrements the retain count of a data consumer.

void CGDataConsumerRelease (
   CGDataConsumerRef consumer
);
Parameters
consumer

The data consumer to release. 
Discussion

This function is equivalent to CFRelease, except that it does not cause an error if the consumer parameter is NULL.
Availability
Available in Mac OS X version 10.0 and later.
Declared In
CGDataConsumer.h*/
void CGDataConsumerRelease (
   CGDataConsumerRef consumer
);

/*CGDataConsumerCreate

Creates a Quartz data consumer.

CGDataConsumerRef CGDataConsumerCreate (
   void *info,
   const CGDataConsumerCallbacks * callbacks
);
Parameters
info

A pointer to data of any type or NULL. When Quartz calls the functions specified in the callbacks parameter, it passes this pointer as the info parameter.
callbacks

A pointer to a CGDataConsumerCallbacks structure that specifies the callback functions you implement to copy data sent to the consumer and to handle the consumer’s basic memory management. For a complete description, see CGDataConsumerCallbacks. 
Return Value

A new data consumer object. You are responsible for releasing this object using CGDataConsumerRelease.
Availability
Available in Mac OS X version 10.0 and later.
Declared In
CGDataConsumer.h*/
CGDataConsumerRef CGDataConsumerCreate (
   void *info,
   const CGDataConsumerCallbacks * callbacks
);

#endif