#ifndef CGIMAGE_H
#define CGIMAGE_H



#include <CoreGraphics/CGBase.h>
#include <CoreGraphics/CGColorSpace.h>
#include <CoreGraphics/CGDataProvider.h>
#include <CoreFoundation/CFBase.h>
#include <AvailabilityMacros.h>


/*Image Alpha Information

Specifies ways that alpha component data is stored in an image.

enum CGImageAlphaInfo {
   kCGImageAlphaNone,
   kCGImageAlphaPremultipliedLast,
   kCGImageAlphaPremultipliedFirst,
   kCGImageAlphaLast,
   kCGImageAlphaFirst,
   kCGImageAlphaNoneSkipLast,
   kCGImageAlphaNoneSkipFirst
};
typedef enum CGImageAlphaInfo CGImageAlphaInfo;
Constants
kCGImageAlphaFirst

The alpha component is stored in the most significant bits of each pixel. For example, non-premultiplied ARGB.

Available in Mac OS X v10.0 and later.
kCGImageAlphaLast

The alpha component is stored in the least significant bits of each pixel. For example, non-premultiplied RGBA.

Available in Mac OS X v10.0 and later.
kCGImageAlphaNone

There is no alpha channel. If the total size of the pixel is greater than the space required for the number of color components in the color space, the least significant bits are ignored. This value is equivalent to kCGImageAlphaNoneSkipLast.

Available in Mac OS X v10.0 and later.
kCGImageAlphaNoneSkipFirst

There is no alpha channel. If the total size of the pixel is greater than the space required for the number of color components in the color space, the most significant bits are ignored.

Available in Mac OS X v10.0 and later.
kCGImageAlphaOnly

There is no color data, only an alpha channel.

Available in Mac OS X v10.3 and later.
kCGImageAlphaNoneSkipLast

There is no alpha channel. If the total size of the pixel is greater than the space required for the number of color components in the color space, the least significant bits are ignored. This value is equivalent to kCGImageAlphaNone.

Available in Mac OS X v10.0 and later.
kCGImageAlphaPremultipliedFirst

The alpha component is stored in the most significant bits of each pixel and the color components have already been multiplied by this alpha value. For example, premultiplied ARGB.

Available in Mac OS X v10.0 and later.
kCGImageAlphaPremultipliedLast

The alpha component is stored in the least significant bits of each pixel and the color components have already been multiplied by this alpha value. For example, premultiplied RGBA.

Available in Mac OS X v10.0 and later.
Discussion

A CGImageAlphaInfo constant specifies (1) whether a bitmap contains an alpha channel, (2) where the alpha bits are located in the image data, and (3) whether the alpha value is premultiplied. You can obtain a CGImageAlphaInfo constant for an image by calling the function CGImageGetAlphaInfo. (You provide a CGBitmapInfo constant to the function CGImageCreate, part of which is a CGImageAlphaInfo constant.)

Quartz accomplishes alpha blending by combining the color components of the source image with the color components of the destination image using the linear interpolation formula, where “source” is one color component of one pixel of the new paint and “destination” is one color component of the background image. 

Quartz supports premultiplied alpha only for images. You should not premultiply any other color values specified in Quartz.*/
enum CGImageAlphaInfo {
   kCGImageAlphaNone,
   kCGImageAlphaPremultipliedLast,
   kCGImageAlphaPremultipliedFirst,
   kCGImageAlphaLast,
   kCGImageAlphaFirst,
   kCGImageAlphaNoneSkipLast,
   kCGImageAlphaNoneSkipFirst
};
typedef enum CGImageAlphaInfo CGImageAlphaInfo;


/*Image Bitmap Information

Specifies bitmap information for an image.

enum {
   kCGBitmapAlphaInfoMask = 0x1F,
   kCGBitmapFloatComponents = (1 << 8),
   
   kCGBitmapByteOrderMask = 0x7000,
   kCGBitmapByteOrderDefault = (0 << 12),
   kCGBitmapByteOrder16Little = (1 << 12),
   kCGBitmapByteOrder32Little = (2 << 12),
   kCGBitmapByteOrder16Big = (3 << 12),
   kCGBitmapByteOrder32Big = (4 << 12)
};
typedef uint32_t CGBitmapInfo;
   
#ifdef __BIG_ENDIAN__
   kCGBitmapByteOrder16Host kCGBitmapByteOrder16Big
   kCGBitmapByteOrder32Host kCGBitmapByteOrder32Big
#else
   kCGBitmapByteOrder16Host kCGBitmapByteOrder16Little
   kCGBitmapByteOrder32Host kCGBitmapByteOrder32Little
#endif
Constants
kCGBitmapAlphaInfoMask

Used to extract alpha information, which specifies whether a bitmap contains an alpha channel and how the alpha channel is generated.

Available in Mac OS X v10.4 and later.
kCGBitmapFloatComponents

Specifies whether the components of a bitmap are floating-point or integer values.

Available in Mac OS X v10.4 and later.
kCGBitmapByteOrderMask

Used to extract information on the byte ordering of pixel formats. 

Available in Mac OS X v10.4 and later.
kCGBitmapByteOrderDefault

Specifies to use the default byte order. 

Available in Mac OS X v10.4 and later.
kCGBitmapByteOrder16Little

Specifies 16-bit, little endian format.

Available in Mac OS X v10.4 and later.
kCGBitmapByteOrder32Little

Specifies 32-bit, little endian format. 

Available in Mac OS X v10.4 and later.
kCGBitmapByteOrder16Big

Specifies 16-bit, big endian format. 

Available in Mac OS X v10.4 and later.
kCGBitmapByteOrder32Big

Specifies 32-bit, big endian format. 

Available in Mac OS X v10.4 and later.
kCGBitmapByteOrder16Host

Specifies 16-bit, host endian format.

Available in Mac OS X v10.4 and later.
kCGBitmapByteOrder32Host

Specifies 32-bit, host endian format. 

Available in Mac OS X v10.4 and later.
Discussion

Applications that store pixel data in memory using ARGB format must take care in how they read data. If the code is not written correctly, it’s possible to misread the data which leads to colors or alpha that appear wrong. The Quartz byte order constants specify the byte ordering of pixel formats. To specify byte ordering to Quartz use a bitwise OR operator to combine the appropriate constant with the bitmapInfo parameter.
Availability
Available in Mac OS X v10.4 and later.*/
enum {
   kCGBitmapAlphaInfoMask = 0x1F,
   kCGBitmapFloatComponents = (1 << 8),
   
   kCGBitmapByteOrderMask = 0x7000,
   kCGBitmapByteOrderDefault = (0 << 12),
   kCGBitmapByteOrder16Little = (1 << 12),
   kCGBitmapByteOrder32Little = (2 << 12),
   kCGBitmapByteOrder16Big = (3 << 12),
   kCGBitmapByteOrder32Big = (4 << 12)
};
typedef uint32_t CGBitmapInfo;


/*CGImageRef

Defines an opaque type that encapsulates bitmap image information.

typedef struct CGImage *CGImageRef;
Discussion

A bitmap image is a rectangular array of pixels, with each pixel representing a single point in an image. Quartz uses CGImage to represent bitmap images and bitmap image masks.

To create a Quartz image, you call CGImageCreate or CGImageMaskCreate with parameters that describe the precise format of the bitmap. To draw the image, you use CGContextDrawImage. In addition to the actual image data and information related to it, a CGImage may have a decode array, a CGColorRenderingIntent constant, and an interpolation setting.

For each color component in the image’s color space, a decode array provides a pair of values denoting the upper and lower limits of a range. For example, the decode array for a source image in the RGB color space would contain six entries total, consisting of one pair each for red, green, and blue. When the image is rendered, Quartz uses a linear transform to map the original component value into a relative number within your designated range that is appropriate for the destination color space.

A CGColorRenderingIntent constant specifies how Quartz should handle colors that are not located within the gamut of the destination color space of a graphics context. The rendering intent determines the exact method used to map colors from one color space to another. For descriptions of the defined rendering-intent constants, see CGColorRenderingIntent.

The interpolation setting is a Boolean value that specifies whether Quartz should apply a pixel-smoothing algorithm to the image. Without interpolation, the image may appear jagged or pixelated when drawn on an output device with higher resolution than the image data.

In Mac OS X version 10.2 and later, CGImageRef is derived from CFTypeRef and inherits the properties that all Core Foundation types have in common. For more information, see CFType Reference.
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct CGImage *CGImageRef;

/*CGImageRelease

Decrements the retain count of a Quartz image.

void CGImageRelease (
   CGImageRef image
);
Parameters
image

The image to release. 
Discussion

This function is equivalent to CFRelease, except that it does not cause an error if the image parameter is NULL.
Availability
Available in Mac OS X version 10.0 and later.
Declared In
CGImage.h*/
void CGImageRelease (
   CGImageRef image
);

/*CGImageRetain

Increments the retain count of a Quartz image.

CGImageRef CGImageRetain (
   CGImageRef image
);
Parameters
image

The image to retain. 
Return Value

The same image you passed in as the image parameter.
Discussion

This function is equivalent to CFRetain, except that it does not cause an error if the image parameter is NULL.
Availability
Available in Mac OS X version 10.0 and later.
Declared In
CGImage.h*/
CGImageRef CGImageRetain (
   CGImageRef image
);

/*CGImageCreate

Creates a Quartz bitmap image.

CGImageRef CGImageCreate (
   size_t width,
   size_t height,
   size_t bitsPerComponent,
   size_t bitsPerPixel,
   size_t bytesPerRow,
   CGColorSpaceRef colorspace,
   CGBitmapInfo bitmapInfo,
   CGDataProviderRef provider,
   const float decode[],
   int shouldInterpolate,
   CGColorRenderingIntent intent
);
Parameters
width

The width, in pixels, of the required image.
height

The height, in pixels, of the required image
bitsPerComponent

The number of bits for each component in a source pixel. For example, if the source image uses the RGBA-32 format, you would specify 8 bits per component.
bitsPerPixel

The total number of bits in a source pixel. This value must be at least bitsPerComponent times the number of components per pixel.
bytesPerRow

The number of bytes of memory for each horizontal row of the bitmap.
colorspace

The color space for the image. Quartz retains the color space you pass in; on return, you may safely release it.
bitmapInfo

A CGBitmapInfo constant that specifies whether the bitmap should contain an alpha channel and its relative location in a pixel, along with whether the components are floating-point or integer values. 
provider

The source of data for the bitmap. For information about supported data formats, see the discussion below. Quartz retains this object; on return, you may safely release it.
decode

The decode array for the image. If you do not want to allow remapping of the image’s color values, pass NULL for the decode array.
shouldInterpolate

A Boolean value that specifies whether interpolation should occur. The interpolation setting specifies whether Quartz should apply a pixel-smoothing algorithm to the image.
intent

A CGColorRenderingIntent constant that specifies how Quartz should handle colors that are not located within the gamut of the destination color space of a graphics context.
Return Value

A new Quartz bitmap image. You are responsible for releasing this object by calling CGImageRelease.
Discussion

The data provider should provide raw data that matches the format specified by the other input parameters. To use encoded data (for example, from a file specified by a URL-based data provider), see CGImageCreateWithJPEGDataProvider and CGImageCreateWithPNGDataProvider. In Mac OS X version 10.3 and later, you can also use the QuickTime function GraphicsImportCreateCGImage to decode an image file in any supported format and create a CGImage, in a single operation.
Availability
Available in Mac OS X version 10.0 and later.
Declared In
CGImage.h*/
CGImageRef CGImageCreate (
   size_t width,
   size_t height,
   size_t bitsPerComponent,
   size_t bitsPerPixel,
   size_t bytesPerRow,
   CGColorSpaceRef colorspace,
   CGBitmapInfo bitmapInfo,
   CGDataProviderRef provider,
   const float decode[],
   int shouldInterpolate,
   CGColorRenderingIntent intent
);

/*CGImageGetWidth

Returns the width of a bitmap image.

size_t CGImageGetWidth (
   CGImageRef image
);
Parameters
image

The image to examine. 
Return Value

The width, in pixels, of the specified bitmap image (or image mask).
Availability
Available in Mac OS X version 10.0 and later.
Declared In
CGImage.h*/
size_t CGImageGetWidth (
   CGImageRef image
);

/*CGImageGetHeight

Returns the height of a bitmap image.

size_t CGImageGetHeight (
   CGImageRef image
);
Parameters
image

The image to examine. 
Return Value

The height in pixels of the bitmap image (or image mask).
Availability
Available in Mac OS X version 10.0 and later.
Declared In
CGImage.h*/
size_t CGImageGetHeight (
   CGImageRef image
);

/*CGImageGetColorSpace

Return the color space for a bitmap image.

CGColorSpaceRef CGImageGetColorSpace (
   CGImageRef image
);
Parameters
image

The image to examine. 
Return Value

The source color space for the specified bitmap image, or NULL if the image is an image mask. You are responsible for retaining and releasing the color space as necessary.
Availability
Available in Mac OS X version 10.0 and later.
Declared In
CGImage.h*/
CGColorSpaceRef CGImageGetColorSpace (
   CGImageRef image
);
/*CGImageCreateWithPNGDataProvider

Creates a Quartz bitmap image using PNG-encoded data.

CGImageRef CGImageCreateWithPNGDataProvider (
   CGDataProviderRef source,
   const float decode[],
   bool shouldInterpolate,
   CGColorRenderingIntent intent
);
Parameters
source

A data provider supplying PNG-encoded data.
decode

The decode array for the image. Typically a decode array is unnecessary, and you should pass NULL.
shouldInterpolate

A Boolean value that specifies whether interpolation should occur. The interpolation setting specifies whether Quartz should apply a pixel-smoothing algorithm to the image.
intent

A CGColorRenderingIntent constant that specifies how Quartz should handle colors that are not located within the gamut of the destination color space of a graphics context.
Return Value

A new Quartz bitmap image. You are responsible for releasing this object by calling CGImageRelease.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGImage.h*/

CGImageRef CGImageCreateWithPNGDataProvider (
   CGDataProviderRef source,
   const float decode[],
   bool shouldInterpolate,
   CGColorRenderingIntent intent
);

#endif