#include <asl.h>

/*asl(3)                   BSD Library Functions Manual                   asl(3)

NAME
     asl_open, asl_close, asl_new, asl_free, asl_set, asl_set_query, asl_get,
     asl_unset, asl_log, asl_vlog, asl_send, asl_key, asl_add_log_file,
     asl_remove_log_file, asl_set_cutoff_level, asl_search, aslresponse_next,
     aslresponse_free -- system log message sending and searching functions

SYNOPSIS
     #include <asl.h>

     aslclient
     asl_open(const char *ident, const char *facility, uint32_t opts);

     void
     asl_close(aslclient asl);

     aslmsg
     asl_new(uint32_t type);

     void
     asl_free(aslmsg msg);

     int
     asl_set(aslmsg msg, const char *key, const char *value);

     int
     asl_set_query(aslmsg msg, const char *key, const char *value, uint32_t op);

     const char *
     asl_key(aslmsg msg, uint32_t n);

     const char *
     asl_get(aslmsg msg, const char *key);

     int
     asl_unset(aslmsg msg, const char *key);

     int
     asl_log(aslclient asl, aslmsg msg, int level, const char *format, ...);

     int
     asl_vlog(aslclient asl, aslmsg msg, int level, const char *format, va_list ap);

     int
     asl_send(aslclient asl, aslmsg msg);

     int
     asl_add_log_file(aslclient asl, int fd);

     int
     asl_remove_log_file(aslclient asl, int fd);

     int
     asl_set_filter(aslclient asl, int f);

     aslresponse
     asl_search(aslclient asl, aslmsg msg);

     aslmsg
     aslresponse_next(aslresponse r);

     void
     aslresponse_free(aslresponse a);

DESCRIPTION
     These routines provide an interface to the Apple system log facility.
     They are intended to be a replacement for the syslog(3) API, which will
     continue to be supported for backwards compatibility.  The new API allows
     client applications to create flexible, structured messages and send them
     to the syslogd server, where they may undergo additional processing.
     Messages received by the server are saved in a data store (subject to
     input filtering constraints).  This API permits clients to create queries
     and search the message data store for matching messages.

   MESSAGES
     At the core of this API is the aslmsg structure.  Although the structure
     is opaque and may not be directly manipulated, it contains a list of
     key/value pairs.  All keys and values are NULL-terminated C language
     character strings.  UTF-8 encoding may be used for non-ASCII characters.

     Message structures are generally used to send log messages, and are cre-ated created
     ated thusly:

         aslmsg m = asl_new(ASL_TYPE_MSG);

     Another message type, ASL_TYPE_QUERY, is used to create queries when
     searching the data store.  Query type messages and searching are
     described in detail in the SEARCHING section below.  For the remainder of
     this section, the messages described will be of the ASL_TYPE_MSG variety.

     Each aslmsg contains a default set of keys and values associated with
     them.  These keys are listed in the asl.h header file.  They are:

         #define ASL_KEY_TIME    "Time"
         #define ASL_KEY_HOST    "Host"
         #define ASL_KEY_SENDER  "Sender"
         #define ASL_KEY_PID     "PID"
         #define ASL_KEY_UID     "UID"
         #define ASL_KEY_GID     "GID"
         #define ASL_KEY_LEVEL   "Level"
         #define ASL_KEY_MSG     "Message"

     Many of these correspond to equivalent parts of messages described in the
     syslog(3) API.  Values associated with these message keys are assigned
     appropriate defaults.  The value for ASL_KEY_HOST is the local host name,
     the value associated with ASL_KEY_SENDER is the process name, the
     ASL_KEY_PID is the client's process ID number, and so on.

     Note the addition of the UID and GID keys.  The values for UID and GID
     are set in library code by the message sender.  The server will attempt
     to confirm the values, but no claim is made that these values cannot be
     maliciously overridden in an attempt to deceive a log message reader as
     to the identity of the sender of a message.  The contents of log messages
     must be regarded as insecure.

     Also note the absence of a Facility key.  The asl(3) API does not require
     a process to choose a facility name.  The syslogd server will use a
     default value of ``user'' if a facility is not set.  However, a client
     may set a facility name using:

         asl_set(m, "Facility", "UsefulService");

     An application may choose any facility name at will.

     Default values are set in the message for each of the keys listed above
     except for ASL_KEY_MSG, which may be explicitly set at any time using the
     asl_set routine, or implicitly set at the time the message is sent using
     the asl_log or asl_vlog routines.  These two routines also have an inte-ger integer
     ger level parameter for specifying the log priority.  The ASL_KEY_LEVEL
     value is set accordingly.  Finally, the value associated with
     ASL_KEY_TIME is set in the sending routine.

     Although it may appear that there is significant overhead required to
     send a log message using this API, the opposite is actually true.  A sim-ple simple
     ple ``Hello World'' program requires only:

         #include <asl.h>
         ...
         asl_log(NULL, NULL, ASL_LEVEL_INFO, "Hello World!");

     Both asl_log and asl_vlog will provide the appropriate default values
     when passed a NULL aslmsg argument.

     In this example, the aslclient argument is NULL.  This is sufficient for
     a single-threaded application, or for an application which only sends log
     messages from a single thread.  When logging from multiple threads, each
     thread must open a separate client handle using asl_open.  The client
     handle may then be closed when it is no longer required using asl_close.

     When an application requires additional keys and values to be associated
     with each log message, a single message structure may be allocated and
     set up as ``template'' message of sorts:

         aslmsg m = asl_new(ASL_TYPE_MSG);
         asl_set(m, "Facility", "Spy vs. Spy");
         asl_set(m, "Clearance", "Top Secret");
         ...
         asl_log(NULL, m, ASL_LEVEL_NOTICE, "Message One");
         ...
         asl_log(NULL, m, ASL_LEVEL_ERR, "Message Two");

     The message structure will carry the values set for the ``Facility'' and
     ``Clearance'' keys so that they are used in each call to asl_log, while
     the log level and the message text are taken from the calling parameters.

     Key/value pairs may be removed from a message structure with asl_unset.
     A message may be freed using asl_free.

     The asl_send routine is used by asl_log and asl_vlog to transmit a mes-sage message
     sage to the server.  This routine sets the value associated with
     ASL_KEY_TIME and send the message.  It may be called directly if all of a
     message's key/value pairs have been created using asl_set.

   CLIENT HANDLES
     When logging is done from a single thread, a NULL value may be used in
     any of the routines that require an aslclient argument.  In this case the
     library will open an internal client handle on behalf of the application.

     If multiple threads must do logging, or if client options are desired,
     then the application should call asl_open to create a client handle for
     each thread.  As a convenience, the asl_open routine may be given an
     ident argument, which becomes the default value for the ASL_KEY_SENDER
     key, and a facility argument, which becomes the default facility name for
     the application.

     Several options are available when creating a client handle.  They are:

     ASL_OPT_STDERR     adds stderr as an output file descriptor
     ASL_OPT_NO_DELAY   connects to the server immediately
     ASL_OPT_NO_REMOTE  disables remote-control filter adjustment

     See the FILTERING section below, and the syslog(1) for additional details
     on filter controls.

     A client handle is closed and it's resources released using asl_close.
     Note that if additional file descriptors were added to the handle either
     using the ASL_OPT_STDERR option or afterwards with the asl_add_log_file
     routine, those file descriptors are not closed by asl_close.

   LOGGING TO ADDITIONAL FILES
     If a client handle is opened with the ASL_OPT_STDERR option to asl_open,
     a copy of each log message will be sent to stderr.  Additional output
     streams may be include using asl_add_log_file.  File descriptors may be
     removed from the list of outputs associated with a client handle with
     asl_remove_log_file.  This routine simply removes the file descriptor
     from the output list.  The file is not closed as a result.

     The ASL_OPT_STDERR option may not be unset after a client handle has been
     opened.

     In the present release of Mac OS X, a ``raw'' format is used to format
     messages sent to file descriptors added to a client handle.  Each message
     is preceded by a 10-character field containing a message length.  The
     message length is padded with leading white space.  The length gives the
     string length of the remainder of the output string.  Following the
     length is a space character, and then the message.  The message is
     encoded as a set of key/value pairs enclosed in square brackets, which
     are themselves separated by a space character.  The key is separated from
     the value by space character.  Embedded closing square brackets are
     escaped by a backslash.  Embedded space characters in keys are escaped by
     a backslash; Embedded newlines are summarily turned into semicolons.  The
     output is terminated by a trailing newline and a NUL character.

   SEARCHING
     The syslogd server archives received messages in a data store that may be
     searched using the asl_search, aslresponse_next, and aslresponse_free
     routines.  A query message is created using:

         aslmsg q = asl_new(ASL_TYPE_QUERY);

     Search settings are made in the query using asl_set_query.  A search is
     performed on the data store with asl_search.  It returns an aslresponse
     structure.  The caller may then call aslresponse_next to iterate through
     matching messages.  The aslresponse structure may be freed with
     aslresponse_free.

     Like other messages, ASL_TYPE_QUERY messages contain keys and values.
     They also associate an operation with each key and value.  The operation
     is used to decide if a message matches the query.  The simplest operation
     is ASL_QUERY_OP_EQUAL, which tests for equality.  For example, the fol-lowing following
     lowing code snippet searches for messages with a Sender value equal to
     ``MyApp''.

         aslmsg m;
         aslresponse r;
         q = asl_new(ASL_TYPE_QUERY);
         asl_set_query(q, ASL_KEY_SENDER, "MyApp", ASL_QUERY_OP_EQUAL);
         r = asl_search(NULL, q);

     More complex searches may be performed using other query operations.

     ASL_QUERY_OP_EQUAL          value equality
     ASL_QUERY_OP_GREATER        value greater than
     ASL_QUERY_OP_GREATER_EQUAL  value greater than or equal to
     ASL_QUERY_OP_LESS           value less than
     ASL_QUERY_OP_LESS_EQUAL     value less than or equal to
     ASL_QUERY_OP_NOT_EQUAL      value not equal
     ASL_QUERY_OP_REGEX          regular expression search
     ASL_QUERY_OP_TRUE           always true - use to test for the existence
                                 of a key

     Regular expression search uses regex(3) library.  Patterns are compiled
     using the REG_EXTENDED and REG_NOSUB options.

     Modifiers that change the behavior of these operations may also be speci-fied specified
     fied by ORing the modifier value with the operation.  The modifiers are:

     ASL_QUERY_OP_CASEFOLD   string comparisons are case-folded
     ASL_QUERY_OP_PREFIX     match a leading substring
     ASL_QUERY_OP_SUFFIX     match a trailing substring
     ASL_QUERY_OP_SUBSTRING  match any substring
     ASL_QUERY_OP_NUMERIC    values are converted to integer using atoi

     The only modifier that is checked for ASL_QUERY_OP_REGEX search is
     ASL_QUERY_OP_CASEFOLD.  This causes the regular expression to be compiled
     with the REG_ICASE option.

     If a query message contains more than one set of key/value/operation
     triples, the result will be a logical AND.  For example, to find messages
     from ``MyApp'' with a priority level less than or equal to ``3'':

         aslmsg q;
         aslresponse r;
         q = asl_new(ASL_TYPE_QUERY);
         asl_set_query(q, ASL_KEY_SENDER, "MyApp", ASL_QUERY_OP_EQUAL);
         asl_set_query(q, ASL_KEY_LEVEL, "3",
                 ASL_QUERY_OP_LESS_EQUAL | ASL_QUERY_OP_NUMERIC);
         r = asl_search(NULL, q);

     After calling asl_search to get an aslresponse structure, use
     aslresponse_next to iterate through all matching messages.  To iterate
     through the keys and values in a message, use asl_key to iterate through
     the keys, then call asl_get to get the value associated with each key.

         aslmsg q, m;
         int i;
         const char *key, *val;

         ...
         r = asl_search(NULL, q);
         while (NULL != (m = aslresponse_next(r)))
         {
             for (i = 0; (NULL != (key = asl_key(m, i))); i++)
             {
                 val = asl_get(m, key);
                 ...
             }
         }
         aslresponse_free(r);

   FILTERING AND REMOTE CONTROL
     Clients may set a filter mask value with asl_set_filter.  The mask speci-fies specifies
     fies which messages should be sent to the syslogd daemon by specifying a
     yes/no setting for each priority level.  Clients typically set a filter
     mask to avoid sending relatively unimportant messages.  For example,
     Debug or Info priority level messages are generally only useful for
     debugging operations.  By setting a filter mask, a process can improve
     performance by avoiding sending messages that are in most cases unneces-sary. unnecessary.
     sary.

     As a convenience, the macros ASL_FILTER_MASK(level) and ASL_FIL-TER_MASK_UPTO(level) ASL_FILTER_MASK_UPTO(level)
     TER_MASK_UPTO(level) may be used to construct a bit mask corresponding to
     a given priority level, or corresponding to a bit mask for all priority
     levels from ASL_LEVEL_EMERG to a given input level.

     The default filter mask is ASL_FILTER_MASK_UPTO(ASL_LEVEL_NOTICE).  This
     means that by default, and in the absence of remote-control changes
     (described below), ASL_LEVEL_DEBUG and ASL_LEVEL_INFO priority level mes-sages messages
     sages are not sent to the server.

     Three different filters exist for each application.  The first is the
     filter mask set using asl_set_filter as described above.  The Apple Sys-tem System
     tem Log facility also manages a ``master'' filter mask.  The master fil-ter filter
     ter mask usually has a value that indicates to the library that it is
     ``off'', and thus it has no effect.  However, the mask filter mask may be
     enabled by giving it a value using the syslog command, using the -c 0
     option.  When the master filter mask has been set, it takes precedence
     over the client's filter mask.  The client's mask is unmodified, and will
     become active again if remote-control filtering is disabled.

     In addition to the master filter mask, The Apple System Log facility also
     manages a per-client remote-control filter mask.  Like the master filter
     mask, the per-client mask is usually ``off'', having no effect on a
     client.  If a per-client filter mask is set using the syslog command,
     using the -c process option, then it takes precedence over both the
     client's filter mask and the master filter mask.  As is the case with the
     master filter mask, a per-client mask ceases having any effect when if is
     disabled.

     The ASL_OPT_NO_REMOTE option to asl_open causes both the master and per-client perclient
     client remote-control masks to be ignored in the library.  In that case,
     only the client's own filter mask is used to determine which messages are
     sent to the server.  This may be useful for Applications that produce log
     messages that should never be filtered due to security considerations.
     Note that root (administrator) access is required to set or change the
     master filter mask, and that only root may change a per-client remote-control remotecontrol
     control filter mask for a root (UID 0) process.

HISTORY
     These functions first appeared in Mac OS X 10.4.

SEE ALSO
     syslogd(8), syslog(1)

Mac OS X                        January 5, 2005                       Mac OS X*/
aslclient
     asl_open(const char *ident, const char *facility, uint32_t opts){};

     aslmsg
     asl_new(uint32_t type){};


 int
     asl_set(aslmsg msg, const char *key, const char *value){};


   int
     asl_log(aslclient asl, aslmsg msg, int level, const char *format, ...){};


     void
     asl_close(aslclient asl){};

     int
     asl_unset(aslmsg msg, const char *key){};

  int
     asl_set_query(aslmsg msg, const char *key, const char *value, uint32_t op){};

     aslresponse
     asl_search(aslclient asl, aslmsg msg){};


     aslmsg
     aslresponse_next(aslresponse r){};

 void
     asl_free(aslmsg msg){};


     void
     aslresponse_free(aslresponse a){};

    const char *
     asl_get(aslmsg msg, const char *key){};






