#ifndef CGDISPLAY_FADE_H
#define CGDISPLAY_FADE_H

#include <CoreGraphics/CGDirectDisplay.h>
#include <CoreGraphics/CGDisplayConfiguration.h>

/*CGDisplayFadeInterval

Represents the duration in seconds of a fade operation or a fade hardware reservation.

typedef float CGDisplayFadeInterval;
Discussion

Quartz uses this data type to specify the duration of both fade-out and fade-in operations. Values may range from zero to kCGMaxDisplayReservationInterval seconds. A zero value means fade immediately—see CGDisplayFade.
Availability
Available in Mac OS X v10.2 and later.*/
typedef float CGDisplayFadeInterval;

/*CGDisplayFadeReservationToken

Defines a token issued by Quartz when reserving one or more displays for a fade operation during a specified interval.

typedef uint32_t CGDisplayFadeReservationToken;

Discussion

Quartz lets you reserve the display hardware to perform a fade operation. Fade reservations are valid for up to 15 seconds. Only one token is needed for both fade-out and fade-in.

You should release a fade reservation immediately when you no longer need it. If the reservation expires, releasing it is safe but not necessary.
Availability
Available in Mac OS X v10.2 and later.*/
typedef uint32_t CGDisplayFadeReservationToken;

/*CGDisplayBlendFraction

Represents the percentage of blend color used in a fade operation.

typedef float CGDisplayBlendFraction;

Discussion

The blend fraction ranges from 0 (no color) to 1 (full intensity). If you specify 0, the blend color is not applied. If you specify 1, the user sees only the blend color on the screen.

In a fade operation, Quartz blends a color specified by the application with the current contents of the frame buffer. The blend color can be applied both at the beginning and the end of a fade operation.

Color blending during a fade operation is analagous to alpha blending in Quartz 2D, and the visual appearance is similar. However, the implementation is quite different. In a fade operation, the blend color is applied at the very end of the graphics pipeline, as the frame buffer is transferred to video output.

For example, the Universal Access preference panel in Mac OS X allows you to select a flashing screen effect (sometimes called a visual bell) to accompany the system alert sound. When you select this option, the system uses a Quartz fade operation to produce the flash. The blend color is applied using a blend fraction of 0.5 or 50%.
Availability
Available in Mac OS X v10.2 and later.*/
typedef float CGDisplayBlendFraction;

/*CGDisplayReservationInterval

Represents the time interval for a fade reservation.

typedef float CGDisplayReservationInterval;
Discussion

A fade reservation interval is a period of time during which a specific display is reserved for a fade operation. Fade reservation intervals range from 1 to kCGMaxDisplayReservationInterval seconds.

For more information about fade reservations, see the function CGAcquireDisplayFadeReservation. Fade reservation tokens are discussed in CGDisplayFadeReservationToken.
Availability
Available in Mac OS X v10.2 and later.*/
typedef float CGDisplayReservationInterval;

/*Display Fade Interval Constant

Specifies the maximum number of seconds for fade hardware reservations and display fade operations.

#define kCGMaxDisplayReservationInterval (15.0)
Discussion

For general information about fade intervals, see the data type CGDisplayFadeInterval.*/
#define kCGMaxDisplayReservationInterval (15.0)

/*Display Fade Blend Fractions

Specify the lower and upper bounds for blend color fractions during a display fade operation.

#define kCGDisplayBlendNormal (0.0)
#define kCGDisplayBlendSolidColor (1.0)

Constants
kCGDisplayBlendNormal

Specifies that the blend color is not applied at the start or end of a fade operation.

Available in Mac OS X v10.2 and later.
kCGDisplayBlendSolidColor

Specifies that the user sees only the blend color at the start or end of a fade operation.

Available in Mac OS X v10.2 and later.
Discussion

For general information about blend fractions, see the data type CGDisplayBlendFraction. For information about how these constants are used, see the function CGDisplayFade.*/
#define kCGDisplayBlendNormal (0.0)
#define kCGDisplayBlendSolidColor (1.0)


//func

/*CGDisplayFade

Performs a single fade operation. 

CGError CGDisplayFade (
   CGDisplayFadeReservationToken token,
   CGDisplayFadeInterval seconds,
   CGDisplayBlendFraction startBlend,
   CGDisplayBlendFraction endBlend,
   float redBlend,
   float greenBlend,
   float blueBlend,
   boolean_t synchronous
);

Parameters
token

A valid reservation token for the fade hardware, acquired by calling CGAcquireDisplayFadeReservation.
seconds

The desired number of seconds for the fade operation. You should use a value in the interval [0, kCGMaxDisplayReservationInterval]. If the value is 0, the ending blend color is applied immediately.
startBlend

An intensity value in the interval [0, 1] that specifies the alpha component of the desired blend color at the beginning of the fade operation. See “Display Fade Blend Fractions”.
endBlend

An intensity value in the interval [0, 1] that specifies the alpha component of the desired blend color at the end of the fade operation. See “Display Fade Blend Fractions”.
redBlend

An intensity value in the interval [0, 1] that specifies the red component of the desired blend color.
greenBlend

An intensity value in the interval [0, 1] that specifies the green component of the desired blend color.
blueBlend

An intensity value in the interval [0, 1] that specifies the blue component of the desired blend color.
synchronous

Pass true if you want the fade operation to be synchronous; otherwise, pass false. If a fade operation is synchronous, the function does not return until the operation is complete.
Return Value

A result code. See “Quartz Display Services Result Codes”. 
Discussion

Over the fade operation time interval, Quartz interpolates a blending coefficient between the starting and ending values given, applying a nonlinear (sine-based) bias term. Using this coefficient, the video output is blended with the specified color.

The following example shows how to perform a two-second synchronous fade-out to black: CGDisplayFade (
    myToken,
    2.0,                        // 2 seconds
    kCGDisplayBlendNormal,      // starting state
    kCGDisplayBlendSolidColor,  // ending state
    0.0, 0.0, 0.0,              // black
    true                        // wait for completion
);


To perform a two-second asynchronous fade-in from black: CGDisplayFade (
    myToken,
    2.0,                        // 2 seconds
    kCGDisplayBlendSolidColor,  // starting state
    kCGDisplayBlendNormal,      // ending state
    0.0, 0.0, 0.0,              // black
    false                       // don't wait for completion
);


If you specify an asynchronous fade operation, it’s safe to call CGReleaseDisplayFadeReservation immediately after this function returns.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGDisplayFade.h*/
CGError CGDisplayFade (
   CGDisplayFadeReservationToken token,
   CGDisplayFadeInterval seconds,
   CGDisplayBlendFraction startBlend,
   CGDisplayBlendFraction endBlend,
   float redBlend,
   float greenBlend,
   float blueBlend,
   boolean_t synchronous
);

/*CGReleaseDisplayFadeReservation

Releases a display fade reservation, and unfades the display if needed.

CGError CGReleaseDisplayFadeReservation (
   CGDisplayFadeReservationToken token
);
Parameters
token

The current fade reservation token to be released. On return, the reservation token is no longer valid and should be discarded.
Return Value

A result code. See “Quartz Display Services Result Codes”. 
Discussion

If you call this function while an asynchronous fade operation is running, there are two possible outcomes:
If the ending blend value is kCGDisplayBlendNormal, the fade operation is allowed to run to completion.
If the ending blend value is not kCGDisplayBlendNormal, the fade operation is terminated immediately and the display is returned to normal.

In both cases, the reservation is actually released when the fade operation completes.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGDisplayFade.h*/
CGError CGReleaseDisplayFadeReservation (
   CGDisplayFadeReservationToken token
);

/*CGAcquireDisplayFadeReservation

Reserves the fade hardware for a specified time interval.

CGError CGAcquireDisplayFadeReservation (
   CGDisplayReservationInterval seconds,
   CGDisplayFadeReservationToken * tokenPtr
);

Parameters
seconds

The desired number of seconds to reserve the fade hardware. An application can specify any value in the interval (0, kCGMaxDisplayReservationInterval].
tokenPtr

A pointer to storage (provided by the caller) for a fade reservation token. On return, the storage contains a new token.
Return Value

Returns kCGErrorNoneAvailable if another fade reservation is in effect. Otherwise, returns kCGErrorSuccess. 
Discussion

Before performing a fade operation, an application must reserve the fade hardware for a specified period of time. Quartz returns a token that represents a new fade reservation. The application uses this token as an argument in subsequent calls to other display fade functions. 

During the fade reservation interval, the application has exclusive rights to use the fade hardware. At the end of the interval, the token becomes invalid and the hardware automatically returns to a normal state. Typically the application calls CGReleaseDisplayFadeReservation to release the fade reservation before it expires.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGDisplayFade.h*/

CGError CGAcquireDisplayFadeReservation (
   CGDisplayReservationInterval seconds,
   CGDisplayFadeReservationToken * tokenPtr
);

#endif