#ifndef  PASTEBOARD_H
#define PASTEBOARD_H

#ifndef  COREFOUNDATION_H
#include <CoreFoundation/CoreFoundation.h>
#endif


#include <AvailabilityMacros.h>
#include <APPLE/HIServices/Pasteboard.h>
#if PRAGMA_ONCE
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*Pasteboard Manager. Result Codes

The table below lists the most common error codes returned by the Pasteboard Manager. 
Result Code	Value	Description
badPasteboardSyncErr	-25130	
The pasteboard has been modified and must be synchronized before use. 
badPasteboardIndexErr	-25131	
The specified pasteboard item index does not exist.
badPasteboardItemErr	-25132	
The item reference does not exist.
badPasteboardFlavorErr	-25133	
The item flavor does not exist. 
duplicatePasteboardFlavorErr	-25134	
The item flavor already exists.
notPasteboardOwnerErr	-25135	
The application did not clear the pasteboard before attempting to add flavor data. 
noPasteboardPromiseKeeperErr	-25136	
The application attempted to add promised data without previously registering a promise keeper callback*/
enum {
badPasteboardSyncErr	=-25130	,
//The pasteboard has been modified and must be synchronized before use. 
badPasteboardIndexErr	=-25131	,
//The specified pasteboard item index does not exist.
badPasteboardItemErr	=-25132	,
//The item reference does not exist.
badPasteboardFlavorErr	=-25133	,
//The item flavor does not exist. a
duplicatePasteboardFlavorErr	=-25134	,
//The item flavor already exists.
notPasteboardOwnerErr	=-25135	,
//The application did not clear the pasteboard before attempting to add flavor data. 
noPasteboardPromiseKeeperErr	=-25136	,
//The application attempted to add promised data without previously registering a promise keeper
};


/*Pasteboard Flavor Flags

Indicate useful information associated with pasteboard item flavors. 

enum {
kPasteboardFlavorNoFlags      = 0,
kPasteboardFlavorSenderOnly   = (1 << 0),
kPasteboardFlavorSenderTranslated = (1 << 1),
kPasteboardFlavorNotSaved     = (1 << 2),
kPasteboardFlavorRequestOnly  = (1 << 3),
kPasteboardFlavorSystemTranslated = (1 << 8),
kPasteboardFlavorPromised     = (1 << 9)
};
typedef UInt32 PasteboardFlavorFlags;
Constants
kPasteboardFlavorNoFlags

No flag information exists for this flavor.

Available in Mac OS X v10.3 and later.
kPasteboardFlavorSenderOnly

Only the process that added this flavor can see it. Typically used to tag proprietary data that can be cut, dragged, or pasted only within the owning application. This flag is identical to the Drag Manager flavorSenderOnly flag.

Available in Mac OS X v10.3 and later.
kPasteboardFlavorSenderTranslated

The sender translated this data in some fashion before adding it to the pasteboard. The Finder cannot store flavor items marked with this flag in clipping files. This flag is identical to the Drag Manager flavorSenderTranslated flag.

Available in Mac OS X v10.3 and later.
kPasteboardFlavorNotSaved

The receiver should not save the data provided for this flavor. The data may become stale after a drag, or the flavor may be used only to augment another flavor. For example, an application may add a flavor whose only purpose is to supply additional information (say text style) about another flavor. The Finder cannot store flavor items marked with this flag in clipping files. This flag is identical to the Drag Manager flavorNotSaved flag.

Available in Mac OS X v10.3 and later.
PasteboardFlavorRequestOnly

When the sender sets this flag, this flavor is hidden from calls to PasteboardCopyItemFlavors. However, you can obtain the flavor flags and data by calling PasteboardGetItemFlavorFlags or PasteboardCopyItemFlavorData. The net result is that applications cannot obtain this flavor unless they are already aware it exists and specifically request it. This functionality can be useful for copying and pasting proprietary data within a suite of applications.
PasteboardFlavorSystemTranslated

This data flavor is available through the Translation Manager. That is, the Translation Manager must translate the sender’s data before the receiver can receive it. This flag is set automatically when appropriate and cannot be set programmatically. The Finder cannot store flavor items marked with this flag in clipping files. This flag is identical to the Drag Manager flavorSystemTranslated flag. 
kPasteboardFlavorPromised

The data associated with this flavor is not yet on the pasteboard. Typically data is promised to the pasteboard if it requires a lot of overhead to generate. If the receiver requests the data, the sender is notified so it can supply the promised data. This flag is set automatically when appropriate and cannot be set programmatically. This flag is identical to the Drag Manager flavorDataPromised flag.

Available in Mac OS X v10.3 and later.*/

enum {
kPasteboardFlavorNoFlags      = 0,
kPasteboardFlavorSenderOnly   = (1 << 0),
kPasteboardFlavorSenderTranslated = (1 << 1),
kPasteboardFlavorNotSaved     = (1 << 2),
kPasteboardFlavorRequestOnly  = (1 << 3),
kPasteboardFlavorSystemTranslated = (1 << 8),
kPasteboardFlavorPromised     = (1 << 9)
};
typedef UInt32 PasteboardFlavorFlags;

/*Pasteboard Synchronization Flags

Indicate the pasteboard status after a call to PasteboardSynchronize.

enum {
kPasteboardModified           = (1 << 0),
kPasteboardClientIsOwner      = (1 << 1)
};
typedef UInt32 PasteboardSyncFlags;
Constants
kPasteboardModifed

The pasteboard was modified since the last time the application accessed it, and the local pasteboard has been synchronized to reflect any changes. Your application should check to see what flavors are now available on the pasteboard and update appropriately (for example, enabling the Paste menu item). 
kPasteboardClientIsOwner

The application recently cleared the pasteboard and is its current owner. The application can add flavor data to the pasteboard as desired. 

Available in Mac OS X v10.3 and later.*/

enum {
kPasteboardModified           = (1 << 0),
kPasteboardClientIsOwner      = (1 << 1)
};
typedef UInt32 PasteboardSyncFlags;


/*Pasteboard Type Constants

Define common pasteboard types.

#define kPasteboardClipboard CFSTR("com.apple.pasteboard.clipboard")
#define kPasteboardFind      CFSTR("com.apple.pasteboard.find")
#define kPasteboardUniqueName(CFStringRef)NULL
Constants
kPasteboardClipboard

The standard clipboard (used for copy and paste). 

Available in Mac OS X v10.3 and later.
kPasteboardFind

The Find pasteboard (used for search fields).

Available in Mac OS X v10.3 and later.
kPasteboardUniqueName

A system-declared pasteboard that is guaranteed to be unique. Each ttime you call PasteboardCreate with this constant, you create a different, unique pasteboard. 

Available in Mac OS X v10.3 and later.
Discussion

You pass these constants in PasteboardCreate when creating a pasteboard reference.*/

#define kPasteboardClipboard CFSTR("com.apple.pasteboard.clipboard")
#define kPasteboardFind      CFSTR("com.apple.pasteboard.find")
#define kPasteboardUniqueName (CFStringRef)NULL


/*PasteboardItemID

Defines a pasteboard item identifier.

typedef void* PasteboardItemID;
Discussion

You can use any arbitrary (but unique) ID to identify your pasteboard items when placing them on a pasteboard. For example, you may want to identify items by their internal memory address. Only the owning application should interpret its pasteboard item identifiers. 

For example, when your application’s promise keeper callback function is invoked, it receives a pasteboard item ID, and your application must be able to map that ID to the corresponding promised data. 
Availability
Available in Mac OS X v10.3 and later.*/
typedef void* PasteboardItemID;

/*PasteboardRef

Defines a reference to a global pasteboard.

typedef struct OpaquePasteboardRef *PasteboardRef;
Discussion

The structure pointed to by this reference is opaque.
Availability
Available in Mac OS X v10.3 and later.*/
typedef struct OpaquePasteboardRef *PasteboardRef;

/*PasteboardPromiseKeeperProcPtr

Defines a pointer to a callback function that supplies the actual data promised on the pasteboard. 

typedef OSStatus (*PasteboardPromiseKeeperProcPtr)
(
PasteboardRef pasteboard,
PasteboardItemID item,
CFStringRef flavorType,
void* context
);


You would declare your promise keeper callback function named MyPromiseKeeper like this:

OSStatus MyPromiseKeeper (
PasteboardRef pasteboard,
PasteboardItemID item,
CFStringRef flavorType,
void* context
);
Parameters
pasteboard

The pasteboard whose promise you need to fulfill.
item

The pasteboard item identifier containing the promised flavor. 
flavorType

The flavor of the data being requested, specified as a uniform type identifier.
context

The pointer you passed to PasteboardSetPromiseKeeper in the inContext parameter. 
Discussion

When promising any flavor data on a pasteboard using PasteboardPutItemFlavor, you must implement a callback function to fulfill that promise. 

If your application delivers the promised data as a file, your callback should call PasteboardCopyPasteLocation to determine where to deliver the requested data and then take the necessary steps to do so.*/
typedef OSStatus (*PasteboardPromiseKeeperProcPtr)
(
PasteboardRef pasteboard,
PasteboardItemID item,
CFStringRef flavorType,
void* context
);



//function
/*PasteboardGetItemIdentifier

Obtains the item identifier for an item in a pasteboard.

OSStatus PasteboardGetItemIdentifier (
PasteboardRef inPasteboard,
UInt32 inIndex,
PasteboardItemID *outItem
Parameters
inPasteboard

The pasteboard containing the data. 
inIndex

The one-based index number of the data item whose identifier you want to obtain.
outItem

On return, outItem points to the item identifier for the data item at index inIndex.
Return Value

A result code. 
Discussion

The item index is one-based to match the convention used by the Drag Manager.
Availability
Available in Mac OS X v10.3 and later. 
Declared In
Pasteboard.h*/
OSStatus PasteboardGetItemIdentifier (
PasteboardRef inPasteboard,
UInt32 inIndex,
PasteboardItemID *outItem);

/*PasteboardCopyItemFlavorData

Obtains data from a pasteboard for the desired flavor.

OSStatus PasteboardCopyItemFlavorData (
PasteboardRef inPasteboard,
PasteboardItemID inItem,
CFStringRef inFlavorType,
CFDataRef *outData
);
Parameters
inPasteboard

The pasteboard containing the data.
inItem

The identifier for the item whose flavor data you want to obtain. 
inFlavorType

The flavor of the data you want to obtain, specified as a uniform type identifier. 
outData

On return, outData points to the flavor data. You must release this data using CFRelease when you are done using it. 
Return Value

A result code.
Availability
Available in Mac OS X v10.3 and later. 
Declared In
Pasteboard.h*/
OSStatus PasteboardCopyItemFlavorData (
PasteboardRef inPasteboard,
PasteboardItemID inItem,
CFStringRef inFlavorType,
CFDataRef *outData
);

/*PasteboardGetItemCount

Obtains the number of data items in the specified pasteboard.

OSStatus PasteboardGetItemCount (
   PasteboardRef inPasteboard,
   ItemCount *outItemCount
);
Parameters
inPasteboard

The pasteboard whose items you want to count.
outItemCount

On return, outItemCount points to the number of items in the pasteboard.
Return Value

A result code.
Discussion

You usually call this function before calling PasteboardGetItemIdentifier.
Availability
Available in Mac OS X v10.3 and later. 
Declared In
Pasteboard.h*/
OSStatus PasteboardGetItemCount (
   PasteboardRef inPasteboard,
   ItemCount *outItemCount
);

/*PasteboardCopyItemFlavors

Obtains an array of flavors for a specified item in a pasteboard. 

OSStatus PasteboardCopyItemFlavors (
PasteboardRef inPasteboard,
PasteboardItemID inItem,
CFArrayRef *outFlavorTypes
);
Parameters
inPasteboard

The pasteboard containing the data.
inItem

The identifier for the item whose flavors you want to obtain. 
outFlavorTypes

On return, outFlavorTypes points to an array of flavors, specified as uniform type identifiers. You must release this array by calling CFRelease when you are done using it. 
Return Value

A result code. 
Availability
Available in Mac OS X v10.3 and later. 
Declared In
Pasteboard.h*/
OSStatus PasteboardCopyItemFlavors (
PasteboardRef inPasteboard,
PasteboardItemID inItem,
CFArrayRef *outFlavorTypes
);

/*PasteboardCreate

Creates a reference to the specified global pasteboard.

OSStatus PasteboardCreate (
CFStringRef inName
PasteboardRef *outPasteboard
);
Parameters
inName

The name of the pasteboard to reference. 
outPasteboard

On return, outPasteboard points to the specified pasteboard. 
Return Value

A result code.
Discussion

If the pasteboard you specify does not yet exist, this function creates it. You can use this function to create your own pasteboards to store and pass data. Pasteboard names should have a reverse DNS-style name to ensure uniqueness. 

You can also specify kPasteboardFind to obtain a reference to the Find pasteboard or kPasteboardClipboard for a reference to the standard clipboard. If you pass kPasteboardUniqueName (= NULL) , the system creates a unique pasteboard for your application.

When you are finished with a pasteboard reference, you can call CFRelease to release it. If you do not release the reference, the pasteboard continues to exist even after your application terminates. A subsequently launched application can then find a previously created pasteboard by name and examine the data within it. 
Availability
Available in Mac OS X v10.3 and later. 
Declared In
Pasteboard.h*/

OSStatus PasteboardCreate (
CFStringRef inName,
PasteboardRef *outPasteboard
);

/*PasteboardClear

Clears the contents of the specified pasteboard.

OSStatus PasteboardClear (
PasteboardRef inPasteboard
);
Parameters
inPasteboard

The pasteboard you want to clear.
Return Value

A result code.
Discussion

After calling this function, the application owns the pasteboard and can add data to it. You must call PasteboardClear before modifying a pasteboard. 
Availability
Available in Mac OS X v10.3 and later. 
Declared In
Pasteboard.h*/

OSStatus PasteboardClear (
PasteboardRef inPasteboard
);

/*PasteboardPutItemFlavor

Adds flavor data or a promise to the specified pasteboard.

OSStatus PasteboardPutItemFlavor (
PasteboardRef inPasteboard,
PasteboardItemID inItem,
CFStringRef inFlavorType,
CFDataRef inData,
PasteboardFlavorFlags inFlags
);
Parameters
inPasteboard

The pasteboard to which to add flavor data or a promise.
inItem

The identifier for the item to which to add flavor data or a promise.
inFlavorType

The flavor type of the data or promise you are adding, specified as a uniform type identifier. 
inData

The data to add. If you pass kPasteboardPromisedData (= NULL), the data is promised. 
inFlags

A bit field of flags for the specified flavor.
Return Value

A result code. 
Discussion

If you promise data, you must have implemented a promise keeper callback beforehand to deliver the data when asked for it (see PasteboardPromiseKeeperProcPtr for more details). Typically you store promises instead of the actual data when the data requires a large overhead to generate. 

You can call this function multiple times to add multiple flavors to a given item. You should add the flavors in your application’s order of preference or richness. 
Availability
Available in Mac OS X v10.3 and later. 
Declared In
Pasteboard.h*/

OSStatus PasteboardPutItemFlavor (
PasteboardRef inPasteboard,
PasteboardItemID inItem,
CFStringRef inFlavorType,
CFDataRef inData,
PasteboardFlavorFlags inFlags
);

/*PasteboardSynchronize

Synchronizes the local pasteboard reference to reflect the contents of the global pasteboard.

PasteboardSyncFlags PasteboardSynchronize (
PasteboardRef inPasteboard
);
Parameters
inPasteboard

The pasteboard you want to synchronize.
Return Value

A flag indicating what synchronization actions occurred.
Discussion

Calling this function compares the local pasteboard reference with its global pasteboard. If the global pasteboard was modified, it updates the local pasteboard reference to reflect this change. You typically call this function whenever your application becomes active, so that its pasteboard information reflects any changes that occurred while it was in the background. This function has low overhead, so you should call it whenever you suspect a global pasteboard may have been changed.
Availability
Available in Mac OS X v10.3 and later. 
Declared In
Pasteboard.h*/
PasteboardSyncFlags PasteboardSynchronize (
PasteboardRef inPasteboard
);


/*PasteboardGetItemFlavorFlags

Obtains the flags for a given flavor. 

OSStatus PasteboardGetItemFlavorFlags (
PasteboardRef inPasteboard,
PasteboardItemID inItem,
CFStringRef inFlavorType,
PasteboardFlavorFlags *outFlags
);

Parameters
inPasteboard

The pasteboard containing the data. 
inItem

The identifier for the item whose flavor flags you want to obtain.
inFlavorType

The flavor whose flags you want to obtain. 
outFlags

On return, outFlags points to a bit field containing the flavor flags. See “Pasteboard Flavor Flags” for a list of possible values. 
Return Value

A result code. 
Availability
Available in Mac OS X v10.3 and later. 
Declared In
Pasteboard.h*/
OSStatus PasteboardGetItemFlavorFlags (
PasteboardRef inPasteboard,
PasteboardItemID inItem,
CFStringRef inFlavorType,
PasteboardFlavorFlags *outFlags
);

/*PasteboardSetPromiseKeeper

Registers the promise keeper callback function for a pasteboard. 

OSStatus PasteboardSetPromiseKeeper (
PasteboardRef inPasteboard,
PasteboardPromiseKeeperProcPtr inPromiseKeeper,
void* inContext
);
Parameters
inPasteboard

The pasteboard to assign the promise keeper callback. If you have multiple pasteboards, you can assign multiple callbacks. 
inPromiseKeeper

A pointer to your promise keeper callback function. See PasteboardPromiseKeeperProcPtr for more information about implementing the callback. 
inContext

A pointer to application-defined data. The value you pass in this parameter is passed to your promise keeper callback function when it is called. 
Return Value

A result code. 
Discussion

You must have registered a promise keeper callback function before promising any data on the specified pasteboard. 
Availability
Available in Mac OS X v10.3 and later. 
Declared In
Pasteboard.h*/
OSStatus PasteboardSetPromiseKeeper (
PasteboardRef inPasteboard,
PasteboardPromiseKeeperProcPtr inPromiseKeeper,
void* inContext
);

#ifdef __cplusplus
}
#endif

#endif