/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/
 

/*

	<HSThread.c>	2006-03-20,21:06

*/

#include "HSThread.h"





/*
 *
 * pool of thread
 *
 */
HSSemaphore gQsema;
HS_UINT gQpos = 0;
HS_UINT gQsize = 0;
HS_TQ *gQarray = NULL;





HS_RESULT OpenThreadPool(HS_UINT pSize)
{
	HS_UINT i;

	if( gQsize != 0 || gQarray != NULL ) return HS_ERR;
	if( pSize==0 ) return HS_ERR_INVALID_PARAM;

	if( (gQarray=(HS_TQ*)HSMalloc(sizeof(HS_TQ)*pSize))==NULL )
		return HS_ERR_MALLOC;

	for(i=0; i<pSize; i++)
		gQarray[i] = HS_INVALID_TQ;
	gQsize = pSize;

	HSOpenSemaphore(gQsema);
	return HS_OK;
}


HS_RESULT CloseThreadPool()
{
	if( gQsize==0 || gQarray==NULL ) return HS_ERR;

	HSLockSemaphore(gQsema);
	gQsize = 0;
	HSFree(gQarray);
	gQarray = NULL;
	HSUnlockSemaphore(gQsema);

	HSCloseSemaphore(gQsema);
	return HS_OK;
}


HS_QID AddThreadQ(HS_TQ pQ)
{
	HS_UINT i;

	if( pQ==HS_INVALID_TQ ) return HS_INVALID_QID;
	if( gQsize==0 || gQarray==NULL ) return HS_INVALID_QID;

	HSLockSemaphore(gQsema);
	for(i=0; i<gQsize; i++)
	{
		gQpos++;
		gQpos = gQpos%gQsize;

		if( gQarray[gQpos]==HS_INVALID_TQ )
		{
			gQarray[gQpos] = pQ;
			HSUnlockSemaphore(gQsema);
			return (HS_QID)gQpos;
		}
	}
	HSUnlockSemaphore(gQsema);

	return HS_INVALID_QID;
}


HS_RESULT UpdateQ(HS_QID pId,HS_TQ pQ)
{
	if( pId >= gQsize ) return HS_ERR_INVALID_PARAM;

	HSLockSemaphore(gQsema);
	gQarray[pId] = pQ;
	HSUnlockSemaphore(gQsema);

	return HS_OK;
}


HS_RESULT SubThreadQ(HS_QID pId)
{
	if( pId > gQsize ) return HS_ERR_INVALID_PARAM;
	if( gQsize==0 || gQarray==NULL ) return HS_ERR;

	HSLockSemaphore(gQsema);
	gQarray[pId] = HS_INVALID_TQ;
	HSUnlockSemaphore(gQsema);

	return HS_OK;
}


void ShowThreadPool()
{
	HS_UINT i;

	if( gQsize==0 )
	{
		HSPrint("\ntp> q size is zero");
		return;
	}
	if( gQarray==NULL )
	{
		HSPrint("\ntp> q array is null");
		return;
	}

	HSLockSemaphore(gQsema);
	for(i=0; i<gQsize; i++)
		HSPrint("\ntp> q index[%u]:%08x",i,gQarray[i]);
	HSUnlockSemaphore(gQsema);
}





/*
 *
 * thread functions
 *
 */
typedef struct
{
	HS_TQ mQ;
	void *mObj;
	void *mArg;
	/* typically param1 is mObj pointer,
	         and param2 is mArg pointer.
	*/
	HS_RESULT (*HSThreadDemon)(HS_TQ,void*,void*);
} HSThreadContainer;





#if defined(_LINUX)
int gLinuxMsgQKey = HS_LINUX_MSGQ_KEY_MIN;
int gLinuxSemKey = HS_LINUX_SEM_KEY_MIN;


HS_TQ HSThread_msgget()
{
	int i;
	HS_TQ tQ;

	for(i=gLinuxMsgQKey; i<HS_LINUX_MSGQ_KEY_MAX; i++)
	{
		if( (tQ=msgget((key_t)i,IPC_CREAT|0666)) < 0 )
			continue;

		gLinuxMsgQKey = (i+1 >= HS_LINUX_MSGQ_KEY_MAX)? HS_LINUX_MSGQ_KEY_MIN:i+1;
		return tQ;
	}

	for(i=HS_LINUX_MSGQ_KEY_MIN; i<gLinuxMsgQKey; i++)
	{
		if( (tQ=msgget((key_t)i,IPC_CREAT|0666)) < 0 )
			continue;

		gLinuxMsgQKey = i+1;
		return tQ;
	}

	return HS_INVALID_TQ;
}


#if defined(_LINUX)
#define MAX_RESOURCE 3

union semun
{
	int val;
	struct semid_ds *buf;
	unsigned short int *array;
};
#endif


int _HSOpenSemaphore()
{
	int i;
	HSSemaphore tSem;
	union semun tSemUnion;

	tSemUnion.val = MAX_RESOURCE;

	for(i=gLinuxSemKey; i<HS_LINUX_SEM_KEY_MAX; i++)
	{
		if( (tSem=semget((key_t)i,MAX_RESOURCE,IPC_CREAT|0666)) < 0 )
			continue;

		gLinuxSemKey = (i+1 >= HS_LINUX_SEM_KEY_MAX)? HS_LINUX_SEM_KEY_MIN:i+1;

		if( semctl(tSem,0,SETVAL,tSemUnion) < 0 )
			return HS_INVALID_SEM;
		else
			return tSem;
	}

	for(i=HS_LINUX_SEM_KEY_MIN; i<gLinuxSemKey; i++)
	{
		if( (tSem=semget((key_t)i,MAX_RESOURCE,IPC_CREAT|0666)) < 0 )
			continue;

		gLinuxSemKey = i+1;

		if( semctl(tSem,0,SETVAL,tSemUnion) < 0 )
			return HS_INVALID_SEM;
		else
			return tSem;
	}

	return HS_INVALID_SEM;
}


HS_RESULT _HSCloseSemaphore(HSSemaphore pSem)
{
	if( semctl(pSem,0,IPC_RMID,0) < 0 )
		return HS_ERR;

	return HS_OK;
}


HS_RESULT _HSLockSemaphore(HSSemaphore pSem)
{
	struct sembuf tSembuf;

	tSembuf.sem_num = 0;
	tSembuf.sem_op = -1;
	tSembuf.sem_flg = SEM_UNDO;

	if( semop(pSem,&tSembuf,1) < 0 )
		return HS_ERR;

	return HS_OK;
}


HS_RESULT _HSUnlockSemaphore(HSSemaphore pSem)
{
	struct sembuf tSembuf;

	tSembuf.sem_num = 0;
	tSembuf.sem_op = 1;
	tSembuf.sem_flg = SEM_UNDO;

	if( semop(pSem,&tSembuf,1) < 0 )
		return HS_ERR;

	return HS_OK;
}


void *HSThreadMain(void *pArg)
{
	HSThreadMainLinux(pArg);
	return NULL;
}
#endif



#if defined(_WIN32) || defined(_WIN32_WCE)
unsigned __stdcall HSThreadMain(void* pArg)
#elif defined (_VXWORKS)
int HSThreadMain(void *pArg)
#elif defined(_LINUX)
int HSThreadMainLinux(void *pArg)
#endif
{
	HS_UINT tRetry = 5;
	HSThreadContainer *tContainer = (HSThreadContainer*)pArg;

	/* check parameter container
	*/
	if( tContainer==NULL )
		return 0;
	if( tContainer->mQ==HS_INVALID_TQ ||
		tContainer->mObj==NULL ||
	 	tContainer->HSThreadDemon==NULL )
	{
		HSFree(tContainer);
		return 0;
	}

	while(tRetry--)
	{
		if( tContainer->mQ != HS_INVALID_TQ )
			break;
		HSSleep(10);
	}

	tContainer->HSThreadDemon(tContainer->mQ,tContainer->mObj,tContainer->mArg);
	HSFree(tContainer);
	return 0;
}


HS_QID _HSThreadStart(char *pName,void *pObj,void *pArg,HS_RESULT (*pFunc)(HS_TQ,void*,void*))
{
	HS_QID tQid;
	HSThreadContainer *tContainer = NULL;

#if defined(_LINUX)
	pthread_t tThread;
#endif

	if( pName==NULL || pObj==NULL || pFunc==NULL ) return HS_INVALID_QID;
	if( (tContainer=(HSThreadContainer*)HSMalloc(sizeof(HSThreadContainer)))==NULL )
		return HS_INVALID_QID;

	tContainer->mObj = pObj;
	tContainer->mArg = pArg;
	tContainer->HSThreadDemon = pFunc;
	tContainer->mQ = HS_INVALID_TQ;

	/* Before starting thread, we have to get Qid slot
	 */
	if( (tQid=AddThreadQ((HS_TQ)1))==HS_INVALID_QID )
	{
		HSFree(tContainer);
		return HS_INVALID_QID;
	}

#if defined(_WIN32) || defined(_WIN32_WCE)
	CreateThread(NULL,0,&HSThreadMain,tContainer,0,&(tContainer->mQ));

#elif defined(_VXWORKS)
	if( (tContainer->mQ=msgQCreate(100,sizeof(HSQmsg),MSG_Q_FIFO))==NULL )
	{
		HSFree(tContainer);
		return HS_INVALID_QID;
	}
	taskSpawn(pName,120,0,16384,HSThreadMain,(int)tContainer,0,0,0,0,0,0,0,0,0);

#elif defined(_LINUX)
	if( (tContainer->mQ=HSThread_msgget())==HS_INVALID_TQ )
	{
		HSFree(tContainer);
		return HS_INVALID_QID;
	}
	pthread_create(&tThread,NULL,HSThreadMain,tContainer);
#endif

	UpdateQ(tQid,tContainer->mQ);
	return tQid;
}


HS_RESULT _HSThreadStop(HS_QID pQid)
{
	HS_RESULT tRet = HS_OK;

	if( pQid==HS_INVALID_QID )
		return HS_ERR_INVALID_PARAM;

	tRet = _HSThreadSendMessage(pQid,HS_QM_STOP,0,0);

	SubThreadQ(pQid);
	return tRet;
}


HS_RESULT HSExitThread(HS_TQ pQ,HS_RESULT pResult)
{
#if defined(_VXWORKS)
	if( pQ != HS_INVALID_TQ )
		msgQDelete(pQ);
#elif defined(_LINUX)
	if( pQ != HS_INVALID_TQ )
		msgctl(pQ,IPC_RMID,0);
#endif

	return pResult;
}


HS_RESULT _HSThreadSendMessage(HS_QID pQid,HS_UINT pMessage,HS_UINT pWparam,HS_UINT pLparam)
{
	HS_TQ tQ = HS_INVALID_TQ;
#if defined(_LINUX) || defined(_VXWORKS)
	HSQmsg tQmsg;
#endif

	if( pQid==HS_INVALID_QID ) return HS_ERR_INVALID_PARAM;

	HSLockSemaphore(gQsema);
	if( pQid < gQsize )
		tQ = gQarray[pQid];
	else
		tQ = HS_INVALID_TQ;
	HSUnlockSemaphore(gQsema);

	if( tQ==HS_INVALID_TQ ) return HS_ERR;

#if defined(_WIN32) || defined(_WIN32_WCE)
	PostThreadMessage(tQ,pMessage,(WPARAM)pWparam,(LPARAM)pLparam);
#elif defined(_VXWORKS)
	tQmsg.message = pMessage;
	tQmsg.wParam  = pWparam;
	tQmsg.lParam  = pLparam;
	msgQSend(tQ,(char*)&tQmsg,sizeof(tQmsg),NO_WAIT,MSG_PRI_NORMAL);
#elif defined(_LINUX)
	tQmsg.type = HS_LINUX_MSGQ_TYPE;
	tQmsg.message = pMessage;
	tQmsg.wParam = pWparam;
	tQmsg.lParam = pLparam;
	if( msgsnd(tQ,(void*)&tQmsg,sizeof(tQmsg)-sizeof(long),IPC_NOWAIT) != 0 )
		return HS_ERR;
#endif

	return HS_OK;
}


HS_RESULT HSThreadGetMessage(HS_TQ pQ,HSQmsg *pQmsg,HS_UINT pMin,HS_UINT pMax,BOOL pIsWait)
{
	if( pQ==HS_INVALID_TQ || pQmsg==NULL )
		return HS_ERR_NULL_PARAM;

	/* wait
	*/
	if( pIsWait==TRUE )
	{
#if defined(_WIN32) || defined(_WIN32_WCE)
		if( GetMessage(pQmsg,NULL,pMin,pMax) )
#elif defined (_VXWORKS)
		if( msgQReceive(pQ,(char*)pQmsg,sizeof(HSQmsg),WAIT_FOREVER) != ERROR/*-1*/ )
#elif defined (_LINUX)
			if( msgrcv(pQ,(void*)pQmsg,sizeof(HSQmsg)-sizeof(long),HS_LINUX_MSGQ_TYPE,MSG_NOERROR) >= 0 )
#endif
			return HS_OK;
		else
			return HS_ERR;
	}

	/* no wait
	*/
#if defined(_WIN32) || defined(_WIN32_WCE)
	if( PeekMessage(pQmsg,NULL,pMin,pMax,PM_REMOVE) )
#elif defined (_VXWORKS)
	if( msgQReceive(pQ,(char*)pQmsg,sizeof(HSQmsg),NO_WAIT) != ERROR/*-1*/ )
#elif defined (_LINUX)
		if( msgrcv(pQ,(void*)pQmsg,sizeof(HSQmsg)-sizeof(long),0,IPC_NOWAIT) >= 0 )
#endif
		return HS_OK;
	else
		return HS_ERR;
}






