/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/


/*

	<asnPrime.h>		2004-07-03,14:20

*/

#ifndef __2004_07_03__14_20__ASNPRIME_H__
#define __2004_07_03__14_20__ASNPRIME_H__


#include "H323Define.h"
#include "asnStream.h"



/*
	## Charactor String Values ##

 # NumericString = " 0123456789"

 # PrintableString = " '()+,-./0123456789:=?"
                     "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                     "abcdefghijklmnopqrstuvwxyz"

 # VisibleString = " !\"#$%&'()*+,-./0123456789:;<=>?"
                   "@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_"
                   "`abcdefghijklmnopqrstuvwxyz{|}~"

 # IA5String = "\000\001\002\003\004\005\006\007"
               "\010\011\012\013\014\015\016\017"
               "\020\021\022\023\024\025\026\027"
               "\030\031\032\033\034\035\036\037"
               " !\"#$%&'()*+,-./0123456789:;<=>?"
               "@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_"
               "`abcdefghijklmnopqrstuvwxyz{|}~\177"

 # GeneralString = "\000\001\002\003\004\005\006\007"
                   "\010\011\012\013\014\015\016\017"
                   "\020\021\022\023\024\025\026\027"
                   "\030\031\032\033\034\035\036\037"
                   " !\"#$%&'()*+,-./0123456789:;<=>?"
                   "@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_"
                   "`abcdefghijklmnopqrstuvwxyz{|}~\177"
                   "\200\201\202\203\204\205\206\207"
                   "\210\211\212\213\214\215\216\217"
                   "\220\221\222\223\224\225\226\227"
                   "\230\231\232\233\234\235\236\237"
                   "\240\241\242\243\244\245\246\247"
                   "\250\251\252\253\254\255\256\257"
                   "\260\261\262\263\264\265\266\267"
                   "\270\271\272\273\274\275\276\277"
                   "\300\301\302\303\304\305\306\307"
                   "\310\311\312\313\314\315\316\317"
                   "\320\321\322\323\324\325\326\327"
                   "\330\331\332\333\334\335\336\337"
                   "\340\341\342\343\344\345\346\347"
                   "\350\351\352\353\354\355\356\357"
                   "\360\361\362\363\364\365\366\367"
                   "\370\371\372\373\374\375\376\377"
*/



#define HS_ASN_STRING_FROM_MAX		256
#define HS_ASN_OPTION_MAP_MAX		128		/* maximum map size : 1024 = 128 x 8 */



typedef enum
{
	e_asnTagBerReserved = 0,
	e_asnTagBoolean = 1,
	e_asnTagInteger = 2,
	e_asnTagBitString = 3,
	e_asnTagOctetString = 4,
	e_asnTagNull = 5,
	e_asnTagObjectIdentifier = 6,
	e_asnTagObjectDescriptor = 7,
	e_asnTagExternal = 8,
	e_asnTagInstanceOf = 8,
	e_asnTagReal = 9,
	e_asnTagEnumerated = 10,
	e_asnTagEmbeddedPDV = 11,
	e_asnTagUTF8String = 12,
	e_asnTagRelativeOID = 13,
	e_asnTagReserved1 = 14,
	e_asnTagReserved2 = 15,
	e_asnTagSequence = 16,
	e_asnTagSequenceOf = 16,
	e_asnTagSet = 17,
	e_asnTagSetOf = 17,
	e_asnTagNumericString = 18,
	e_asnTagPrintableString = 19,
	e_asnTagTeletexString = 20,
	e_asnTagT61String = 20,
	e_asnTagVideotexString = 21,
	e_asnTagIA5String = 22,
	e_asnTagUTCTime = 23,
	e_asnTagGeneralizedTime = 24,
	e_asnTagGraphicString = 25,
	e_asnTagVisibleString = 26,
	e_asnTagISO646String = 26,
	e_asnTagGeneralString = 27,
	e_asnTagUniversalString = 28,
	e_asnTagCharacterString = 29,
	e_asnTagBMPString = 30,
	e_asnTagReservedAfter
} AsnTag;






/************************/
/* Functional Functions */
HS_UINT		Log2Size( HS_UINT len, BOOL zeroPossible );
HS_UINT		Log2SizeInterval( HS_UINT pMins, HS_UINT pMaxs );
HS_UINT		Log256Size( HS_UINT len, BOOL zeroPossible );
HS_UINT		Log256SizeInterval( HS_UINT pMins, HS_UINT pMaxs );
HS_RESULT	SetLengthOctet( unsigned char *pLength, AsnStream *pStrm );



/*******************/
/* Number Encoding */
HS_RESULT	UintFixedOctetEncoding( AsnStream *pStrm, HS_UINT byteSize, HS_UINT pValue );
HS_RESULT	UintEncoding( AsnStream *pStrm, HS_UINT pValue );
HS_RESULT	UintDecoding( AsnStream *pStrm, HS_UINT byteSize, HS_UINT *pValue );

HS_RESULT	LengthEncoding( AsnStream *pStrm, HS_UINT pMin, HS_UINT pMax, HS_UINT pLength );
HS_RESULT	LengthDecoding( AsnStream *pStrm, HS_UINT pMin, HS_UINT pMax, HS_UINT *pLength );

HS_RESULT	ConstrainedNumberEncoding( AsnStream *pStrm, int pMin, HS_UINT pMax, HS_UINT pValue );
HS_RESULT	SemiConstrainedNumberEncoding( AsnStream *pStrm, int pMin, HS_UINT pValue );
HS_RESULT	UnconstrainedNumberEncoding( AsnStream *pStrm, int pValue );
HS_RESULT	SmallNumberEncoding( AsnStream *pStrm, HS_UINT pValue );

HS_RESULT	ConstrainedNumberDecoding( AsnStream *pStrm, int pMin, HS_UINT pMax, HS_UINT *pValue );
HS_RESULT	SemiConstrainedNumberDecoding( AsnStream *pStrm, int pMin, HS_UINT *pValue );
HS_RESULT	UnconstrainedNumberDecoding( AsnStream *pStrm, int *pValue );
HS_RESULT	SmallNumberDecoding( AsnStream *pStrm, HS_UINT *pValue );



/**************************/
/* Constraint Information */
typedef enum
{
	e_unconstrained,
	e_semiConstrained,
	e_constrained,
	e_constrainedExt
} ConstraintType;


typedef struct
{
	ConstraintType	type;
	int				minValue;
	HS_UINT			maxValue;
	HS_UINT			distance;
} Constraint;






/*******************/
/* AsnObject Class */
#ifdef HS_CPLUSPLUS
class AsnObject
{
public:
	AsnObject( AsnTag pTag = e_asnTagBerReserved );
	~AsnObject();

#ifdef HS_DEBUG_ASN_PRINT
	virtual HS_RESULT Print(HS_UINT pDepth, char *pTypeName ){	return HS_OK;	};
#endif

	virtual HS_RESULT Encode(AsnStream *pStrm){	return HS_OK;	};
	virtual HS_RESULT Decode(AsnStream *pStrm){	return HS_OK;	};
	virtual HS_RESULT MakeMold(){ return HS_OK;	};
public:
	AsnTag	tag;
	HS_UINT	length;
};
#else
typedef struct
{
	void *obj;
	int	 size;
	HS_RESULT (*MakeMoldFunction)(void*);
	HS_RESULT (*EncodeFunction)(void*,AsnStream*);
	HS_RESULT (*DecodeFunction)(void*,AsnStream*);

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT (*PrintFunction)(void*,HS_UINT,char*);
#endif

	HS_RESULT (*DeleteFunction)(void*);
} AsnObject;

HS_RESULT new_AsnObject(
	void *pObject,
	void *pElement,
	int pSize,
	HS_RESULT (*MakeMoldFunction)(void*),
	HS_RESULT (*pEncodeFunction)(void*,AsnStream*),
	HS_RESULT (*pDecodeFunction)(void*,AsnStream*),
#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT (*pPrintFunction)(void*,HS_UINT,char*),
#endif
	HS_RESULT (*pDeleteFunction)(void*)
);

HS_RESULT delete_AsnObject( void *pObject );
#endif/*HS_CPLUSPLUS*/






#ifdef HS_CPLUSPLUS
HS_RESULT WellKnownExtensionEncoding(
	AsnObject *pEncodeObject,
	AsnStream *pStrm,
	AsnStream *pTempStrm
);
HS_RESULT WellKnownExtensionDecoding(
	AsnObject *pDecodeObject,
	AsnStream *pStrm
);
#else
HS_RESULT WellKnownExtensionEncoding(
	void *pEncodeObject,
	HS_RESULT(*pEncodeFunction)(void*,AsnStream*),
	AsnStream *pStrm,
	AsnStream *pTempStrm
);
HS_RESULT WellKnownExtensionDecoding(
	void *pDecodeObject,
	HS_RESULT(*pDecodeFunction)(void*,AsnStream*),
	AsnStream *pStrm
);
#endif






/***********************************************************************************************************/
/*                                                                                                         */
/*                                                                                                         */
/*                                          ASN Primitive Classes                                          */
/*                                                                                                         */
/*                                                                                                         */
/***********************************************************************************************************/

/********************/
/* AsnBoolean Class */
#ifdef HS_CPLUSPLUS
class AsnBoolean : public AsnObject
#else
typedef struct
#endif
{

#ifdef HS_CPLUSPLUS
public:
	AsnBoolean();
	~AsnBoolean();

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT Print( HS_UINT pDepth, char *pTypeName );
#endif

	HS_RESULT SetValue( BOOL pValue );

	HS_RESULT Encode( AsnStream *pStrm );
	HS_RESULT Decode( AsnStream *pStrm );
#endif

#ifdef HS_CPLUSPLUS
public:
#endif
	BOOL value;
#ifndef HS_CPLUSPLUS
	AsnTag		tag;
	HS_UINT		length;
#endif

#ifdef HS_CPLUSPLUS
};
#else
} AsnBoolean;

HS_RESULT new_AsnBoolean( void *pObject );
HS_RESULT delete_AsnBoolean( void *pObject );
HS_RESULT AsnBoolean_MakeMold( void *pObject );

#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT AsnBoolean_Print( void *pObject, HS_UINT pDepth, char *pTypeName );
#endif

HS_RESULT AsnBoolean_Encode( void *pObject, AsnStream *pStrm );
HS_RESULT AsnBoolean_Decode( void *pObject, AsnStream *pStrm );

HS_RESULT AsnBoolean_SetValue( void *pObject, BOOL pValue );
#endif






/*****************/
/* AsnNull Class */
#ifdef HS_CPLUSPLUS
class AsnNull : public AsnObject
#else
typedef struct
#endif
{

#ifdef HS_CPLUSPLUS
public:
	AsnNull();
	~AsnNull();

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT Print( HS_UINT pDepth, char *pTypeName );
#endif

	HS_RESULT Encode( AsnStream *pStrm );
	HS_RESULT Decode( AsnStream *pStrm );
#endif

#ifndef HS_CPLUSPLUS
	AsnTag		tag;
	HS_UINT		length;
#endif

#ifdef HS_CPLUSPLUS
};
#else
} AsnNull;

HS_RESULT new_AsnNull( void *pObject );
HS_RESULT delete_AsnNull( void *pObject );
HS_RESULT AsnNull_MakeMold( void *pObject );
#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT AsnNull_Print( void *pObject, HS_UINT pDepth, char *pTypeName );
#endif

HS_RESULT AsnNull_Encode( void *pObject, AsnStream *pStrm );
HS_RESULT AsnNull_Decode( void *pObject, AsnStream *pStrm );
#endif






/********************/
/* AsnInteger Class */
#ifdef HS_CPLUSPLUS
class AsnInteger : public AsnObject
#else
typedef struct
#endif
{

#ifdef HS_CPLUSPLUS
public:
	AsnInteger(
		ConstraintType pType = e_unconstrained,
		int pMinValue = 0,
		HS_UINT pMaxValue = HS_UINT_MAX,
		int pExtMinValue = 0,
		HS_UINT pExtMaxValue = HS_UINT_MAX
	);
	~AsnInteger();

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT Print( HS_UINT pDepth, char *pTypeName );
#endif

	HS_RESULT Encode( AsnStream *pStrm );
	HS_RESULT Decode( AsnStream *pStrm );

	HS_RESULT SetValue( HS_UINT pValue );
#endif

#ifdef HS_CPLUSPLUS
public:
#endif
	HS_UINT		value;
	Constraint	constraint;

	BOOL		extended;
	int			extMinValue;
	HS_UINT		extMaxValue;

#ifndef HS_CPLUSPLUS
	AsnTag		tag;
	HS_UINT		length;
#endif

#ifdef HS_CPLUSPLUS
};
#else
} AsnInteger;

HS_RESULT new_AsnInteger(
			void *pObject,
			ConstraintType pType,
			int pMinValue,
			HS_UINT pMaxValue,
			int pExtMinValue,
			HS_UINT pExtMaxValue
		);
HS_RESULT delete_AsnInteger( void *pObject );
HS_RESULT AsnInteger_MakeMold( void *pObject );

#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT AsnInteger_Print( void *pObject, HS_UINT pDepth, char *pTypeName );
#endif

HS_RESULT AsnInteger_Encode( void *pObject, AsnStream *pStrm );
HS_RESULT AsnInteger_Decode( void *pObject, AsnStream *pStrm );

HS_RESULT AsnInteger_SetValue( void *pObject, HS_UINT pValue );
#endif





#if 0	/* Sample Code */
/***********************/
/* AsnEnumerated Class */
typedef enum
{
	e_enum0,
	e_enum1,
	e_enum2,
	e_enum3,
	e_enum4,
	e_enum5,
	e_enum6,
	e_enum7,
	e_enum8,
	e_enum9,
} AsnEnumerated_AsnEnumerated;

#ifdef HS_CPLUSPLUS
class AsnEnumerated : public AsnObject
#else
typedef struct
#endif
{

#ifdef HS_CPLUSPLUS
public:
	AsnEnumerated( AsnEnumerated_AsnEnumerated pExtensionPoint = e_enum9 );
	~AsnEnumerated();

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT Print( HS_UINT pDepth, char *pTypeName );
#endif

	HS_RESULT Encode( AsnStream *pStrm );
	HS_RESULT Decode( AsnStream *pStrm );

	HS_RESULT AddEnumValue( int pEnumValue );
	HS_RESULT GetEnumValue( HS_UINT pPoint, int *pEnumValue );

	HS_RESULT SetValue( AsnEnumerated_AsnEnumerated pEnum );
#endif

#ifdef HS_CPLUSPLUS
public:
#endif
	AsnEnumerated_AsnEnumerated	extensionPoint;

	AsnEnumerated_AsnEnumerated	value;
	int							enumValue[HS_MAX_ENUMERATED];
	HS_UINT						enumSize;

#ifndef HS_CPLUSPLUS
	AsnTag	tag;
	HS_UINT	length;
#endif

#ifdef HS_CPLUSPLUS
};
#else
} AsnEnumerated;

HS_RESULT new_AsnEnumerated( void *pObject, AsnEnumerated_AsnEnumerated pExtensionPoint );
HS_RESULT delete_AsnEnumerated( void *pObject );
HS_RESULT AsnEnumerated_MakeMold( void *pObject );

#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT AsnEnumerated_Print( void *pObject, HS_UINT pDepth, char *pTypeName );
#endif

HS_RESULT AsnEnumerated_Encode( void *pObject, AsnStream *pStrm );
HS_RESULT AsnEnumerated_Decode( void *pObject, AsnStream *pStrm );

HS_RESULT AsnEnumerated_AddEnumValue( void *pObject, int pEnumValue );
HS_RESULT ASnEnumerated_GetEnumValue( void *pObject, HS_UINT pPoint, int *pEnumValue );

HS_RESULT AsnEnumerated_SetValue( void *pObject, AsnEnumerated_AsnEnumerated pEnum );
#endif
#endif/*0 : Sample Code*/






/*****************/
/* AsnReal Class */
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/






/**********************/
/* AsnBitString Class */
#ifdef HS_CPLUSPLUS
class AsnBitString : public AsnObject
#else
typedef struct
#endif
{
#ifdef HS_CPLUSPLUS
public:
	AsnBitString(
		ConstraintType pType = e_unconstrained,
		int pMinValue = 0,
		HS_UINT pMaxValue = HS_UINT_MAX,
		int pExtMinValue = 0,
		HS_UINT pExtMaxValue = HS_UINT_MAX
	);
	~AsnBitString();

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT Print( HS_UINT pDepth, char *pTypeName );
#endif

	HS_RESULT SetValue( HS_UINT pBitSize, unsigned char *pValue );

	HS_RESULT Encode( AsnStream *pStrm );
	HS_RESULT Decode( AsnStream *pStrm );
#endif

#ifdef HS_CPLUSPLUS
public:
#endif
	unsigned char *		value;
	Constraint			constraint;

	BOOL				extended;
	int					extMinValue;
	HS_UINT				extMaxValue;

#ifndef HS_CPLUSPLUS
	AsnTag		tag;
	HS_UINT		length;
#endif

#ifdef HS_CPLUSPLUS
};
#else
} AsnBitString;

HS_RESULT new_AsnBitString(
	void *pObject,
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
);
HS_RESULT delete_AsnBitString( void *pObject );
HS_RESULT AsnBitString_MakeMold( void *pObject );

#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT AsnBitString_Print( void *pObject, HS_UINT pDepth, char *pTypeName );
#endif

HS_RESULT AsnBitString_SetValue( void *pObject, HS_UINT pBitSize, unsigned char *pValue );

HS_RESULT AsnBitString_Encode( void *pObject, AsnStream *pStrm );
HS_RESULT AsnBitString_Decode( void *pObject, AsnStream *pStrm );
#endif






/************************/
/* AsnOctetString Class */
#ifdef HS_CPLUSPLUS
class AsnOctetString : public AsnObject
#else
typedef struct
#endif
{
#ifdef HS_CPLUSPLUS
public:
	AsnOctetString(
		ConstraintType pType = e_unconstrained,
		int pMinValue = 0,
		HS_UINT pMaxValue = HS_UINT_MAX,
		int pExtMinValue = 0,
		HS_UINT pExtMaxValue = HS_UINT_MAX
	);
	~AsnOctetString();

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT Print( HS_UINT pDepth, char *pTypeName );
#endif

	HS_RESULT SetValue( HS_UINT pByteSize, unsigned char *pValue );

	HS_RESULT Encode( AsnStream *pStrm );
	HS_RESULT Decode( AsnStream *pStrm );
#endif

#ifdef HS_CPLUSPLUS
public:
#endif
	unsigned char *		value;
	Constraint			constraint;

	BOOL				extended;
	int					extMinValue;
	HS_UINT				extMaxValue;

#ifndef HS_CPLUSPLUS
	AsnTag		tag;
	HS_UINT		length;
#endif

#ifdef HS_CPLUSPLUS
};
#else
} AsnOctetString;

HS_RESULT new_AsnOctetString(
	void *pObject,
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
);
HS_RESULT delete_AsnOctetString(void *pObject);
HS_RESULT AsnOctetString_MakeMold( void *pObject );

#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT AsnOctetString_Print( void *pObject, HS_UINT pDepth, char *pTypeName );
#endif

HS_RESULT AsnOctetString_SetValue( void *pObject, HS_UINT pByteSize, unsigned char *pValue );

HS_RESULT AsnOctetString_Encode( void *pObject, AsnStream *pStrm );
HS_RESULT AsnOctetString_Decode( void *pObject, AsnStream *pStrm );
#endif






/*****************************/
/* AsnObjectIdentifier Class */
#ifdef HS_CPLUSPLUS
class AsnObjectIdentifier : public AsnObject
#else
typedef struct
#endif
{

#ifdef HS_CPLUSPLUS
public:
	AsnObjectIdentifier();
	~AsnObjectIdentifier();

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT Print( HS_UINT pDepth, char *pTypeName );
#endif

	HS_RESULT Encode( AsnStream *pStrm );
	HS_RESULT Decode( AsnStream *pStrm );

	HS_RESULT SetValue( unsigned char *pValue, unsigned pLength );
#endif

#ifdef HS_CPLUSPLUS
public:
#endif
	unsigned char	*value;

#ifndef HS_CPLUSPLUS
	AsnTag		tag;
	HS_UINT		length;
#endif

#ifdef HS_CPLUSPLUS
};
#else
} AsnObjectIdentifier;

HS_RESULT new_AsnObjectIdentifier( void *pObject );
HS_RESULT delete_AsnObjectIdentifier( void *pObject );
HS_RESULT AsnObjectIdentifier_MakeMold( void *pObject );

#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT AsnObjectIdentifier_Print( void *pObject, HS_UINT pDepth, char *pTypeName );
#endif

HS_RESULT AsnObjectIdentifier_Encode( void *pObject, AsnStream *pStrm );
HS_RESULT AsnObjectIdentifier_Decode( void *pObject, AsnStream *pStrm );

HS_RESULT AsnObjectIdentifier_SetValue( void *pObject, unsigned char *pValue, unsigned pLength );
#endif






/************************/
/* AsnRelativeOID Class */
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/






/**************************/
/* AsnSingleString Class

	NumericString
	PrintableString
	VisibleString
	IA5String
	GeneralString
*/
#define ASN_NUMERIC_STRING_VMIN			32
#define ASN_NUMERIC_STRING_VMAX			57
#define ASN_PRINTABLE_STRING_VMIN		122
#define ASN_PRINTABLE_STRING_VMAX		57
#define ASN_VISIBLE_STRING_VMIN			126
#define ASN_VISIBLE_STRING_VMAX			57
#define ASN_IA5_STRING_VMIN				0
#define ASN_IA5_STRING_VMAX				127
#define ASN_GENERAL_STRING_VMIN			0
#define ASN_GENERAL_STRING_VMAX			255


BOOL GetStringTypeEncodingBits( HS_UINT pN, HS_UINT pVmax, HS_UINT *pAlignBitSize, HS_UINT *pUnalignBitSize );

#ifdef HS_CPLUSPLUS
class AsnSingleString : public AsnObject
#else
typedef struct
#endif
{
#ifdef HS_CPLUSPLUS
public:
	AsnSingleString(
		AsnTag pTag = e_asnTagNumericString,
		ConstraintType pType = e_unconstrained,
		int pMinValue = 0,
		HS_UINT pMaxValue = HS_UINT_MAX,
		int pExtMinValue = 0,
		HS_UINT pExtMaxValue = HS_UINT_MAX
	);
	~AsnSingleString();

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT	Print( HS_UINT pDepth, char *pTypeName );
#endif

	HS_RESULT	SetValue( unsigned char *pValue, HS_UINT pLength );
	HS_RESULT	SetValueInVisible( char *pValue );

	HS_RESULT	Encode( AsnStream *pStrm );
	HS_RESULT	Decode( AsnStream *pStrm );
#endif

#ifdef HS_CPLUSPLUS
public:
#endif
	unsigned char	*value;
	Constraint		constraint;

	BOOL			extended;
	int				extMinValue;
	HS_UINT			extMaxValue;

	/* ASN Parser will making the from constraint */
	unsigned char	fromConstraint[HS_UCHAR_MAX];
	HS_UINT			fromSize;
	BOOL			reIndexing;

	HS_UINT			encodingBitsOfAlign;
	HS_UINT			encodingBitsOfUnalign;

#ifndef HS_CPLUSPLUS
	AsnTag		tag;
	HS_UINT		length;
#endif

#ifdef HS_CPLUSPLUS
};
#else
} AsnSingleString;

HS_RESULT new_AsnSingleString(
	void *pObject,
	AsnTag pTag,
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
);
HS_RESULT delete_AsnSingleString( void *pObject );
HS_RESULT AsnSingleString_MakeMold( void *pObject );

#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT AsnSingleString_Print( void *pObject, HS_UINT pDepth, char *pTypeName );
#endif

HS_RESULT AsnSingleString_SetValue( void *pObject, unsigned char *pValue, HS_UINT pLength );
HS_RESULT AsnSingleString_SetValueInVisible( void *pObject, char *pValue );

HS_RESULT AsnSingleString_Encode( void *pObject, AsnStream *pStrm );
HS_RESULT AsnSingleString_Decode( void *pObject, AsnStream *pStrm );
#endif






/**************************/
/* AsnTeletexString Class */
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/

/***************************/
/* AsnVideotexString Class */
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/

/**************************/
/* AsnGraphicString Class */
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/






/*************************/
/* AsnDoubleString Class

	BMPString
*/
#define ASN_BMP_STRING_VMIN				0
#define ASN_BMP_STRING_VMAX				65535

#ifdef HS_CPLUSPLUS
class AsnDoubleString : public AsnObject
#else
typedef struct
#endif
{
#ifdef HS_CPLUSPLUS
public:
	AsnDoubleString(
		AsnTag pTag = e_asnTagBMPString,
		ConstraintType pType = e_unconstrained,
		int pMinValue = 0,
		HS_UINT pMaxValue = HS_UINT_MAX,
		int pExtMinValue = 0,
		HS_UINT pExtMaxValue = HS_UINT_MAX
	);
	~AsnDoubleString();

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT	Print( HS_UINT pDepth, char *pTypeName );
#endif

	HS_RESULT	SetValue( unsigned short *pValue, HS_UINT pLength );
	HS_RESULT	SetValueInVisible( char *pValue );

	HS_RESULT	Encode( AsnStream *pStrm );
	HS_RESULT	Decode( AsnStream *pStrm );
#endif

#ifdef HS_CPLUSPLUS
public:
#endif
	unsigned short	*value;
	Constraint		constraint;

	BOOL			extended;
	int				extMinValue;
	HS_UINT			extMaxValue;

	/* ASN Parser will making the from constraint */
	unsigned short	fromConstraint[HS_USHORT_MAX];
	HS_UINT			fromSize;
	BOOL			reIndexing;

	HS_UINT			encodingBitsOfAlign;
	HS_UINT			encodingBitsOfUnalign;

#ifndef HS_CPLUSPLUS
	AsnTag		tag;
	HS_UINT		length;
#endif

#ifdef HS_CPLUSPLUS
};
#else
} AsnDoubleString;

HS_RESULT new_AsnDoubleString(
	void *pObject,
	AsnTag pTag,
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
);
HS_RESULT delete_AsnDoubleString( void *pObject );
HS_RESULT AsnDoubleString_MakeMold( void *pObject );

#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT AsnDoubleString_Print( void *pObject, HS_UINT pDepth, char *pTypeName );
#endif

HS_RESULT AsnDoubleString_SetValue( void *pObject, unsigned short *pValue, HS_UINT pLength );
HS_RESULT AsnDoubleString_SetValueInVisible( void *pObject, char *pValue );

HS_RESULT AsnDoubleString_Encode( void *pObject, AsnStream *pStrm );
HS_RESULT AsnDoubleString_Decode( void *pObject, AsnStream *pStrm );
#endif






/***************************/
/* AsnUniveralString Class */
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/

/***********************/
/* AsnUTF8String Class */
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/

/*****************************/
/* AsnObjectDescriptor Class */
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/

/****************************/
/* AsnGeneralizedTime Class */
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/

/********************/
/* AsnUTCTime Class */
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/





/***********************************************************************************************************/
/* SequenceOf Class
*/
typedef struct
{
	void *unit;
	void *next;
} UnitOf;


#ifdef HS_CPLUSPLUS
template <class T_ASNOBJECT>
class AsnSequenceOf : public AsnObject
#else
typedef struct
#endif
{
#ifdef HS_CPLUSPLUS
public:
	AsnSequenceOf(
		ConstraintType pType = e_unconstrained,
		int pMinValue = 0,
		HS_UINT pMaxValue = HS_UINT_MAX,
		int pExtMinValue = 0,
		HS_UINT pExtMaxValue = HS_UINT_MAX
	);
	~AsnSequenceOf();

	HS_RESULT	SetMold(T_ASNOBJECT *pMold);

#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT	Print(HS_UINT pDepth, char *pTypeName);
#endif

	HS_RESULT	Clear();
	HS_UINT		GetSize();
	HS_RESULT	SetSize(HS_UINT pSize);
	T_ASNOBJECT* GetUnit(HS_UINT pIndex);

	HS_RESULT Encode(AsnStream *pStrm);
	HS_RESULT Decode(AsnStream *pStrm);
#endif

#ifdef HS_CPLUSPLUS
public:
	T_ASNOBJECT	*mold;
#else
	AsnObject	*mold;
#endif

	UnitOf		units;
	HS_UINT		size;

	Constraint	constraint;
	BOOL		extended;
	int			extMinValue;
	HS_UINT		extMaxValue;
#ifdef HS_CPLUSPLUS
};
#else
} AsnSequenceOf;

HS_RESULT new_AsnSequenceOf(
	void *pObject,
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
);
HS_RESULT delete_AsnSequenceOf(void *pObject);

HS_RESULT AsnSequenceOf_SetMold(void *pObject, AsnObject *pMold);

#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT AsnSequenceOf_Print(void *pObject, HS_UINT pDepth, char *pTypeName);
#endif

HS_RESULT AsnSequenceOf_Clear(void *pObject);
HS_UINT	  AsnSequenceOf_GetSize(void *pObject);
HS_RESULT AsnSequenceOf_SetSize(void *pObject, HS_UINT pSize);
void*	  AsnSequenceOf_GetUnit(void *pObject, HS_UINT pIndex);

HS_RESULT AsnSequenceOf_Encode(void *pObject, AsnStream *pStrm);
HS_RESULT AsnSequenceOf_Decode(void *pObject, AsnStream *pStrm);
#endif




#ifdef HS_CPLUSPLUS
/***********************************************************************************************************/
/* SequenceOf Class
*/
/*****************************/
/* AsnSequenceOf Constructor */
template <class T_ASNOBJECT>
AsnSequenceOf<T_ASNOBJECT>::AsnSequenceOf(
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
)
{
	AsnSequenceOf *pObj = this;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::AsnSequenceOf(NULL,%d,%d,%u,%d,%u) -> null parameter",
			(int)pType, pMinValue, pMaxValue, pExtMinValue, pExtMaxValue
		);
#endif
		return;
	}

	pObj->mold = HS_NULL;

	pObj->units.unit = HS_NULL;
	pObj->units.next = HS_NULL;
	pObj->size = 0;

	pObj->constraint.type = pType;
	pObj->constraint.minValue = pMinValue;
	pObj->constraint.maxValue = pMaxValue;
	pObj->constraint.distance = pMaxValue - pMinValue + 1;
	if( pObj->constraint.distance == 0 ) pObj->constraint.distance = HS_UINT_MAX;

	pObj->extMinValue = pExtMinValue;
	pObj->extMaxValue = pExtMaxValue;
	pObj->extended = FALSE;
}


/****************************/
/* AsnSequenceOf Destructor */
template <class T_ASNOBJECT>
AsnSequenceOf<T_ASNOBJECT>::~AsnSequenceOf()
{
	HS_RESULT tRet;

	AsnSequenceOf *pObj = this;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::~AsnSequenceOf(NULL) -> null parameter" );
#endif
		return;
	}

	tRet = pObj->Clear();

	if( pObj->mold != HS_NULL )
	{
		delete pObj->mold;
		pObj->mold = HS_NULL;
	}

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::~AsnSequenceOf(Object) -> %d, line(%d)", tRet, __LINE__ );
#endif
		return;
	}
}


/*************************/
/* AsnSequenceOf SetMold */
template <class T_ASNOBJECT>
HS_RESULT AsnSequenceOf<T_ASNOBJECT>::SetMold(T_ASNOBJECT *pMold)
{
	AsnSequenceOf *pObj = this;

	if( pObj == HS_NULL || pMold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::SetMold(%x,%s) -> null parameter", pObj, pMold );
#endif
		return HS_ERR_NULL_PARAM;
	}

	pObj->mold = pMold;
	return HS_OK;
}


/***********************/
/* AsnSequenceOf Clear */
template <class T_ASNOBJECT>
HS_RESULT AsnSequenceOf<T_ASNOBJECT>::Clear()
{
	HS_UINT	i;
	UnitOf *rounder = HS_NULL;
	UnitOf *deleter = HS_NULL;

	T_ASNOBJECT	*dUnit = HS_NULL;
	AsnSequenceOf *pObj = this;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Clear(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Clear(Object) -> no mold on 'sequence of'" );
#endif
		return HS_ERR_ASN_NO_MOLD;
	}

	
	if( pObj->size == 0 )
	{
		if( pObj->units.unit != HS_NULL || pObj->units.next != HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_ERR_ASN_CONFLICT;
		}
		else
			return HS_OK;
	}

	rounder = &(pObj->units);

	for( i=0; i<pObj->size; i++ )
	{
		if( rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_ERR_ASN_CONFLICT;
		}
		if( rounder->unit == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_ERR_ASN_CONFLICT;
		}

		dUnit = (T_ASNOBJECT *)rounder->unit;
		delete dUnit;

		deleter = rounder;
		rounder = (UnitOf *)rounder->next;

		if( i ) HSFree( deleter );
	}

	if( rounder != HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
		return HS_ERR_ASN_CONFLICT;
	}

	pObj->units.unit = HS_NULL;
	pObj->units.next = HS_NULL;
	pObj->size = 0;
	return HS_OK;
}


/*************************/
/* AsnSequenceOf GetSize */
template <class T_ASNOBJECT>
HS_UINT AsnSequenceOf<T_ASNOBJECT>::GetSize()
{
	AsnSequenceOf *pObj = this;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::GetSize(NULL) -> null parameter" );
#endif
		return 0;
	}

	return pObj->size;
}


/*************************/
/* AsnSequenceOf SetSize */
template <class T_ASNOBJECT>
HS_RESULT AsnSequenceOf<T_ASNOBJECT>::SetSize(HS_UINT pSize)
{

	HS_RESULT	tRet;
	HS_UINT		i;
	UnitOf		*rounder = HS_NULL;
	T_ASNOBJECT *newUnit = HS_NULL;

	AsnSequenceOf *pObj = this;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::SetSize(NULL,%u) -> null parameter", pSize );
#endif
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::SetSize(Object,%u) -> no mold on 'sequence of'", pSize );
#endif
		return HS_ERR_ASN_NO_MOLD;
	}

	pObj->extended = FALSE;
	if( pObj->constraint.type == e_semiConstrained )
	{
		if( (int)pSize < pObj->constraint.minValue )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSequenceOf::SetSize(Object,%u) -> size is too small, minValue(%d)", pSize, pObj->constraint.minValue );
#endif
			return HS_ERR_ASN_OUTOF_RANGE;
		}
	}
	else if( pObj->constraint.type == e_constrained )
	{
		if( (int)pSize < pObj->constraint.minValue || pSize > pObj->constraint.maxValue )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSequenceOf::SetSize(Object,%u) -> asn size conflict, minValue(%d), maxValue(%u)", pSize, pObj->constraint.minValue, pObj->constraint.maxValue );
#endif
			return HS_ERR_ASN_OUTOF_RANGE;
		}
	}
	else if( pObj->constraint.type == e_constrainedExt )
	{
		if( (int)pSize < pObj->constraint.minValue || pSize > pObj->constraint.maxValue ) pObj->extended = TRUE;
	}

	
	tRet = Clear();

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::SetSize(Object,%u) -> %d, line(%d)", pSize, __LINE__ );
#endif
		return tRet;
	}

	pObj->size = pSize;
	if( pSize == 0 ) return HS_OK;


	rounder = &(pObj->units);
	newUnit = new T_ASNOBJECT( *(pObj->mold) );
	newUnit->MakeMold();
	rounder->unit = newUnit;
	rounder->next = HS_NULL;

	if( pSize == 1 ) return HS_OK;


	for( i=1; i<pSize; i++ )
	{
		UnitOf *maker = (UnitOf*)HSMalloc( sizeof(UnitOf) );
		newUnit = new T_ASNOBJECT( *(pObj->mold) );
		newUnit->MakeMold();
		maker->unit = newUnit;
		maker->next = HS_NULL;

		rounder->next = maker;
		rounder = (UnitOf*)rounder->next;
	}

	return HS_OK;
}


/*************************/
/* AsnSequenceOf GetUnit */
template <class T_ASNOBJECT>
T_ASNOBJECT *AsnSequenceOf<T_ASNOBJECT>::GetUnit(HS_UINT pIndex)
{
	HS_UINT i;
	UnitOf *rounder = HS_NULL;

	AsnSequenceOf *pObj = this;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::GetUnit(NULL,%u) -> null parameter", pIndex );
#endif
		return HS_NULL;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::GetUnit(Object,%u) -> no mold on 'sequence of'", pIndex );
#endif
		return HS_NULL;
	}
	if( pIndex > pObj->size - 1 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::GetUnit(Object,%u) -> index is too big, size(%u)", pIndex, pObj->size );
#endif
		return HS_NULL;
	}

	rounder = &(pObj->units);
	for( i=0; i<pIndex; i++ )
	{
		if( rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::GetUnit(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_NULL;
		}

		rounder = (UnitOf*)rounder->next;
	}

	if( rounder == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::GetUnit(Object) -> conflict size, line(%d)", __LINE__ );
#endif
		return HS_NULL;
	}

	return (T_ASNOBJECT *)rounder->unit;
}


/************************/
/* AsnSequenceOf Encode */
template <class T_ASNOBJECT>
HS_RESULT AsnSequenceOf<T_ASNOBJECT>::Encode(AsnStream *pStrm)
{
	HS_RESULT	tRet;
	HS_UINT		i;
	UnitOf		*rounder = HS_NULL;

	T_ASNOBJECT *encoder = HS_NULL;
	AsnSequenceOf *pObj = this;

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Encode(Object,pStrm) -> no mold on 'sequence of'" );
#endif
		return HS_ERR_ASN_NO_MOLD;
	}
	if( pObj->size == 0 && pObj->constraint.minValue > 0 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Encode(Object,pStrm) -> size is too small, size(0), minValue(%d), constraintType(%d)",
			pObj->constraint.minValue, (int)pObj->constraint.type
		);
#endif
		return HS_ERR_ASN_CONFLICT;
	}

	if( pObj->constraint.type == e_constrainedExt )
		pStrm->BitEncoding(pObj->extended);

	if( pObj->extended )
		tRet = LengthEncoding(pStrm,0,HS_UINT_MAX,pObj->size);
	else
		tRet = LengthEncoding(pStrm,pObj->constraint.minValue,pObj->constraint.maxValue,pObj->size);

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Encode(Object,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif
		return tRet;
	}

	pStrm->Aligning();

	rounder = &(pObj->units);
	for( i=0; i<pObj->size; i++ )
	{
		if( rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Encode(Object,pStrm) -> asn size conflict, line(%d)", __LINE__ );
#endif
			return HS_NULL;
		}
		if( rounder->unit == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Encode(Object,pStrm) -> asn size conflict, line(%d)", __LINE__ );
#endif
			return HS_NULL;
		}

		encoder = (T_ASNOBJECT *)rounder->unit;
		tRet = encoder->Encode(pStrm);

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSequenceOf::Encode(Object,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif
			return tRet;
		}

		/*Mis-Standard:ASN{
		pStrm->Aligning();
		/*}*/

		rounder = (UnitOf*)rounder->next;
	}

	return HS_OK;
}


/************************/
/* AsnSequenceOf Decode */
template <class T_ASNOBJECT>
HS_RESULT AsnSequenceOf<T_ASNOBJECT>::Decode(AsnStream *pStrm)
{
	HS_RESULT	tRet;
	HS_UINT		tSize, i;
	UnitOf		*rounder = HS_NULL;

	T_ASNOBJECT	*decoder = HS_NULL;
	AsnSequenceOf *pObj = this;

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Decode(Object,pStrm) -> no mold on 'sequence of'" );
#endif
		return HS_ERR_ASN_NO_MOLD;
	}

	if( pObj->constraint.type == e_constrainedExt )
		pStrm->BitDecoding( &(pObj->extended) );

	if( pObj->extended )
		tRet = LengthDecoding(pStrm,0,HS_UINT_MAX, &(tSize) );
	else
		tRet = LengthDecoding(pStrm,pObj->constraint.minValue,pObj->constraint.maxValue, &(tSize) );

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Decode(Object,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif
		return tRet;
	}

	pStrm->Aligning();

	tRet = SetSize(tSize);

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Decode(Object,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif
		return tRet;
	}


	rounder = &(pObj->units);
	for( i=0; i<tSize; i++ )
	{
		if(rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Decode(Object,pStrm) -> asn size conflict, line(%d)", __LINE__ );
#endif
			return HS_NULL;
		}
		if(rounder->unit == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Decode(Object,pStrm) -> asn size conflict, line(%d)", __LINE__ );
#endif
			return HS_NULL;
		}

		decoder = (T_ASNOBJECT *)rounder->unit;
		tRet = decoder->Decode(pStrm);

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Decode(Object,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif
			return HS_NULL;
		}

		/*Mis-Standard:ASN{
		pStrm->Aligning();
		/*}*/

		rounder = (UnitOf *)rounder->next;
	}

	return HS_OK;
}


#ifdef HS_DEBUG_ASN_PRINT
/***********************/
/* AsnSequenceOf Print */
template <class T_ASNOBJECT>
HS_RESULT AsnSequenceOf<T_ASNOBJECT>::Print(HS_UINT pDepth, char *pTypeName)
{
	HS_RESULT	tRet;
	HS_UINT		i;
	UnitOf		*rounder = HS_NULL;
	char		indexString[16];

	T_ASNOBJECT *printer = HS_NULL;
	AsnSequenceOf *pObj = this;

	if( pObj == HS_NULL )
	{
		HSPrint( "\n Error on AsnSequenceOf::Print(NULL,%u,%s) -> null parameter", pDepth, (pTypeName==HS_NULL)? "":pTypeName );
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->mold == HS_NULL )
	{
		HSPrint( "\n Error AsnSequenceOf::Print(Object,%u,%s) -> no mold on 'sequence of'", pDepth, (pTypeName==HS_NULL)? "":pTypeName );
		return HS_ERR_ASN_NO_MOLD;
	}

	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );


	if( pTypeName != HS_NULL )
		HSPrint( "%s<SEQUENCE OF> = size[%u]", pTypeName, pObj->size );
	else
		HSPrint( "<SEQUENCE OF> = size[%u]", pObj->size );


	rounder = &(pObj->units);
	for( i=0; i<pObj->size; i++ )
	{
		if(rounder == HS_NULL )
		{
			HSPrint( "\n Error AsnSequenceOf::Print(Object,%u,%s) -> asn size conflict, line(%d)",
				pDepth, (pTypeName==HS_NULL)? "":pTypeName, __LINE__
			);
			return HS_ERR_ASN_CONFLICT;
		}
		if(rounder->unit == HS_NULL )
		{
			HSPrint( "\n Error AsnSequenceOf::Print(Object,%u,%s) -> asn size conflict, line(%d)",
				pDepth, (pTypeName==HS_NULL)? "":pTypeName, __LINE__
			);
			return HS_ERR_ASN_CONFLICT;
		}

		sprintf( indexString, "[%u] ", i );
		printer = (T_ASNOBJECT *)rounder->unit;
		tRet = printer->Print(pDepth+1,indexString);

		if( tRet != HS_OK )
		{
			HSPrint( "\n Error AsnSequenceOf::Print(Object,%u,%s) -> %d, line(%d), index(%u)",
				pDepth, (pTypeName==HS_NULL)? "":pTypeName, tRet, __LINE__, i
			);
			return HS_NULL;
		}

		rounder = (UnitOf *)rounder->next;
	}

	return HS_OK;
}
#endif/*HS_DEBUG_ASN_PRINT*/
#endif/*HS_CPLUSPLUS*/











/***********************************************************************************************************/
/*                                                                                                         */
/*                                                                                                         */
/*                                          ASN Constructed Classes                                        */
/*                                                                                                         */
/*                                                                                                         */
/***********************************************************************************************************/






void PrintNoUseEncode( const char *myName );
void PrintNoUseDecode( const char *myName );

#ifdef HS_CPLUSPLUS
#define ASN_NO_USE_YET(noUseName,nameString)							\
class ASN##noUseName : public AsnObject									\
{																		\
public:																	\
	ASN##noUseName(){}													\
	~ASN##noUseName(){}													\
	HS_RESULT Encode(AsnStream *pStrm){									\
		PrintNoUseEncode( ##nameString );								\
		return HS_ERR_ASN_NOUSEYET;										\
	}																	\
	HS_RESULT Decode(AsnStream *pStrm){									\
		PrintNoUseDecode( ##nameString );								\
		return HS_ERR_ASN_NOUSEYET;										\
	}																	\
};
#else/*HS_CPLUSPLUS*/
#define ASN_NO_USE_YET(noUseName,nameString)							\
typedef struct															\
{																		\
	int a;																\
} ASN##noUseName;														\
HS_RESULT new_ASN##noUseName( void *pObject );			\
HS_RESULT delete_ASN##noUseName( void *pObject );		\
HS_RESULT ASN##noUseName##_Print( void *pObject, HS_UINT pDepth, char *pTypeName );		\
HS_RESULT ASN##noUseName##_MakeMold( void *pObject );	\
HS_RESULT ASN##noUseName##_Encode( void *pObject, AsnStream *pStrm );	\
HS_RESULT ASN##noUseName##_Decode( void *pObject, AsnStream *pStrm );

#define ASN_NO_USE_YET_C(noUseName,nameString)							\
HS_RESULT new_ASN##noUseName( void *pObject ){ return HS_OK; }			\
HS_RESULT delete_ASN##noUseName( void *pObject ){ return HS_OK; }		\
HS_RESULT ASN##noUseName##_Print( void *pObject, HS_UINT pDepth, char *pTypeName ){ return HS_OK; }		\
HS_RESULT ASN##noUseName##_MakeMold( void *pObject ){ return HS_OK; }	\
HS_RESULT ASN##noUseName##_Encode( void *pObject, AsnStream *pStrm ){	\
	PrintNoUseEncode( nameString );										\
	return HS_ERR_ASN_NOUSEYET;											\
}																		\
HS_RESULT ASN##noUseName##_Decode( void *pObject, AsnStream *pStrm ){	\
		PrintNoUseDecode( nameString );									\
		return HS_ERR_ASN_NOUSEYET;										\
}
#endif/*HS_CPLUSPLUS*/



#endif/*__2004_07_03__14_20__ASNPRIME_H__*/


