/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/


/*

	<asnStream.cxx>		2004-07-03,11:42

*/

#include "asnStream.h"




unsigned char gLMASK[9] =	/* Left Mask */
{
	0x00,	/* 0000 0000 */
	0x80,	/* 1000 0000 */
	0xc0,	/* 1100 0000 */
	0xe0,	/* 1110 0000 */
	0xf0,	/* 1111 0000 */
	0xf8,	/* 1111 1000 */
	0xfc,	/* 1111 1100 */
	0xfe,	/* 1111 1110 */
	0xff	/* 1111 1111 */
};
unsigned char gRMASK[9] =	/* Right Mask */
{
	0xff,	/* 1111 1111 */
	0x7f,	/* 0111 1111 */
	0x3f,	/* 0011 1111 */
	0x1f,	/* 0001 1111 */
	0x0f,	/* 0000 1111 */
	0x07,	/* 0000 0111 */
	0x03,	/* 0000 0011 */
	0x01,	/* 0000 0001 */
	0x00	/* 0000 0000 */
};
unsigned char gBMASK[8] =	/* Bit Mask */
{
	0x80,	/* 1000 0000 */
	0x40,	/* 0100 0000 */
	0x20,	/* 0010 0000 */
	0x10,	/* 0001 0000 */
	0x08,	/* 0000 1000 */
	0x04,	/* 0000 0100 */
	0x02,	/* 0000 0010 */
	0x01	/* 0000 0001 */
};





/************************************************************************************/
/* Static Functions
*/
/************************/
/* Check Bytes Overflow */
HS_RESULT CheckBytesOverflow( AsnStream *pStrm, unsigned pAddingOffset )
{
	if( pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on CheckBytesOverflow(NULL,%u) -> null parameter", pAddingOffset );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pStrm->byteOffset + pAddingOffset > pStrm->size )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on CheckBytesOverflow(pStrm,%u) -> stream overflow, line(%d) : {%u,%u,%u}", pAddingOffset, __LINE__, pStrm->size, pStrm->byteOffset, pStrm->bitOffset );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_ASN_STREAM_OVERFLOW;
	}

	return HS_OK;
}


/**********************/
/* Check Bits Overflow*/
HS_RESULT CheckBitsOverflow( AsnStream *pStrm, unsigned pAddingOffset )
{
	if( pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on CheckBitsOverflow(NULL,%u) -> null parameter", pAddingOffset );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( ( (int)((pStrm->bitOffset + pAddingOffset)/8) + pStrm->byteOffset ) > pStrm->size )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on CheckBitsOverflow(pStrm,%u) -> stream overflow, line(%d) : {%u,%u,%u}", pAddingOffset, __LINE__, pStrm->size, pStrm->byteOffset, pStrm->bitOffset );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_ASN_STREAM_OVERFLOW;
	}

	return HS_OK;
}





/************************************************************************************/
/* AsnStream Members
*/
/*************************/
/* AsnStream Constructor */
#ifdef HS_CPLUSPLUS
AsnStream::AsnStream( unsigned pMaxSize, BOOL pAlign )
#else
HS_RESULT new_AsnStream( void *pObject, unsigned pMaxSize, BOOL pAlign )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::AsnStream(NULL,%d,%s) -> null parameter", pMaxSize, pAlign? "TRUE":"FALSE");
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

	pObj->isAlign = pAlign;

	pObj->datas = (unsigned char*)HSMalloc(pMaxSize+16);
	memset( pObj->datas, 0, pMaxSize );

	pObj->size = pMaxSize;
	pObj->byteOffset = 0;
	pObj->bitOffset = 0;

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/************************/
/* AsnStream Destructor */
#ifdef HS_CPLUSPLUS
AsnStream::~AsnStream()
#else
HS_RESULT delete_AsnStream( void *pObject )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::~AsnStream(NULL) -> null parameter");
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}
	
	HSFree( pObj->datas );

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/**********************/
/* AsnStream Aligning */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnStream::Aligning()
#else
HS_RESULT AsnStream_Aligning( void *pObject )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::Aligning(NULL) -> null parameter");
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->bitOffset == 0 ) return HS_OK;

	pObj->byteOffset++;
	pObj->bitOffset=0;
	return HS_OK;
}


/***********************/
/* AsnStream GetOffset */
#ifdef HS_CPLUSPLUS
unsigned AsnStream::GetOffset()
#else
unsigned AsnStream_GetOffset( void *pObject )
#endif
{
	unsigned result;

#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::GetOffset(NULL) -> null parameter");
#endif/*HS_DEBUG_ASN*/
		return 0;
	}

	if(pObj->bitOffset == 0) result = pObj->byteOffset;
	else					 result = pObj->byteOffset+1;

	return result;
}


/*******************/
/* AsnStream Reset
>> Initialization For Decoding
*/
#ifdef HS_CPLUSPLUS
HS_RESULT AsnStream::Reset()
#else
HS_RESULT AsnStream_Reset( void *pObject )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::Reset(NULL) -> null parameter");
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	pObj->byteOffset = 0;
	pObj->bitOffset = 0;

	return HS_OK;
}


/*******************/
/* AsnStream Clear
>> Initialization For Encoding
*/
#ifdef HS_CPLUSPLUS
HS_RESULT AsnStream::Clear()
#else
HS_RESULT AsnStream_Clear( void *pObject )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::Clear(NULL) -> null parameter");
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	memset( pObj->datas, 0, pObj->size );
	pObj->byteOffset = 0;
	pObj->bitOffset = 0;

	return HS_OK;
}



/**********************/
/* AsnStream GetDatas */
#ifdef HS_CPLUSPLUS
unsigned char *AsnStream::GetData()
#else
unsigned char *AsnStream_GetData( void *pObject )
#endif
{
	unsigned char *result = HS_NULL;
#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::GetDatas(NULL) -> null parameter" );
#endif/*HS_DEBUG_ASN*/
		return HS_NULL;
	}

	result = pObj->datas;

	pObj->datas = HS_NULL;
	pObj->size = 0;

	pObj->bitOffset = 0;
	pObj->byteOffset = 0;

	return result;
}



/*********************/
/* AsnStream PlusBit */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnStream::PlusBit( unsigned pPlus )
#else
HS_RESULT AsnStream_PlusBit( void *pObject, unsigned pPlus )
#endif
{
	HS_RESULT	tRet;
	int tOverValue=0;
	unsigned tPlusByte=0;

#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::PlusBit(NULL,%u) -> null parameter",pPlus);
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( (tRet = CheckBitsOverflow(pObj, pPlus)) != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::PlusBit(pObj,%u) -> %d, line(%d)", pPlus, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

	tOverValue = (int)(pObj->bitOffset + pPlus - 8);

	if( tOverValue < 0 )
	{
		pObj->bitOffset += pPlus;
	}
	else
	{
		pObj->byteOffset += ( (unsigned)(tOverValue/8) + 1 );
		pObj->bitOffset = (unsigned)(tOverValue%8);
	}

	return HS_OK;
}


/*************************/
/* AsnStream BitEncoding */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnStream::BitEncoding( BOOL pBit )
#else
HS_RESULT AsnStream_BitEncoding( void *pObject, BOOL pBit )
#endif
{

#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::BitEncoding(NULL,%s) -> null parameter", pBit? "TRUE":"FLASE" );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if(pBit) pObj->datas[ pObj->byteOffset ] |= gBMASK[pObj->bitOffset];

	if( pObj->bitOffset == 7 )
	{
		pObj->bitOffset = 0;
		pObj->byteOffset++;
	}else
		pObj->bitOffset++;

	return HS_OK;
}


/**************************/
/* AsnStream BitsEncoding */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnStream::BitsEncoding( unsigned char *pBits, unsigned pCounts, AlignDirection pDirection )
#else
HS_RESULT AsnStream_BitsEncoding( void *pObject, unsigned char *pBits, unsigned pCounts, AlignDirection pDirection )
#endif
{
	HS_RESULT tRet;
	unsigned tByteCounts, tBitCounts;

#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL || pBits == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::BitsEncoding(%x,%x,%u,%s) -> null parameter",
				 pObj, pBits, pCounts, pDirection==e_alignRight? "RIGHT":"LEFT"
		);
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	tByteCounts = pCounts/8;
	tBitCounts = pCounts%8;

	if( tBitCounts == 0 )
	{
#ifdef HS_CPLUSPLUS
		return pObj->BytesEncoding( pBits, tByteCounts );
#else
		return AsnStream_BytesEncoding( pObj, pBits, tByteCounts );
#endif
	}

	if( pDirection == e_alignLeft )
	{
#ifdef HS_CPLUSPLUS
		tRet = pObj->BytesEncoding( pBits, tByteCounts );
#else
		tRet = AsnStream_BytesEncoding( pObj, pBits, tByteCounts );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnStream::BitsEncoding(pObj,pBits,%u,%s) -> %d line(%d)", pCounts, pDirection==e_alignRight? "RIGHT":"LEFT", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

		pObj->datas[ pObj->byteOffset ] |= ( (pBits[tByteCounts] & gLMASK[tBitCounts]) >> pObj->bitOffset );
		
		if( (pObj->bitOffset + tBitCounts) > 8 )
			pObj->datas[ pObj->byteOffset+1 ] |= ( (pBits[tByteCounts] & gLMASK[tBitCounts]) << (8 - pObj->bitOffset) );

#ifdef HS_CPLUSPLUS
		tRet = pObj->PlusBit(tBitCounts);
#else
		tRet = AsnStream_PlusBit(pObj,tBitCounts);
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnStream::BitsEncoding(pObj,pBits,%u,%s) -> %d line(%d)", pCounts, pDirection==e_alignRight? "RIGHT":"LEFT", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else /* direction == e_alignRight */
	{
		if( (pObj->bitOffset + tBitCounts) <= 8 )
			pObj->datas[ pObj->byteOffset ] |= ( (pBits[0] & gRMASK[8-tBitCounts]) << (8 - pObj->bitOffset - tBitCounts) );
		else
		{
			pObj->datas[ pObj->byteOffset ]   |= ( (pBits[0] & gRMASK[8-tBitCounts]) >> (pObj->bitOffset + tBitCounts - 8 ) );
			pObj->datas[ pObj->byteOffset+1 ] |= ( (pBits[0] & gRMASK[8-tBitCounts]) << (16 - pObj->bitOffset - tBitCounts) );
		}

#ifdef HS_CPLUSPLUS
		tRet = pObj->PlusBit(tBitCounts);
#else
		tRet = AsnStream_PlusBit(pObj,tBitCounts);
#endif
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnStream::BitsEncoding(pObj,pBits,%u,%s) -> %d line(%d)", pCounts, pDirection==e_alignRight? "RIGHT":"LEFT", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

#ifdef HS_CPLUSPLUS
		return pObj->BytesEncoding( pBits+1, tByteCounts );
#else
		return AsnStream_BytesEncoding( pObj, pBits+1, tByteCounts );
#endif
	}

	return HS_OK;
}


/***************************/
/* AsnStream BytesEncoding */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnStream::BytesEncoding( unsigned char *pBytes, unsigned pCounts )
#else
HS_RESULT AsnStream_BytesEncoding( void *pObject, unsigned char *pBytes, unsigned pCounts )
#endif
{
	unsigned	i;
	HS_RESULT	tRet;

#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL || pBytes == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::BytesEncoding(%x,%x,%u) -> null parameter", pObj, pBytes, pCounts );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pCounts == 0 ) return HS_OK;

	if( (tRet = CheckBytesOverflow( pObj, pCounts )) != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::BytesEncoding(pObj,pBytes,%u) -> %d line(%d)", pCounts, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

	if( pObj->bitOffset == 0 )
	{
		memcpy( pObj->datas + pObj->byteOffset, pBytes, pCounts );
		pObj->byteOffset += pCounts;
		return HS_OK;
	}

	for( i=0; i<pCounts; i++ )
	{
		pObj->datas[ pObj->byteOffset++ ] |= ( pBytes[i] >> pObj->bitOffset );
		pObj->datas[ pObj->byteOffset   ] |= ( pBytes[i] << (8 - pObj->bitOffset) );
	}

	return HS_OK;
}


/*************************/
/* AsnStream BitDecoding */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnStream::BitDecoding( BOOL *pBit )
#else
HS_RESULT AsnStream_BitDecoding( void *pObject, BOOL *pBit )
#endif
{

#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL || pBit == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::BitDecoding(%x,%x) -> null parameter", pObj, pBit );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	*pBit = pObj->datas[ pObj->byteOffset ] & gBMASK[pObj->bitOffset];

	if( pObj->bitOffset == 7 )
	{
		pObj->bitOffset = 0;
		pObj->byteOffset++;
	}else
		pObj->bitOffset++;

	return HS_OK;
}


/**************************/
/* AsnStream BitsDecoding */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnStream::BitsDecoding( unsigned char *pBits, unsigned pCounts, AlignDirection pDirection )
#else
HS_RESULT AsnStream_BitsDecoding( void *pObject, unsigned char *pBits, unsigned pCounts, AlignDirection pDirection )
#endif
{
	HS_RESULT	tRet;
	unsigned	tByteCounts, tBitCounts;

#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL || pBits == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::BitsDecoding(%x,%x,%u,%s) -> null parameter",
				 pObj, pBits, pCounts, pDirection==e_alignRight? "RIGHT":"LEFT"
		);
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}
	
	tByteCounts = pCounts/8;
	tBitCounts = pCounts%8;

	if( tBitCounts == 0 )
	{
#ifdef HS_CPLUSPLUS
		return pObj->BytesDecoding( pBits, tByteCounts );
#else
		return AsnStream_BytesDecoding( pObj, pBits, tByteCounts );
#endif
	}

	if( pDirection == e_alignLeft )
	{
#ifdef HS_CPLUSPLUS
		tRet = pObj->BytesDecoding( pBits, tByteCounts+1 );
#else
		tRet = AsnStream_BytesDecoding( pObj, pBits, tByteCounts+1 );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnStream::BitsDecoding(pObj,pBits,%u,%s) -> %d line(%d)", pCounts, pDirection==e_alignRight? "RIGHT":"LEFT", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

		pBits[tByteCounts] &= gLMASK[tBitCounts];
		pObj->byteOffset--;

#ifdef HS_CPLUSPLUS
		pObj->PlusBit(tBitCounts);
#else
		AsnStream_PlusBit(pObj,tBitCounts);
#endif

		return HS_OK;
	}

	/* direction == e_alignRight */
	if( (8 - pObj->bitOffset) == tBitCounts )
	{
		/* obj->bitOffset Cant be 0 */
		memcpy( pBits, pObj->datas + pObj->byteOffset, tByteCounts+1 );
		pObj->byteOffset += (tByteCounts+1);
		pObj->bitOffset = 0;
		pBits[0] &= gRMASK[8-tBitCounts];
	}
	else
	{
		if( (8 - pObj->bitOffset) < tBitCounts )
		{
			memset( pBits, 0, tByteCounts+1 );
			pBits[0] |= ( (pObj->datas[ pObj->byteOffset ] & gRMASK[pObj->bitOffset]) << (tBitCounts + pObj->bitOffset - 8 ) );
			pBits[0] |= ( (pObj->datas[ pObj->byteOffset+1 ] & gLMASK[tBitCounts + pObj->bitOffset - 8]) >> (16 - tBitCounts - pObj->bitOffset) );
		}
		else/*   (8 - pObj->bitOffset) > tBitCounts */
		{
			memset( pBits, 0, tByteCounts+1 );
			pBits[0] |= ( (pObj->datas[ pObj->byteOffset ] & gRMASK[pObj->bitOffset]) >> (8 - pObj->bitOffset - tBitCounts) );
		}

#ifdef HS_CPLUSPLUS
		pObj->PlusBit(tBitCounts);
		return pObj->BytesDecoding( pBits+1, tByteCounts );
#else
		AsnStream_PlusBit(pObj,tBitCounts);
		return AsnStream_BytesDecoding( pObj, pBits+1, tByteCounts );
#endif
	}

	return HS_OK;
}


/**************************/
/* AsnStream BitsDecoding */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnStream::BytesDecoding( unsigned char *pBytes, unsigned pCounts )
#else
HS_RESULT AsnStream_BytesDecoding( void *pObject, unsigned char *pBytes, unsigned pCounts )
#endif
{
	unsigned	i;
	HS_RESULT	tRet;

#ifdef HS_CPLUSPLUS
	AsnStream *pObj = this;
#else
	AsnStream *pObj = (AsnStream *)pObject;
#endif

	if( pObj == HS_NULL || pBytes == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::BytesDecoding(%x,%x,%u) -> null parameter", pObj, pBytes, pCounts );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}
	
	if( pCounts == 0 ) return HS_OK;

	if( (tRet = CheckBytesOverflow(pObj, pCounts)) != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnStream::BytesDecoding(pObj,pBytes,%u) -> %d line(%d)", pCounts, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

	if( pObj->bitOffset == 0 )
	{
		memcpy( pBytes, pObj->datas + pObj->byteOffset, pCounts );
		pObj->byteOffset += pCounts;
	}
	else
	{
		memset( pBytes, 0, pCounts );

		for( i=0; i<pCounts; i++ )
		{
			pBytes[i] |= ( (pObj->datas[ pObj->byteOffset++ ] & gRMASK[pObj->bitOffset]) << pObj->bitOffset );
			pBytes[i] |= ( (pObj->datas[ pObj->byteOffset ] & gLMASK[pObj->bitOffset]) >> (8 - pObj->bitOffset) );
		}
	}

	return HS_OK;
}


#ifdef HS_DEBUG_ASN_PRINT
/*******************/
/* AsnStream Print */
HS_RESULT AsnStream_Print( void *pObject )
{
	unsigned i, tSize;
	AsnStream *pObj = (AsnStream *)pObject;

	if( pObj == HS_NULL )
	{
		HSPrint( "\n Error on AsnStream::Print() -> null parameter" );
		return HS_ERR_NULL_PARAM;
	}

	HSPrint( "\nASN Stream [size:%d/%d]{", pObj->byteOffset, pObj->bitOffset );
	tSize = pObj->byteOffset + (pObj->bitOffset? 1:0) ;
	for( i=0; i<tSize; i++ )
	{
		if( i%16 == 0 ) HSPrint( "\n   " );
		if( i%8 == 0 ) HSPrint( " " );
		HSPrint( "%02x ", pObj->datas[i] );
	}
	HSPrint( "\n} ASN Stream;" );

	return HS_OK;
}
#endif/*HS_DEBUG_ASN_PRINT*/






