/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<RasMessageMaker.c>		2005-03-13,13:29

*/

#include "RasMessageMaker.h"



unsigned char gPID_H225_V2_RAS[] = {gPID_H225_V2};
unsigned gSize_PID_H225_V2_RAS = gSize_PID_H225_V2;


/* MakeGRQ */
RasMsg *MakeGRQ( IEndpoint *pEndpoint, HS_USHORT *pSN )
{
	HS_USHORT tSN = 0;
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225GatekeeperRequest *grq = HS_NULL;

	if( pEndpoint == HS_NULL ) return HS_NULL;

	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	grq = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_gatekeeperRequest );
	if( grq == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	tSN = IEndpoint_MakeRasRequestSequenceNumber( pEndpoint );
	if( tSN == 0 )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	if( pSN != HS_NULL ) *pSN = tSN;

	/* fill message fields
	*/
	AsnInteger_SetValue( &(grq->m_requestSeqNum.inheritance), tSN );
	AsnObjectIdentifier_SetValue( &(grq->m_protocolIdentifier), gPID_H225_V2_RAS, gSize_PID_H225_V2_RAS );
	ASNH225TransportAddress_Copy( &(grq->m_rasAddress), &(pEndpoint->rasTsap) );
	
	grq->m_endpointType.m_mc.value = FALSE;
	grq->m_endpointType.m_undefinedNode.value = FALSE;

	if( pEndpoint->type == e_EndpointType_gateway )
		ASNH225EndpointType_IncludeOptionField( &(grq->m_endpointType), e_ASNH225EndpointTypeOptionMap_gateway );
	else if( pEndpoint->type == e_EndpointType_terminal )
		ASNH225EndpointType_IncludeOptionField( &(grq->m_endpointType), e_ASNH225EndpointTypeOptionMap_terminal );

	ASNH225EndpointType_IncludeOptionField( &(grq->m_endpointType), e_ASNH225EndpointTypeOptionMap_vendor );
	AsnInteger_SetValue( &(grq->m_endpointType.m_vendor.m_vendor.m_t35CountryCode), pEndpoint->countryCode );
	AsnInteger_SetValue( &(grq->m_endpointType.m_vendor.m_vendor.m_t35Extension), pEndpoint->extensionCode );
	AsnInteger_SetValue( &(grq->m_endpointType.m_vendor.m_vendor.m_manufacturerCode), pEndpoint->manufactureCode );

	ASNH225VendorIdentifier_IncludeOptionField( &(grq->m_endpointType.m_vendor), e_ASNH225VendorIdentifierOptionMap_productId );
	AsnOctetString_SetValue( &(grq->m_endpointType.m_vendor.m_productId), strlen(pEndpoint->productId), pEndpoint->productId );
	ASNH225VendorIdentifier_IncludeOptionField( &(grq->m_endpointType.m_vendor), e_ASNH225VendorIdentifierOptionMap_versionId );
	AsnOctetString_SetValue( &(grq->m_endpointType.m_vendor.m_versionId), strlen(pEndpoint->versionId), pEndpoint->versionId );

	return msg;
}


/* MakeRRQ */
RasMsg *MakeRRQ( IEndpoint *pEndpoint, HS_USHORT *pSN )
{
	HS_USHORT tSN = 0;
	HS_RESULT tRet;
	BOOL tIsRegisted = FALSE;

	ASNH225RasMessage *msg = HS_NULL;
	ASNH225RegistrationRequest *rrq = HS_NULL;

	if( pEndpoint == HS_NULL ) return HS_NULL;
	
	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	tIsRegisted = IGatekeeper_IsRegisted( &(pEndpoint->gatekeeper) );

	rrq = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_registrationRequest );
	if( rrq == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	tSN = IEndpoint_MakeRasRequestSequenceNumber( pEndpoint );
	if( tSN == 0 )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	if( pSN != HS_NULL ) *pSN = tSN;

	/* fill message fields
	*/
	AsnInteger_SetValue( &(rrq->m_requestSeqNum.inheritance), tSN );
	AsnObjectIdentifier_SetValue( &(rrq->m_protocolIdentifier), gPID_H225_V2_RAS, gSize_PID_H225_V2_RAS );

	rrq->m_discoveryComplete.value = FALSE;

	if( tIsRegisted == FALSE || pEndpoint->gatekeeper.isUseLightRRQ == FALSE )
	{
		AsnSequenceOf_SetSize( &(rrq->m_callSignalAddress), 1 );
		tRet=ASNH225TransportAddress_Copy( 
				(ASNH225TransportAddress*)AsnSequenceOf_GetUnit( &(rrq->m_callSignalAddress), 0 ),
				&(pEndpoint->csaTsap) 
		);
		if( tRet != HS_OK )
		{
			delete_ASNH225RasMessage( msg );
			HSFree( msg );
			return HS_NULL;
		}
	}

	AsnSequenceOf_SetSize( &(rrq->m_rasAddress), 1 );
	tRet=ASNH225TransportAddress_Copy( 
			(ASNH225TransportAddress*)AsnSequenceOf_GetUnit( &(rrq->m_rasAddress), 0 ),
			&(pEndpoint->rasTsap) 
	);
	if( tRet != HS_OK )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	rrq->m_terminalType.m_mc.value = FALSE;
	rrq->m_terminalType.m_undefinedNode.value = FALSE;

	if( tIsRegisted == FALSE || pEndpoint->gatekeeper.isUseLightRRQ == FALSE )
	{
		if( pEndpoint->type == e_EndpointType_gateway )
			ASNH225EndpointType_IncludeOptionField( &(rrq->m_terminalType), e_ASNH225EndpointTypeOptionMap_gateway );
		else if( pEndpoint->type == e_EndpointType_terminal )
			ASNH225EndpointType_IncludeOptionField( &(rrq->m_terminalType), e_ASNH225EndpointTypeOptionMap_terminal );

		ASNH225EndpointType_IncludeOptionField( &(rrq->m_terminalType), e_ASNH225EndpointTypeOptionMap_vendor );
		AsnInteger_SetValue( &(rrq->m_terminalType.m_vendor.m_vendor.m_t35CountryCode), pEndpoint->countryCode );
		AsnInteger_SetValue( &(rrq->m_terminalType.m_vendor.m_vendor.m_t35Extension), pEndpoint->extensionCode );
		AsnInteger_SetValue( &(rrq->m_terminalType.m_vendor.m_vendor.m_manufacturerCode), pEndpoint->manufactureCode );

		ASNH225VendorIdentifier_IncludeOptionField( &(rrq->m_terminalType.m_vendor), e_ASNH225VendorIdentifierOptionMap_productId );
		AsnOctetString_SetValue( &(rrq->m_terminalType.m_vendor.m_productId), strlen(pEndpoint->productId), pEndpoint->productId );
		ASNH225VendorIdentifier_IncludeOptionField( &(rrq->m_terminalType.m_vendor), e_ASNH225VendorIdentifierOptionMap_versionId );
		AsnOctetString_SetValue( &(rrq->m_terminalType.m_vendor.m_versionId), strlen(pEndpoint->versionId), pEndpoint->versionId );

		ASNH225RegistrationRequest_IncludeOptionField( rrq, e_ASNH225RegistrationRequestOptionMap_terminalAlias );
		tRet = AliasAddresses2AsnSequenceOf( &(rrq->m_terminalAlias), &(pEndpoint->aliases) );
		if( tRet != HS_OK )
		{
			delete_ASNH225RasMessage( msg );
			HSFree( msg );
			return HS_NULL;
		}
	}

	if( tIsRegisted == TRUE )
	{
		if( pEndpoint->gatekeeper.gatekeeperIdentifier.inheritance.length > 0 )
		{
			ASNH225RegistrationRequest_IncludeOptionField( rrq, e_ASNH225RegistrationRequestOptionMap_gatekeeperIdentifier );
			ASNH225GatekeeperIdentifier_Copy( &(rrq->m_gatekeeperIdentifier), &(pEndpoint->gatekeeper.gatekeeperIdentifier) );
		}
	}

	AsnInteger_SetValue( &(rrq->m_endpointVendor.m_vendor.m_t35CountryCode), pEndpoint->countryCode );
	AsnInteger_SetValue( &(rrq->m_endpointVendor.m_vendor.m_t35Extension), pEndpoint->extensionCode );
	AsnInteger_SetValue( &(rrq->m_endpointVendor.m_vendor.m_manufacturerCode), pEndpoint->manufactureCode );

	ASNH225VendorIdentifier_IncludeOptionField( &(rrq->m_endpointVendor), e_ASNH225VendorIdentifierOptionMap_productId );
	AsnOctetString_SetValue( &(rrq->m_endpointVendor.m_productId), strlen(pEndpoint->productId), pEndpoint->productId );
	ASNH225VendorIdentifier_IncludeOptionField( &(rrq->m_endpointVendor), e_ASNH225VendorIdentifierOptionMap_versionId );
	AsnOctetString_SetValue( &(rrq->m_endpointVendor.m_versionId), strlen(pEndpoint->versionId), pEndpoint->versionId );

	ASNH225RegistrationRequest_IncludeOptionField( rrq, e_ASNH225RegistrationRequestOptionMap_timeToLive );
	AsnInteger_SetValue( &(rrq->m_timeToLive.inheritance), pEndpoint->gatekeeper.ttl );

	ASNH225RegistrationRequest_IncludeOptionField( rrq, e_ASNH225RegistrationRequestOptionMap_keepAlive );
	ASNH225RegistrationRequest_IncludeOptionField( rrq, e_ASNH225RegistrationRequestOptionMap_willSupplyUUIEs );
	ASNH225RegistrationRequest_IncludeOptionField( rrq, e_ASNH225RegistrationRequestOptionMap_maintainConnection );

	rrq->m_keepAlive.value = tIsRegisted;
	rrq->m_willSupplyUUIEs.value = TRUE;
	rrq->m_maintainConnection.value = FALSE;

	if( pEndpoint->gatekeeper.endpointIdentifier.inheritance.length > 0 )
	{
		ASNH225RegistrationRequest_IncludeOptionField( rrq, e_ASNH225RegistrationRequestOptionMap_endpointIdentifier );
		ASNH225EndpointIdentifier_Copy( &(rrq->m_endpointIdentifier), &(pEndpoint->gatekeeper.endpointIdentifier) );
	}

	return msg;
}


/* MakeURQ */
RasMsg *MakeURQ( IEndpoint *pEndpoint, HS_USHORT *pSN )
{
	HS_USHORT tSN = 0;
	HS_RESULT tRet;
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225UnregistrationRequest *urq = HS_NULL;

	if( pEndpoint == HS_NULL ) return HS_NULL;
	
	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	urq = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_unregistrationRequest );
	if( urq == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	tSN = IEndpoint_MakeRasRequestSequenceNumber( pEndpoint );
	if( tSN == 0 )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	if( pSN != HS_NULL ) *pSN = tSN;

	/* fill message fields
	*/
	AsnInteger_SetValue( &(urq->m_requestSeqNum.inheritance), tSN );

	AsnSequenceOf_SetSize( &(urq->m_callSignalAddress), 1 );
	tRet=ASNH225TransportAddress_Copy( 
			(ASNH225TransportAddress*)AsnSequenceOf_GetUnit( &(urq->m_callSignalAddress), 0 ),
			&(pEndpoint->csaTsap)
	);
	if( tRet != HS_OK )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	ASNH225UnregistrationRequest_IncludeOptionField( urq, e_ASNH225UnregistrationRequestOptionMap_endpointAlias );
	tRet = AliasAddresses2AsnSequenceOf( &(urq->m_endpointAlias), &(pEndpoint->aliases) );
	if( tRet != HS_OK )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	if( pEndpoint->gatekeeper.endpointIdentifier.inheritance.length > 0 )
	{
		ASNH225UnregistrationRequest_IncludeOptionField( urq, e_ASNH225UnregistrationRequestOptionMap_endpointIdentifier );
		ASNH225EndpointIdentifier_Copy( &(urq->m_endpointIdentifier), &(pEndpoint->gatekeeper.endpointIdentifier) );
	}

	if( pEndpoint->gatekeeper.gatekeeperIdentifier.inheritance.length > 0 )
	{
		ASNH225UnregistrationRequest_IncludeOptionField( urq, e_ASNH225UnregistrationRequestOptionMap_gatekeeperIdentifier );
		ASNH225GatekeeperIdentifier_Copy( &(urq->m_gatekeeperIdentifier), &(pEndpoint->gatekeeper.gatekeeperIdentifier) );
	}

	return msg;
}


/* MakeARQ */
RasMsg *MakeARQ( IEndpoint *pEndpoint, ICall *pCall, HS_USHORT *pSN )
{
	HS_USHORT tSN = 0;
	HS_RESULT tRet;
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225AdmissionRequest *arq = HS_NULL;

	if( pEndpoint == HS_NULL || pCall == HS_NULL ) return HS_NULL;
	
	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	arq = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_admissionRequest );
	if( arq == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	tSN = IEndpoint_MakeRasRequestSequenceNumber( pEndpoint );
	if( tSN == 0 )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}
	if( pSN != HS_NULL ) *pSN = tSN;

	/* fill message fields
	*/
	AsnInteger_SetValue( &(arq->m_requestSeqNum.inheritance), tSN );
	ASNH225CallType_SetChoice( &(arq->m_callType), e_ASNH225CallTypeChoice_pointToPoint );
	ASNH225EndpointIdentifier_Copy( &(arq->m_endpointIdentifier), &(pEndpoint->gatekeeper.endpointIdentifier) );

	ASNH225AdmissionRequest_IncludeOptionField( arq, e_ASNH225AdmissionRequestOptionMap_destinationInfo );
	tRet = AliasAddresses2AsnSequenceOf( &(arq->m_destinationInfo), &(pCall->destAliases) );
	if( tRet != HS_OK )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	if( pCall->dcsaTsap.alter != HS_NULL )
	{
		ASNH225AdmissionRequest_IncludeOptionField( arq, e_ASNH225AdmissionRequestOptionMap_destCallSignalAddress );
		tRet = ASNH225TransportAddress_Copy( &(arq->m_destCallSignalAddress), &(pCall->dcsaTsap) );
		if( tRet != HS_OK )
		{
			delete_ASNH225RasMessage( msg );
			HSFree( msg );
			return HS_NULL;
		}
	}

	tRet = AliasAddresses2AsnSequenceOf( &(arq->m_srcInfo), &(pCall->sourceAliases) );
	if( tRet != HS_OK )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	if( pCall->scsaTsap.alter != HS_NULL )
	{
		ASNH225AdmissionRequest_IncludeOptionField( arq, e_ASNH225AdmissionRequestOptionMap_srcCallSignalAddress );
		tRet = ASNH225TransportAddress_Copy( &(arq->m_srcCallSignalAddress), &(pCall->scsaTsap) );
		if( tRet != HS_OK )
		{
			delete_ASNH225RasMessage( msg );
			HSFree( msg );
			return HS_NULL;
		}
	}

	AsnInteger_SetValue( &(arq->m_bandWidth.inheritance), pCall->bandwidth );
	AsnInteger_SetValue( &(arq->m_callReferenceValue.inheritance), pCall->crv );
	AsnOctetString_SetValue( &(arq->m_conferenceID.inheritance), 16, pCall->confId );
	arq->m_activeMC.value = FALSE;
	arq->m_answerCall.value = pCall->isAnswerCall;
	
	ASNH225AdmissionRequest_IncludeOptionField( arq, e_ASNH225AdmissionRequestOptionMap_canMapAlias );
	arq->m_canMapAlias.value = pCall->canMapAlias;
	ASNH225AdmissionRequest_IncludeOptionField( arq, e_ASNH225AdmissionRequestOptionMap_callIdentifier );
	AsnOctetString_SetValue( &(arq->m_callIdentifier.m_guid.inheritance), 16, pCall->callId );

	if( pEndpoint->gatekeeper.gatekeeperIdentifier.inheritance.length > 0 )
	{
		ASNH225AdmissionRequest_IncludeOptionField( arq, e_ASNH225AdmissionRequestOptionMap_gatekeeperIdentifier );
		ASNH225GatekeeperIdentifier_Copy( &(arq->m_gatekeeperIdentifier), &(pEndpoint->gatekeeper.gatekeeperIdentifier) );
	}

	ASNH225AdmissionRequest_IncludeOptionField( arq, e_ASNH225AdmissionRequestOptionMap_willSupplyUUIEs );
	arq->m_willSupplyUUIEs.value = TRUE;

	return msg;
}


/* MakeBRQ */
RasMsg *MakeBRQ( IEndpoint *pEndpoint, ICall *pCall, HS_UINT pBandwidth, HS_USHORT *pSN )
{
	HS_USHORT tSN = 0;
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225BandwidthRequest *brq = HS_NULL;

	if( pEndpoint == HS_NULL || pCall == HS_NULL ) return HS_NULL;
	
	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	brq = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_bandwidthRequest );
	if( brq == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	tSN = IEndpoint_MakeRasRequestSequenceNumber( pEndpoint );
	if( tSN == 0 )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	if( pSN != HS_NULL ) *pSN = tSN;

	/* fill message fields
	*/
	AsnInteger_SetValue( &(brq->m_requestSeqNum.inheritance), tSN );
	ASNH225EndpointIdentifier_Copy( &(brq->m_endpointIdentifier), &(pEndpoint->gatekeeper.endpointIdentifier) );
	AsnOctetString_SetValue( &(brq->m_conferenceID.inheritance), 16, pCall->confId );
	AsnInteger_SetValue( &(brq->m_callReferenceValue.inheritance), pCall->crv );

	/* Optional CallType Field {
	}*/

	AsnInteger_SetValue( &(brq->m_bandWidth.inheritance), pBandwidth );

	ASNH225BandwidthRequest_IncludeOptionField( brq, e_ASNH225BandwidthRequestOptionMap_callIdentifier );
	AsnOctetString_SetValue( &(brq->m_callIdentifier.m_guid.inheritance), 16, pCall->callId );

	if( pEndpoint->gatekeeper.gatekeeperIdentifier.inheritance.length > 0 )
	{
		ASNH225BandwidthRequest_IncludeOptionField( brq, e_ASNH225BandwidthRequestOptionMap_gatekeeperIdentifier );
		ASNH225GatekeeperIdentifier_Copy( &(brq->m_gatekeeperIdentifier), &(pEndpoint->gatekeeper.gatekeeperIdentifier) );
	}

	ASNH225BandwidthRequest_IncludeOptionField( brq, e_ASNH225BandwidthRequestOptionMap_answeredCall );
	brq->m_answeredCall.value = pCall->isAnswerCall;

	return msg;
}


/* MakeDRQ */
RasMsg *MakeDRQ( IEndpoint *pEndpoint, ICall *pCall, ASNH225DisengageReasonChoice pReason, HS_USHORT *pSN )
{
	HS_USHORT tSN = 0;
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225DisengageRequest *drq = HS_NULL;

	if( pEndpoint == HS_NULL || pCall == HS_NULL ||
		pReason > e_ASNH225DisengageReasonChoiceSizeExt - 1 )
		return HS_NULL;
	
	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	drq = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_disengageRequest );
	if( drq == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	tSN = IEndpoint_MakeRasRequestSequenceNumber( pEndpoint );
	if( tSN == 0 )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	if( pSN != HS_NULL ) *pSN = tSN;

	/* fill message fields
	*/
	AsnInteger_SetValue( &(drq->m_requestSeqNum.inheritance), tSN );
	ASNH225EndpointIdentifier_Copy( &(drq->m_endpointIdentifier), &(pEndpoint->gatekeeper.endpointIdentifier) );
	AsnOctetString_SetValue( &(drq->m_conferenceID.inheritance), 16, pCall->confId );
	AsnInteger_SetValue( &(drq->m_callReferenceValue.inheritance), pCall->crv );
	ASNH225DisengageReason_SetChoice( &(drq->m_disengageReason), pReason );

	ASNH225DisengageRequest_IncludeOptionField( drq, e_ASNH225DisengageRequestOptionMap_callIdentifier );
	AsnOctetString_SetValue( &(drq->m_callIdentifier.m_guid.inheritance), 16, pCall->callId );

	if( pEndpoint->gatekeeper.gatekeeperIdentifier.inheritance.length > 0 )
	{
		ASNH225DisengageRequest_IncludeOptionField( drq, e_ASNH225DisengageRequestOptionMap_gatekeeperIdentifier );
		ASNH225GatekeeperIdentifier_Copy( &(drq->m_gatekeeperIdentifier), &(pEndpoint->gatekeeper.gatekeeperIdentifier) );
	}

	ASNH225DisengageRequest_IncludeOptionField( drq, e_ASNH225DisengageRequestOptionMap_answeredCall );
	drq->m_answeredCall.value = pCall->isAnswerCall;

	return msg;
}





/* endpoint RAS response messages:
	cause by
	FORCE URQ, DRQ and BRQ
	or IRQ for information from GK
	or receiving unknown message.
*/

/* make UCF message
*/
RasMsg *MakeUCF(HS_USHORT pMsgSN)
{
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225UnregistrationConfirm *ucf = HS_NULL;

	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	ucf = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_unregistrationConfirm );
	if( ucf == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	/* fill message fields
	*/
	AsnInteger_SetValue( &(ucf->m_requestSeqNum), pMsgSN );

	return msg;
}


/* make URJ message
*/
RasMsg *MakeURJ( HS_USHORT pMsgSN, ASNH225UnregRejectReasonChoice pReason )
{
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225UnregistrationReject *urj = HS_NULL;

	if( pReason > e_ASNH225UnregRejectReasonChoiceSizeExt-1 ) return HS_NULL;

	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	urj = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_unregistrationReject );
	if( urj == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	/* fill message fields
	*/
	AsnInteger_SetValue( &(urj->m_requestSeqNum), pMsgSN );
	ASNH225UnregRejectReason_SetChoice( &(urj->m_rejectReason), pReason );

	return msg;
}


/* make DCF message
*/
RasMsg *MakeDCF(HS_USHORT pMsgSN)
{
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225DisengageConfirm *dcf = HS_NULL;

	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	dcf = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_disengageConfirm );
	if( dcf == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	/* fill message fields
	*/
	AsnInteger_SetValue( &(dcf->m_requestSeqNum), pMsgSN );

	return msg;
}


/* make DRJ message
*/
RasMsg *MakeDRJ( HS_USHORT pMsgSN, ASNH225DisengageRejectReasonChoice pReason )
{
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225DisengageReject *drj = HS_NULL;

	if( pReason > e_ASNH225DisengageRejectReasonChoiceSizeExt-1 ) return HS_NULL;

	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	drj = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_disengageReject );
	if( drj == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	/* fill message fields
	*/
	AsnInteger_SetValue( &(drj->m_requestSeqNum), pMsgSN );
	ASNH225DisengageRejectReason_SetChoice( &(drj->m_rejectReason), pReason );

	return msg;
}


/* make BCF message
*/
RasMsg *MakeBCF( HS_USHORT pMsgSN, HS_UINT pBandwidth )
{
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225BandwidthConfirm *bcf = HS_NULL;

	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	bcf = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_bandwidthConfirm );
	if( bcf == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	/* fill message fields
	*/
	AsnInteger_SetValue( &(bcf->m_requestSeqNum), pMsgSN );
	AsnInteger_SetValue( &(bcf->m_bandWidth), pBandwidth );

	return msg;
}


/* make BRJ message
*/
RasMsg *MakeBRJ( HS_USHORT pMsgSN, ASNH225BandRejectReasonChoice pReason, HS_UINT pBandwidth )
{
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225BandwidthReject *brj = HS_NULL;

	if( pReason > e_ASNH225BandRejectReasonChoiceSizeExt-1 ) return HS_NULL;

	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	brj = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_bandwidthReject );
	if( brj == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	/* fill message fields
	*/
	AsnInteger_SetValue( &(brj->m_requestSeqNum), pMsgSN );
	ASNH225BandRejectReason_SetChoice( &(brj->m_rejectReason), pReason );

	return msg;
}


/* make XRS message
*/
RasMsg *MakeXRS(HS_USHORT pMsgSN)
{
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225UnknownMessageResponse *xrs = HS_NULL;

	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	xrs = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_unknownMessageResponse );
	if( xrs == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	/* fill message fields
	*/
	AsnInteger_SetValue( &(xrs->m_requestSeqNum), pMsgSN );

	return msg;
}


/* make general IRR message
*/
static RasMsg *MakeIRR( IEndpoint *pEndpoint )
{
	HS_RESULT tRet = HS_OK;
	ASNH225RasMessage *msg = HS_NULL;
	ASNH225InfoRequestResponse *irr = HS_NULL;

	if( pEndpoint == HS_NULL ) return HS_NULL;

	msg=(ASNH225RasMessage*)HSMalloc(sizeof(ASNH225RasMessage));
	if( msg == HS_NULL ) return HS_NULL;

	new_ASNH225RasMessage( msg );
	ASNH225RasMessage_MakeMold( msg );

	irr = ASNH225RasMessage_SetChoice( msg, e_ASNH225RasMessageChoice_infoRequestResponse );
	if( irr == HS_NULL )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	/* fill message fields
	*/
	/* set endpointType
	*/
	ASNH225EndpointType_IncludeOptionField( &(irr->m_endpointType), e_ASNH225EndpointTypeOptionMap_vendor );
	AsnInteger_SetValue( &(irr->m_endpointType.m_vendor.m_vendor.m_t35CountryCode), pEndpoint->countryCode );
	AsnInteger_SetValue( &(irr->m_endpointType.m_vendor.m_vendor.m_t35Extension), pEndpoint->extensionCode );
	AsnInteger_SetValue( &(irr->m_endpointType.m_vendor.m_vendor.m_manufacturerCode), pEndpoint->manufactureCode );

	ASNH225VendorIdentifier_IncludeOptionField( &(irr->m_endpointType.m_vendor), e_ASNH225VendorIdentifierOptionMap_productId );
	AsnOctetString_SetValue( &(irr->m_endpointType.m_vendor.m_productId), strlen(pEndpoint->productId), pEndpoint->productId );
	ASNH225VendorIdentifier_IncludeOptionField( &(irr->m_endpointType.m_vendor), e_ASNH225VendorIdentifierOptionMap_versionId );
	AsnOctetString_SetValue( &(irr->m_endpointType.m_vendor.m_versionId), strlen(pEndpoint->versionId), pEndpoint->versionId );

	if( pEndpoint->type == e_EndpointType_gateway )
		ASNH225EndpointType_IncludeOptionField( &(irr->m_endpointType), e_ASNH225EndpointTypeOptionMap_gateway );
	else if( pEndpoint->type == e_EndpointType_terminal )
		ASNH225EndpointType_IncludeOptionField( &(irr->m_endpointType), e_ASNH225EndpointTypeOptionMap_terminal );
	irr->m_endpointType.m_mc.value = FALSE;
	irr->m_endpointType.m_undefinedNode.value = FALSE;
	/* set endpointIdentifier
	*/
	if( pEndpoint->gatekeeper.endpointIdentifier.inheritance.length > 0 )
		tRet = ASNH225EndpointIdentifier_Copy( &(irr->m_endpointIdentifier), &(pEndpoint->gatekeeper.endpointIdentifier) );
	else
		tRet = AsnDoubleString_SetValueInVisible( &(irr->m_endpointIdentifier.inheritance), "No Identifier" );
	if( tRet != HS_OK )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}
	/* set rasAddress
	*/
	tRet=ASNH225TransportAddress_Copy( &(irr->m_rasAddress), &(pEndpoint->rasTsap) );
	if( tRet != HS_OK )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}
	/* set callSignalAddress
	*/
	AsnSequenceOf_SetSize( &(irr->m_callSignalAddress), 1 );
	tRet=ASNH225TransportAddress_Copy( 
			(ASNH225TransportAddress*)AsnSequenceOf_GetUnit( &(irr->m_callSignalAddress), 0 ),
			&(pEndpoint->csaTsap)
	);
	if( tRet != HS_OK )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}
	/* set endpointAlias
	*/
	ASNH225InfoRequestResponse_IncludeOptionField( irr, e_ASNH225InfoRequestResponseOptionMap_endpointAlias );
	tRet = AliasAddresses2AsnSequenceOf( &(irr->m_endpointAlias), &(pEndpoint->aliases) );
	if( tRet != HS_OK )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}
	/* include perCallInfo field
	*/
	ASNH225InfoRequestResponse_IncludeOptionField( irr, e_ASNH225InfoRequestResponseOptionMap_perCallInfo );
	/* include extension mandatory fields
	*/
	ASNH225InfoRequestResponse_IncludeOptionField( irr, e_ASNH225InfoRequestResponseOptionMap_needResponse );
	ASNH225InfoRequestResponse_IncludeOptionField( irr, e_ASNH225InfoRequestResponseOptionMap_unsolicited );
	/* I don't need IACK or INAK
	*/
	irr->m_needResponse.value = HS_NO;
	irr->m_unsolicited.value = HS_NO;

	return msg;
}


/* fill perCallInfo field on IRR
*/
static HS_RESULT MakeIRR_perCallInfo( ASNH225InfoRequestResponse_perCallInfo *pPerCallInfo, ICall *pCall, ASNH225H323_UU_PDU *pPdu, BOOL pIsSent )
{
	HS_RESULT tRet = HS_OK;
	HS_UCHAR tUcharArr[16] = {0,};
	if( pPerCallInfo == HS_NULL ) return HS_ERR_NULL_PARAM;

	/* set CRV and conferenceID
	*/
	if( pCall == HS_NULL )
	{
		pPerCallInfo->m_callReferenceValue.inheritance.value = 0;
		AsnOctetString_SetValue( &(pPerCallInfo->m_conferenceID.inheritance), 16, tUcharArr );
	}
	else
	{
		pPerCallInfo->m_callReferenceValue.inheritance.value = pCall->crv;
		AsnOctetString_SetValue( &(pPerCallInfo->m_conferenceID.inheritance), 16, pCall->confId );
	}
	/* set originator
	*/
	if( pCall != HS_NULL )
	{
		ASNH225InfoRequestResponse_perCallInfo_IncludeOptionField( pPerCallInfo,
			e_ASNH225InfoRequestResponse_perCallInfoOptionMap_originator
		);
		if( pCall->isAnswerCall )	pPerCallInfo->m_originator.value = HS_NO;
		else						pPerCallInfo->m_originator.value = HS_YES;
	}
	/* set call status
	*/
	ASNH225CallType_SetChoice( &(pPerCallInfo->m_callType), e_ASNH225CallTypeChoice_pointToPoint );
	if( pCall == HS_NULL )
	{
		AsnInteger_SetValue( &(pPerCallInfo->m_bandWidth.inheritance), 0 );
		ASNH225CallModel_SetChoice( &(pPerCallInfo->m_callModel), e_ASNH225CallModelChoice_direct );
	}
	else
	{
		AsnInteger_SetValue( &(pPerCallInfo->m_bandWidth.inheritance), pCall->bandwidth );
		ASNH225CallModel_SetChoice( &(pPerCallInfo->m_callModel), pCall->callModel );
	}
	/* set callIdentifier
	*/
	ASNH225InfoRequestResponse_perCallInfo_IncludeOptionField( pPerCallInfo,
		e_ASNH225InfoRequestResponse_perCallInfoOptionMap_callIdentifier
	);
	if( pCall != HS_NULL )
		AsnOctetString_SetValue( &(pPerCallInfo->m_callIdentifier.m_guid.inheritance), 16, pCall->callId );
	else
		AsnOctetString_SetValue( &(pPerCallInfo->m_callIdentifier.m_guid.inheritance), 16, tUcharArr );
	/* include extension mandatory field
	*/
	ASNH225InfoRequestResponse_perCallInfo_IncludeOptionField( pPerCallInfo,
		e_ASNH225InfoRequestResponse_perCallInfoOptionMap_substituteConfIDs
	);
	/* set Q.931 pdu
	*/
	if( pPdu != HS_NULL )
	{
		ASNH225InfoRequestResponse_perCallInfo_pdu *tPdu = HS_NULL;

		ASNH225InfoRequestResponse_perCallInfo_IncludeOptionField( pPerCallInfo,
			e_ASNH225InfoRequestResponse_perCallInfoOptionMap_pdu
		);
		HS_TRY( AsnSequenceOf_SetSize( &(pPerCallInfo->m_pdu), 1 ) )
		tPdu = (ASNH225InfoRequestResponse_perCallInfo_pdu*)AsnSequenceOf_GetUnit( &(pPerCallInfo->m_pdu), 0 );
		if( tPdu == HS_NULL ) return HS_ERR_H323;

		tPdu->m_sent.value = pIsSent;
		HS_TRY( ASNH225H323_UU_PDU_Copy( &(tPdu->m_h323pdu), pPdu ) )
	}

	return HS_OK;
}
static HS_RESULT MakeIRR_perCallInfoOne(AsnSequenceOf *pPerCallInfos, ICall *pCall, ASNH225H323_UU_PDU *pPdu, BOOL pIsSent )
{
	if( pPerCallInfos == HS_NULL ) return HS_ERR_NULL_PARAM;

	AsnSequenceOf_SetSize(pPerCallInfos,1);
	return MakeIRR_perCallInfo(
		AsnSequenceOf_GetUnit(pPerCallInfos,0), pCall, pPdu, pIsSent
	);
}
static HS_RESULT MakeIRR_perCallInfoAll(AsnSequenceOf *pPerCallInfos, NoLockList *pCalls)
{
	HS_UINT i;
	HS_RESULT tRet;

	if( pPerCallInfos==HS_NULL || pCalls==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pCalls->size == 0 )
		return MakeIRR_perCallInfoOne(pPerCallInfos, HS_NULL, HS_NULL, HS_NO);

	HS_TRY( AsnSequenceOf_SetSize(pPerCallInfos, pCalls->size) )
	for( i=0; i<pCalls->size; i++ )
	{
		HS_TRY( 
			MakeIRR_perCallInfo(
				AsnSequenceOf_GetUnit(pPerCallInfos,i),
				NoLockList_GetData(pCalls,i),
				HS_NULL, HS_NO
			)
		)
	}

	return HS_OK;
}


/* IRR message by IRQ that include unique CRV
*/
RasMsg *MakeIRRwithOne( IEndpoint *pEndpoint, HS_USHORT pMsgSN, ICall *pCall )
{
	HS_RESULT tRet = HS_OK;
	RasMsg *msg = HS_NULL;
	ASNH225InfoRequestResponse *irr = HS_NULL;

	if( pEndpoint == HS_NULL ) return HS_NULL;
	if( (msg=MakeIRR(pEndpoint)) == HS_NULL ) return HS_NULL;
	if( (irr=(ASNH225InfoRequestResponse*)(msg->alter)) == HS_NULL )
	{
		delete_RasMsg(msg);
		HSFree(msg);
		return HS_NULL;
	}

	irr->m_requestSeqNum.inheritance.value = pMsgSN;

	tRet = MakeIRR_perCallInfoOne(&(irr->m_perCallInfo), pCall, HS_NULL, HS_NO);
	if( tRet != HS_OK )
	{
		delete_RasMsg(msg);
		HSFree(msg);
		return HS_NULL;
	}

	return msg;
}


/* IRR message by IRQ that include '0' CRV
*/
RasMsg *MakeIRRwithAll( IEndpoint *pEndpoint, HS_USHORT pMsgSN, NoLockList *pCalls )
{
	HS_RESULT tRet = HS_OK;
	RasMsg *msg = HS_NULL;
	ASNH225InfoRequestResponse *irr = HS_NULL;

	if( pEndpoint == HS_NULL || pCalls == HS_NULL ) return HS_NULL;
	if( (msg=MakeIRR(pEndpoint)) == HS_NULL ) return HS_NULL;
	if( (irr=(ASNH225InfoRequestResponse*)(msg->alter)) == HS_NULL )
	{
		delete_RasMsg(msg);
		HSFree(msg);
		return HS_NULL;
	}

	irr->m_requestSeqNum.inheritance.value = pMsgSN;

	tRet = MakeIRR_perCallInfoAll(&(irr->m_perCallInfo), pCalls);
	if( tRet != HS_OK )
	{
		delete_RasMsg(msg);
		HSFree(msg);
		return HS_NULL;
	}

	return msg;
}


/* IRR message by UUIEsRequest Field on ACF
*/
RasMsg *MakeIRRwithPDU( IEndpoint *pEndpoint, ICall *pCall, ASNH225H323_UU_PDU *pPdu, BOOL pIsSent, HS_USHORT *pSN )
{
	HS_USHORT tSN = 0;
	HS_RESULT tRet = HS_OK;
	RasMsg *msg = HS_NULL;
	ASNH225InfoRequestResponse *irr = HS_NULL;

	if( pEndpoint==HS_NULL || pCall==HS_NULL || pPdu==HS_NULL || pSN==HS_NULL ) return HS_NULL;
	if( (msg=MakeIRR(pEndpoint)) == HS_NULL ) return HS_NULL;
	if( (irr=(ASNH225InfoRequestResponse*)(msg->alter)) == HS_NULL )
	{
		delete_RasMsg(msg);
		HSFree(msg);
		return HS_NULL;
	}

	tSN = IEndpoint_MakeRasRequestSequenceNumber( pEndpoint );
	if( tSN == 0 )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	if( pSN != HS_NULL ) *pSN = tSN;
	irr->m_requestSeqNum.inheritance.value = tSN;

	tRet = MakeIRR_perCallInfoOne(&(irr->m_perCallInfo), pCall, pPdu, pIsSent);
	if( tRet != HS_OK )
	{
		delete_RasMsg(msg);
		HSFree(msg);
		return HS_NULL;
	}

	return msg;
}


/* IRR message by irrFrequency Field on ACF
*/
RasMsg *MakeIRRwithTimer( IEndpoint *pEndpoint, ICall *pCall, HS_USHORT *pSN )
{
	HS_USHORT tSN = 0;
	HS_RESULT tRet = HS_OK;
	RasMsg *msg = HS_NULL;
	ASNH225InfoRequestResponse *irr = HS_NULL;

	if( pEndpoint==HS_NULL || pCall==HS_NULL || pSN==HS_NULL ) return HS_NULL;
	if( (msg=MakeIRR(pEndpoint)) == HS_NULL ) return HS_NULL;
	if( (irr=(ASNH225InfoRequestResponse*)(msg->alter)) == HS_NULL )
	{
		delete_RasMsg(msg);
		HSFree(msg);
		return HS_NULL;
	}

	tSN = IEndpoint_MakeRasRequestSequenceNumber( pEndpoint );
	if( tSN == 0 )
	{
		delete_ASNH225RasMessage( msg );
		HSFree( msg );
		return HS_NULL;
	}

	if( pSN != HS_NULL ) *pSN = tSN;
	irr->m_requestSeqNum.inheritance.value = tSN;

	tRet = MakeIRR_perCallInfoOne(&(irr->m_perCallInfo), pCall, HS_NULL, HS_NO);
	if( tRet != HS_OK )
	{
		delete_RasMsg(msg);
		HSFree(msg);
		return HS_NULL;
	}

	return msg;
}


