/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<RtpSnder.c>	2006-01-11,22:39

*/

#include <time.h>

#ifdef _WIN32_WCE
#include "codec/Typedef.h"
#include "codec/va_g729ab_intf.h"
#include "codec/OS_depend_func.h"
#else
#include <process.h>
#include "codec/va_g729.h"
#endif

#include "codec/g711.h"
#include "codec/gsm/gsm.h"
#include "codec/iLBC/iLBC_encode.h"

#include "RtpSnder.h"





typedef struct  iLBC_Enc_Inst_t_*	iLBCEnc;





/*
 *
 * struct for RFC 2833 outbound dtmf profile.
 *
 */
HS_RESULT new_IRfc2833(void *pObject,HS_UCHAR pSignal)
{
	IRfc2833 *pObj = (IRfc2833*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	pObj->mSignal = pSignal;
	pObj->mDuration = 0;
	pObj->mCount = 0;
	pObj->mCountMax = HS_RFC2833_COUNT_MAX;
	return HS_OK;
}


IRfc2833 *newm_IRfc2833(HS_UCHAR pSignal)
{
	IRfc2833 *tResult = NULL;

	if( (tResult=(IRfc2833*)HSMalloc(sizeof(IRfc2833)))==NULL ) return NULL;
	if( new_IRfc2833(tResult,pSignal) != HS_OK )
	{
		HSFree(tResult);
		return NULL;
	}

	return tResult;
}


HS_RESULT delete_IRfc2833(void *pObject)
{
	IRfc2833 *pObj = (IRfc2833*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	return HS_OK;
}


HS_RESULT deletem_IRfc2833(void *pObject)
{
	IRfc2833 *pObj = (IRfc2833*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	delete_IRfc2833(pObj);
	HSFree(pObj);
	return HS_OK;
}


HS_RESULT IRfc2833_Encode(void *pObject,HS_UINT pInterval,HS_UCHAR *pData)
{
	HS_UINT tRemain = 0;
	HS_USHORT tDuration;
	IRfc2833 *pObj = (IRfc2833*)pObject;

	if( pObj==NULL || pData==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mCount >= pObj->mCountMax ) return HS_ERR;
	tRemain = pObj->mCountMax - pObj->mCount;

	pData[1] = HS_RFC2833_TYPE;
	if( pObj->mCount==0 ) pData[1] |= 0x80;/*marker bit*/

	pData[HS_RTP_HEADER_SIZE] = pObj->mSignal;

	if( tRemain < 5 ) pData[HS_RTP_HEADER_SIZE+1] = 0x3c;
	else			  pData[HS_RTP_HEADER_SIZE+1] = 0x03;
	if( tRemain < 4 ) pData[HS_RTP_HEADER_SIZE+1] |= 0x80;/*end of event*/

	tDuration = htons(pObj->mDuration);
	memcpy(pData+HS_RTP_HEADER_SIZE+2,(HS_UCHAR*)(&tDuration),2);
	if( tRemain > 3 ) pObj->mDuration += pInterval;

	pObj->mCount++;
	return HS_OK;
}





/*
 *
 * RtpSnder member functions
 *
 */
HS_RESULT new_RtpSnder(void *pObject,HS_UINT pHertz,HS_UINT pFpp)
{
	RtpSnder *pObj = (RtpSnder*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	pObj->mSocket = INVALID_SOCKET;
	memset(&(pObj->mRemoteIn),0,sizeof(pObj->mRemoteIn));
	pObj->mRemoteIn.sin_family = AF_MAX;

	pObj->mMyQ = HS_INVALID_QID;

	pObj->mHertz = pHertz;
	pObj->mType = e_RtpPayloadTypeMax;
	pObj->mFpp = pFpp;
	pObj->mSleep = 5;
	pObj->mSeqNum = 124;
	pObj->mTimeStamp = 1120;

	/* srand(0); 'at HSBase' */
	pObj->mSsrc = 771120000 + (rand()%10000);
	new_UcharRing(&(pObj->mRing),HS_RING_SIZE);

	pObj->mRfc2833 = NULL;
	new_NoLockList(&(pObj->mRfc2833s),delete_IRfc2833);
	pObj->mDtmfInband = NULL;
	new_NoLockList(&(pObj->mDtmfInbands),delete_RtpDtmf);

	pObj->mHold = FALSE;

#ifdef _WIN32_WCE
	pObj->m729Handle = NULL;
#endif
	pObj->mGsm = NULL;
	pObj->mIlbc = NULL;
	return HS_OK;
}


RtpSnder *newm_RtpSnder(HS_UINT pHertz,HS_UINT pFpp)
{
	RtpSnder *tResult = NULL;

	if( (tResult=(RtpSnder*)HSMalloc(sizeof(RtpSnder)))==NULL )
		return NULL;
	if( new_RtpSnder(tResult,pHertz,pFpp) != HS_OK )
	{
		HSFree(tResult);
		return NULL;
	}
	return tResult;
}


HS_RESULT delete_RtpSnder(void *pObject)
{
	RtpSnder *pObj = (RtpSnder*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	if( pObj->mSocket != INVALID_SOCKET )
	{
		closesocket(pObj->mSocket);
		pObj->mSocket = INVALID_SOCKET;
	}

	delete_UcharRing(&(pObj->mRing));
	
	if( pObj->mRfc2833 != NULL )
	{
		deletem_IRfc2833(pObj->mRfc2833);
		pObj->mRfc2833 = NULL;
	}
	delete_NoLockList(&(pObj->mRfc2833s));
	delete_NoLockList(&(pObj->mDtmfInbands));

	return HS_OK;
}


HS_RESULT deletem_RtpSnder(void *pObject)
{
	RtpSnder *pObj = (RtpSnder*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	delete_RtpSnder(pObj);
	HSFree(pObj);
	return HS_OK;
}


HS_RESULT RtpSnder_OpenCodec(void *pObject)
{
	RtpSnder *pObj = (RtpSnder*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mType)
	{
		case e_RtpPayloadType_Gsm:
			if( (pObj->mGsm=gsm_create())==NULL )
				return HS_ERR;
			break;
		case e_RtpPayloadType_G729:
#ifdef _WIN32_WCE
			if( (pObj->m729Handle=E_IF_g729ab_init())==NULL )
				return HS_ERR;
#else
			va_g729a_init_encoder();
#endif
			break;
		case e_RtpPayloadType_iLbc:
			if( (pObj->mIlbc=(void*)HSMalloc((unsigned)sizeof(struct iLBC_Enc_Inst_t_)))==NULL )
				return HS_ERR;
			initEncode((iLBCEnc)(pObj->mIlbc),30/*ms*/); 
			break;
		default:
			break;
	}

	return HS_OK;
}


HS_RESULT RtpSnder_CloseCodec(void *pObject)
{
	RtpSnder *pObj = (RtpSnder*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mType)
	{
		case e_RtpPayloadType_Gsm:
			if( pObj->mGsm != NULL )
			{
				gsm_destroy((gsm)(pObj->mGsm));
				pObj->mGsm = NULL;
			}
			break;
		case e_RtpPayloadType_G729:
#ifdef _WIN32_WCE
			if( pObj->m729Handle != NULL )
			{
				E_IF_g729ab_exit(pObj->m729Handle);
				pObj->m729Handle = NULL;
			}
#endif
			break;
		case e_RtpPayloadType_iLbc:
			if( pObj->mIlbc != NULL )
			{
				HSFree(pObj->mIlbc);
				pObj->mIlbc = NULL;
			}
			break;
		default:
			break;
	}

	return HS_OK;
}


HS_UINT RtpSnder_GetPcmSize(void *pObject)
{
	RtpSnder *pObj = (RtpSnder*)pObject;

	if( pObj==NULL ) return 0;

	switch(pObj->mType)
	{
		case e_RtpPayloadType_Pcmu:
		case e_RtpPayloadType_Pcma:
			return HS_G711_PCM_SIZE;
		case e_RtpPayloadType_Gsm:
			return HS_GSM_PCM_SIZE;
		case e_RtpPayloadType_G729:
			return HS_G729_PCM_SIZE;
		case e_RtpPayloadType_iLbc:
			return HS_ILBC_PCM_SIZE;
		default:
			break;
	}
	return 0;
}


HS_RESULT RtpSnder_MakeRtpHeader(void *pObject,HS_UCHAR *pHeader)
{
	RtpSnder *pObj = (RtpSnder*)pObject;

	if( pObj==NULL || pHeader==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mType)
	{
		case e_RtpPayloadType_Pcmu:
		case e_RtpPayloadType_Pcma:
			pObj->mTimeStamp += (pObj->mFpp*HS_G711_TIMESTAMP);
			break;
		case e_RtpPayloadType_Gsm:
			pObj->mTimeStamp += (pObj->mFpp*HS_GSM_TIMESTAMP);
			break;
		case e_RtpPayloadType_G729:
			pObj->mTimeStamp += (pObj->mFpp*HS_G729_TIMESTAMP);
			break;
		case e_RtpPayloadType_iLbc:
			pObj->mTimeStamp += (pObj->mFpp*HS_ILBC_TIMESTAMP);
			break;
		default:
			return HS_ERR_NO_EXIST;
	}
	pObj->mSeqNum++;

	pHeader[0]  =0x80; /*RTP Version 2*/
	pHeader[1]	=(HS_UCHAR)   pObj->mType;
	pHeader[2]	=(HS_UCHAR) ((pObj->mSeqNum)>>8);
	pHeader[3]	=(HS_UCHAR) ((pObj->mSeqNum));
	pHeader[4]	=(HS_UCHAR) ((pObj->mTimeStamp)>>24);
	pHeader[5]	=(HS_UCHAR)(((pObj->mTimeStamp)<<8)>>24);
	pHeader[6]	=(HS_UCHAR)(((pObj->mTimeStamp)<<16)>>24);
	pHeader[7]	=(HS_UCHAR)(((pObj->mTimeStamp)<<24)>>24);
	pHeader[8]	=(HS_UCHAR) ((pObj->mSsrc)>>24);
	pHeader[9]	=(HS_UCHAR)(((pObj->mSsrc)<<8)>>24);
	pHeader[10]	=(HS_UCHAR)(((pObj->mSsrc)<<16)>>24);
	pHeader[11]	=(HS_UCHAR)(((pObj->mSsrc)<<24)>>24);

	return HS_OK;
}


HS_UINT RtpSnder_Encode(void *pObject,HS_UCHAR *pOut,HS_UCHAR *pIn,HS_UINT pSize)
{
	HS_SHORT *tShort = NULL;
	HS_UINT i, tFrames, tResult = 0;
	RtpSnder *pObj = (RtpSnder*)pObject;

#ifdef _WIN32_WCE
	HS_UINT tT = 11;
	HS_UCHAR tBuffer[128];
#endif

	if( pObj==NULL || pOut==NULL || pIn==NULL || pSize==0 ) return 0;

	switch(pObj->mType)
	{
		case e_RtpPayloadType_Pcmu:
		case e_RtpPayloadType_Pcma:
			if( (tFrames=pSize/HS_G711_PCM_SIZE)==0 )
				return 0;
			pSize = tFrames * HS_G711_PCM_SIZE;
			break;
		case e_RtpPayloadType_Gsm:
			if( (tFrames=pSize/HS_GSM_PCM_SIZE)==0 )
				return 0;
			pSize = tFrames * HS_GSM_PCM_SIZE;
			break;
		case e_RtpPayloadType_G729:
			if( (tFrames=pSize/HS_G729_PCM_SIZE)==0 )
				return 0;
			pSize = tFrames * HS_G729_PCM_SIZE;
			break;
		case e_RtpPayloadType_iLbc:
			if( (tFrames=pSize/HS_ILBC_PCM_SIZE)==0 )
				return 0;
			pSize = tFrames * HS_ILBC_PCM_SIZE;
			break;
		default:
			return 0;
	}

	switch(pObj->mType)
	{
		case e_RtpPayloadType_Pcmu:
		case e_RtpPayloadType_Pcma:
			tResult = tFrames * HS_G711_PKT_SIZE;
			if( G711Encode(pIn,pSize,pOut,&((int)tResult),pObj->mType) != 0 )
				return 0;
			break;
		case e_RtpPayloadType_Gsm:
			tResult = 0;
			if( pObj->mGsm==NULL ) return 0;
			for(i=0; i<tFrames; i++)
			{
				gsm_encode((gsm)(pObj->mGsm),(short*)(pIn+i*HS_GSM_PCM_SIZE),pOut+tResult);
				tResult += HS_GSM_PKT_SIZE;
			}
			break;
		case e_RtpPayloadType_G729:
			tResult = 0;
#ifdef _WIN32_WCE
			if( pObj->m729Handle==NULL ) return 0;
#endif
			for(i=0; i<tFrames; i++)
			{
#ifdef _WIN32_WCE
				if( E_IF_g729ab_encode(pObj->m729Handle,(short*)(pIn+i*HS_G729_PCM_SIZE),tBuffer,&tT,0) != 0) continue;
				if( tT != 11 ) continue;
				memcpy(pOut+tResult,tBuffer+1,tT-1);
				tResult += (tT-1);
#else
				va_g729a_encoder((short*)(pIn+i*HS_G729_PCM_SIZE),pOut+tResult);
				tResult += HS_G729_PKT_SIZE;
#endif
			}
			break;
		case e_RtpPayloadType_iLbc:
		{
			int j;
			float tBlock[BLOCKL_MAX];
			iLBCEnc tEnc = (iLBCEnc)(pObj->mIlbc);

			tResult = 0;
			if( tEnc==NULL ) return 0;

			tShort = (short*)pIn;
			for(j=0; j<tEnc->blockl; j++)
				tBlock[j] = (float)tShort[j];

			iLBC_encode(pOut,tBlock,tEnc);
			tResult = tEnc->no_of_bytes;
			break;
		}
		default:
			return 0;
	}

	return tResult;
}


/* Temp global variable 'gDtmfInbandCountMax,gDTmfInbandCountPos'
   In fact, Inband DTMF signal must send until pop off the button.
   But this program decide that Inband DTMF signal send for HS_DTMF_INBAND_DUR sec duration.
   So, we need the variable for counting duration.
*/
#define HS_DTMF_INBAND_DUR		200/*ms*/
HS_UINT gDtmfInbandCountMax = 0;
HS_UINT gDtmfInbandCountPos = 0;
HS_RESULT RtpSnder_CheckDtmfInband(void *pObject,HS_UCHAR *pData,HS_UINT pSize)
{
	RtpSnder *pObj = (RtpSnder*)pObject;

	if( pObj==NULL || pData==NULL ) return HS_ERR_NULL_PARAM;

	if( pObj->mDtmfInband==NULL )
	{
		if( (pObj->mDtmfInband = NoLockList_DetachData(&(pObj->mDtmfInbands),0))!=NULL )
		{
			gDtmfInbandCountPos = 0;
			switch(pObj->mType)
			{
				case e_RtpPayloadType_Pcmu:
				case e_RtpPayloadType_Pcma:
					gDtmfInbandCountMax = HS_DTMF_INBAND_DUR/(HS_G711_TIMEGAP*(pObj->mFpp));
					break;
				case e_RtpPayloadType_Gsm:
					gDtmfInbandCountMax = HS_DTMF_INBAND_DUR/(HS_GSM_TIMEGAP*(pObj->mFpp));
					break;
				case e_RtpPayloadType_G729:
					gDtmfInbandCountMax = HS_DTMF_INBAND_DUR/(HS_G729_TIMEGAP*(pObj->mFpp));
					break;
				case e_RtpPayloadType_iLbc:
					gDtmfInbandCountMax = HS_DTMF_INBAND_DUR/(HS_ILBC_TIMEGAP*(pObj->mFpp));
					break;
				default:
					gDtmfInbandCountMax = 10;
					break;
			}
		}
	}

	if( pObj->mDtmfInband!=NULL )
	{
		gDtmfInbandCountPos++;
		RtpDtmf_Generate(pObj->mDtmfInband,pData,pSize);

		if( gDtmfInbandCountPos >= gDtmfInbandCountMax )
		{
			deletem_RtpDtmf(pObj->mDtmfInband);
			pObj->mDtmfInband = NULL;
		}
	}

	return HS_OK;
}


HS_RESULT RtpSnder_CheckDtmfRfc2833(void *pObject,HS_UCHAR *pBuffer)
{
	HS_RESULT tRet = HS_ERR;
	RtpSnder *pObj = (RtpSnder*)pObject;

	if( pObj==NULL || pBuffer==NULL ) return HS_ERR_NULL_PARAM;

	if( pObj->mRfc2833==NULL )
		pObj->mRfc2833 = NoLockList_DetachData(&(pObj->mRfc2833s),0);
	if( pObj->mRfc2833!=NULL )
	{
		if( IRfc2833_Encode(pObj->mRfc2833,240,pBuffer)==HS_OK )
		{
			RtpSnder_Send(pObj,pBuffer,HS_RTP_HEADER_SIZE+4);
			tRet = HS_OK;
		}

		if( pObj->mRfc2833->mCount >= pObj->mRfc2833->mCountMax )
		{
			deletem_IRfc2833(pObj->mRfc2833);
			pObj->mRfc2833 = NULL;
		}
	}

	return tRet;
}





/*
 *
 * RtpSnder thread functions
 *
 */
void RtpSnder_Demon(void *pObject)
{
	MSG tQmsg;
	BOOL tThreadOut = FALSE;
	WAVEHDR *tWaveHeader = NULL;
	RtpSnder *pObj = (RtpSnder*)pObject;

	HS_UINT tLen;
	HS_UCHAR tBuffer[1024];
	HS_UINT tReadSize = 0;
	HS_UCHAR *tRead = NULL;

	if( pObj==NULL ) return;
	if( RtpSnder_OpenCodec(pObj) != HS_OK ) return;
	if( (tReadSize=RtpSnder_GetPcmSize(pObj)*(pObj->mFpp))==0 ) return;

	while(tThreadOut==FALSE)
	{
		if( GetMessage(&tQmsg,NULL,HS_QM,HS_QM_RTP_MAX) )
		{
            switch(tQmsg.message)
			{
				case HS_QM_STOP:
					tThreadOut = TRUE;
					break;
				case HS_QM_CHECK:
					HSPrint("\nthread checking> RtpSnder-live");
					break;
				case HS_QM_HOLD:
					pObj->mHold = TRUE;
					break;
				case HS_QM_RESUME:
					pObj->mHold = FALSE;
					break;
				case HS_QM_PCM:
					if( (tWaveHeader=(WAVEHDR*)(tQmsg.lParam))==NULL )
						break;
					UcharRing_ForcePut(&(pObj->mRing),tWaveHeader->lpData,tWaveHeader->dwBufferLength);
					DestroyWaveHeader(tWaveHeader);
					tWaveHeader=NULL;
					break;
				case HS_QM_DTMF_INBAND:
					NoLockList_AttachData(
						&(pObj->mDtmfInbands),
						newm_RtpDtmf(
							(HS_UCHAR)(tQmsg.lParam),pObj->mHertz,(DtmfSignal)(tQmsg.wParam)
						)
					);
					break;
				case HS_QM_DTMF_RFC2833:
					NoLockList_AttachData(&(pObj->mRfc2833s),newm_IRfc2833((HS_UCHAR)(tQmsg.lParam)));
					break;
			}
		}

		while( (tRead=UcharRing_Get(&(pObj->mRing),tReadSize))!=NULL )
		{
			RtpSnder_MakeRtpHeader(pObj,tBuffer);

			/* check RFC 2833 DTMF signal.
			*/
			if( RtpSnder_CheckDtmfRfc2833(pObj,tBuffer)==HS_OK )
			{
				HSFree(tRead);
				continue;
			}

			/* check DTMF In-band signal
			*/
			RtpSnder_CheckDtmfInband(pObj,tRead,tReadSize);

			/* encode RTP payload
			*/
			if( (tLen=RtpSnder_Encode(pObj,tBuffer+HS_RTP_HEADER_SIZE,tRead,tReadSize)) > 0 )
				RtpSnder_Send(pObj,tBuffer,tLen+HS_RTP_HEADER_SIZE);

			HSFree(tRead);
		}
	}

	RtpSnder_CloseCodec(pObj);
	return;
}


unsigned __stdcall RtpSnder_Thread(void* pArg)
{
	RtpSnder *pObj = (RtpSnder*)pArg;

	/* check validation
	*/
	if( pObj==NULL )
	{
#ifdef _WIN32_WCE
		ExitThread(0);
#else
		_endthreadex(0);
#endif
		return 0;
	}
	if( pObj->mSocket == INVALID_SOCKET )
	{
#ifdef _WIN32_WCE
		ExitThread(0);
#else
		_endthreadex(0);
#endif
		return 0;
	}
	if( pObj->mRemoteIn.sin_family == AF_MAX )
	{
#ifdef _WIN32_WCE
		ExitThread(0);
#else
		_endthreadex(0);
#endif
		return 0;
	}

	/* thread
	*/
	RtpSnder_Demon(pObj);

	deletem_RtpSnder(pObj);
	HSPrint("\nRtpSnder-End");

#ifdef _WIN32_WCE
	ExitThread(0);
#else
	_endthreadex(0);
#endif
	return 0;
}


HS_QID RtpSnder_Start(void *pObject)
{
	RtpSnder *pObj = (RtpSnder*)pObject;
	if( pObj==NULL ) return HS_INVALID_QID;

#ifdef _WIN32_WCE
	CreateThread(NULL,0,&RtpSnder_Thread,pObj,0,&(pObj->mMyQ));
#else
	_beginthreadex(NULL,0,&RtpSnder_Thread,pObj,0,&(pObj->mMyQ));
#endif
	return pObj->mMyQ;
}


HS_RESULT RtpSnder_Stop(void *pObject)
{
	RtpSnder *pObj = (RtpSnder*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_STOP,(WPARAM)0,(LPARAM)NULL);
	return HS_OK;
}


HS_RESULT RtpSnder_Send(void *pObject,HS_UCHAR *pBuffer,HS_UINT pSize)
{
	RtpSnder *pObj = (RtpSnder*)pObject;

	if( pObj==NULL || pBuffer==NULL || pSize==0 ) return HS_ERR_NULL_PARAM;
	if( pObj->mHold==TRUE ) return HS_OK;
	if( sendto(pObj->mSocket,pBuffer,pSize,0,(SOCKADDR*)&(pObj->mRemoteIn),sizeof(pObj->mRemoteIn)) > 0 )
		return HS_OK;
	return HS_ERR;
}


HS_RESULT RtpSnder_AddDtmfInband(void *pObject,HS_UCHAR pSignal,HS_UINT pVolume)
{
	RtpSnder *pObj = (RtpSnder*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_DTMF_INBAND,(WPARAM)pVolume,(LPARAM)pSignal);
	return HS_OK;
}

HS_RESULT RtpSnder_AddDtmfRfc2833(void *pObject,HS_UCHAR pSignal)
{
	RtpSnder *pObj = (RtpSnder*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_DTMF_RFC2833,(WPARAM)0,(LPARAM)pSignal);
	return HS_OK;
}


HS_RESULT RtpSnder_Hold(void *pObject)
{
	RtpSnder *pObj = (RtpSnder*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_HOLD,(WPARAM)0,(LPARAM)NULL);
	return HS_OK;
}


HS_RESULT RtpSnder_Resume(void *pObject)
{
	RtpSnder *pObj = (RtpSnder*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_RESUME,(WPARAM)0,(LPARAM)NULL);
	return HS_OK;
}


HS_RESULT RtpSnder_Check(void *pObject)
{
	RtpSnder *pObj = (RtpSnder*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_CHECK,(WPARAM)0,(LPARAM)NULL);
	return HS_OK;
}

