/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<IDialogue.c>	2005-10-06,21:48

*/

#include "IDialogue.h"

/* reference header
*/
#include "ITransaction.h"





/* MediaInfo member functions
*/
HS_RESULT new_MediaInfo(void *pObject)
{
	MediaInfo *pObj = (MediaInfo*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	pObj->mNumber = HS_UINT_MAX;
	pObj->mType = NULL;
	pObj->mStarted = FALSE;
	pObj->mLocalRtpAddr = NULL;
	pObj->mRemoteRtpAddr = NULL;
	pObj->mLocalRtpPort = HS_INVALID_TSAP_PORT;
	pObj->mRemoteRtpPort = HS_INVALID_TSAP_PORT;
	return HS_OK;
}


HS_RESULT delete_MediaInfo(void *pObject)
{
	MediaInfo *pObj = (MediaInfo*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	pObj->mNumber = HS_UINT_MAX;
	if( pObj->mType != NULL )
	{
		HSFree(pObj->mType);
		pObj->mType = NULL;
	}
	if( pObj->mLocalRtpAddr != NULL )
	{
		HSFree(pObj->mLocalRtpAddr);
		pObj->mLocalRtpAddr = NULL;
	}
	if( pObj->mRemoteRtpAddr != NULL )
	{
		HSFree(pObj->mRemoteRtpAddr);
		pObj->mRemoteRtpAddr = NULL;
	}
	pObj->mLocalRtpPort = HS_INVALID_TSAP_PORT;
	pObj->mRemoteRtpPort = HS_INVALID_TSAP_PORT;
	return HS_OK;
}





/* SDP status manage functions
*/
SdpState SdpState_ChangeSend(SdpState pNow)
{
	switch(pNow)
	{
		case e_SdpState_Idle:
		case e_SdpState_Sent:
			return e_SdpState_Sent;
		case e_SdpState_Received:
			return e_SdpState_Idle;
	}
	return e_SdpStateMax;
}
SdpState SdpState_ChangeReceive(SdpState pNow)
{
	switch(pNow)
	{
		case e_SdpState_Idle:
		case e_SdpState_Received:
			return e_SdpState_Received;
		case e_SdpState_Sent:
			return e_SdpState_Idle;
	}
	return e_SdpStateMax;
}






/* IDialogue member functions
*/
HS_RESULT new_IDialogue(void *pObject, IUa *pUa)
{
	HS_RESULT tRet = HS_OK;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || pUa==NULL ) return HS_ERR_NULL_PARAM;

	HS_TRY( new_IDialogue_ByUser(pObj) );
	/* ! NOTE : DON'T remove IDialogue level
	*/pObj->mUa = pUa;

	if( pUa->mOutBoundProxy[0] != '\0' )
		strcpy(pObj->mTarget,pUa->mOutBoundProxy);
	else
		strcpy(pObj->mTarget,pUa->mProxy);
	pObj->mTargetPort = pUa->mProxyPort;

	NoLockList_AttachData(&(pUa->mDialogues),pObj);

	return HS_OK;
}


HS_RESULT new_IDialogue_ByUser(void *pObject)
{
	char tString[16];
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	memset(pObj->mCallId,0,MAX_NAME_SIZE);
	pObj->mCseqSeed = 1;
	pObj->mExpires = NULL;
	memset(pObj->mLocalTag,0,MAX_NAME_SIZE);
	memset(pObj->mRemoteTag,0,MAX_NAME_SIZE);
	pObj->mIsTcp = FALSE;

	pObj->mAuthorization = NULL;
	pObj->mNonceCount = 1;
	pObj->mIsAuthProxy = FALSE;
	pObj->mAuthCount = HS_SIP_MAX_AUTH_COUNT;

	pObj->mDestDisplayName = NULL;
	pObj->mDestUser = NULL;
	pObj->mDestAddress = NULL;
	pObj->mDestPort = HS_INVALID_TSAP_PORT;

	memset(pObj->mTarget,0,MAX_NAME_SIZE);
	pObj->mTargetPort = HS_INVALID_TSAP_PORT;

	pObj->mState = e_DialogueStateMax;
	new_SdpMessage(&(pObj->mSdp));
	pObj->mSdpState = e_SdpState_Idle;

	pObj->mAudio = NULL;
	pObj->mVideo = NULL;
	sprintf(tString,"123567%d",SipRand(10000));
	pObj->mSessionId = SipStringAlloc(tString);

	pObj->mUa = NULL;

	pObj->mHandle = HS_INVALID_HANDLE;
	new_NoLockList(&(pObj->mTransactions),delete_ITransaction);
	return HS_OK;
}


HS_RESULT delete_IDialogue(void *pObject)
{
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	if( pObj->mExpires != NULL )
	{
		HSFree(pObj->mExpires);
		pObj->mExpires=NULL;
	}
	if( pObj->mAuthorization != NULL )
	{
		delete_SipAuthorization(pObj->mAuthorization);
		HSFree(pObj->mAuthorization);
		pObj->mAuthorization = NULL;
	}

	if( pObj->mDestDisplayName != NULL )
	{
		HSFree(pObj->mDestDisplayName);
		pObj->mDestDisplayName = NULL;
	}
	if( pObj->mDestUser != NULL )
	{
		HSFree(pObj->mDestUser);
		pObj->mDestUser = NULL;
	}
	if( pObj->mDestAddress != NULL )
	{
		HSFree(pObj->mDestAddress);
		pObj->mDestAddress = NULL;
	}

	delete_SdpMessage(&(pObj->mSdp));
	if( pObj->mAudio != NULL )
	{
		delete_MediaInfo(pObj->mAudio);
		HSFree(pObj->mAudio);
		pObj->mAudio = NULL;
	}
	if( pObj->mVideo != NULL )
	{
		delete_MediaInfo(pObj->mVideo);
		HSFree(pObj->mVideo);
		pObj->mVideo = NULL;
	}
	if( pObj->mSessionId != NULL )
	{
		HSFree(pObj->mSessionId);
		pObj->mSessionId = NULL;
	}

	delete_NoLockList(&(pObj->mTransactions));
	return HS_OK;
}


char *IDialouge_MakeBranchId(void *pObject)
{
	char *tResult = NULL;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL ) return NULL;

	if( (tResult=(char*)HSMalloc(16))==NULL ) return NULL;
	/* [RFC-3261 8.1.1.7] magic cookie = "z9hG4bK"
	*/
	sprintf(tResult,"z9hG4bK%02x%02x%02x%d",
		SipRand(256),SipRand(256),
		SipRand(256),SipRand(100)
	);

	return tResult;
}


HS_RESULT IDialogue_MakeAuthResponse(void *pObject, char *pMethod)
{
	HASHHEX HA1;
	HASHHEX HA2 = "";
	HASHHEX Response;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || pMethod==NULL ) return HS_ERR_NULL_PARAM;

	memset(Response,0,HASHHEXLEN+1);
	DigestCalcHA1(
		pObj->mAuthorization->mAlgorithm,
		pObj->mAuthorization->mUserName,
		pObj->mAuthorization->mRealm,
		pObj->mUa->mPassword,
		pObj->mAuthorization->mNonce,
		pObj->mAuthorization->mCnonce,
		HA1
	);
	DigestCalcResponse(
		HA1,
		pObj->mAuthorization->mNonce,
		pObj->mAuthorization->mNc,
		pObj->mAuthorization->mCnonce,
		pObj->mAuthorization->mQop,
		pMethod,
		pObj->mAuthorization->mUri,
		HA2,
		Response
	);

	pObj->mAuthorization->mResponse = SipStringAlloc(Response);
	return HS_OK;
}


HS_RESULT IDialogue_MakeAuthorization(void *pObject, SipMessage *pMessage)
{
	char tString[128];
	SipContactUnit *tUnit = NULL;
	SipAuthorization *tRemoteAuth = NULL;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mUa==NULL ) return HS_ERR_NULL_PARAM;

	if( pMessage->mHeadLine.mResponseValue == e_SipResponse_Unauthorized )
	{
		/* may be REGISTER
		*/
		if( pMessage->mWWWAuthenticate==NULL )
			return HS_ERR_SIP_NO_EXIST;
		tRemoteAuth = pMessage->mWWWAuthenticate;
		pObj->mIsAuthProxy = FALSE;
	}
	else if( pMessage->mHeadLine.mResponseValue == e_SipResponse_ProxyAuthenticationRequired )
	{
		/* may be INVITE
		*/
		if( pMessage->mProxyAuthenticate==NULL )
			return HS_ERR_SIP_NO_EXIST;
		tRemoteAuth = pMessage->mProxyAuthenticate;
		pObj->mIsAuthProxy = TRUE;
	}
	else
		return HS_ERR_SIP_NOT_SUPPORT;

	/* copy authenticate data
	*/
	if( pObj->mAuthorization != NULL )
		deletem_SipAuthorization(pObj->mAuthorization);
	if( (pObj->mAuthorization=newm_SipAuthorization())==NULL )
		return HS_ERR_MALLOC;
	pObj->mAuthorization->mAlgorithm = SipStringAlloc(tRemoteAuth->mAlgorithm);
	pObj->mAuthorization->mNonce = SipStringAlloc(tRemoteAuth->mNonce);
	pObj->mAuthorization->mOpaque = SipStringAlloc(tRemoteAuth->mOpaque);
	pObj->mAuthorization->mQop = SipStringAlloc(tRemoteAuth->mQop);
	pObj->mAuthorization->mRealm = SipStringAlloc(tRemoteAuth->mRealm);
	pObj->mAuthorization->mStale = SipStringAlloc(tRemoteAuth->mStale);
	pObj->mAuthorization->mType = SipStringAlloc(tRemoteAuth->mType);

	pObj->mAuthorization->mCnonce = SipUintAlloc( SipRand(100000000) );
	sprintf(tString,"%08x",pObj->mNonceCount++);
	pObj->mAuthorization->mNc = SipStringAlloc(tString);
	sprintf(tString,"sip:%s",pObj->mUa->mProxy);
	pObj->mAuthorization->mUri = SipStringAlloc(tString);

	if( pObj->mUa->mAuthUser != NULL )
		pObj->mAuthorization->mUserName = SipStringAlloc(pObj->mUa->mAuthUser);
	else
	{
		if( (tUnit=IUa_GetPrimeContactUnit(pObj->mUa)) == NULL )
			pObj->mAuthorization->mUserName = SipStringAlloc("username");
		else
			pObj->mAuthorization->mUserName = SipStringAlloc(tUnit->mUri.mUserName);
	}

	IDialogue_MakeAuthResponse(pObj,pMessage->mCSeq->mMethod);
	return HS_OK;
}


HS_RESULT IDialogue_FindMatchMedia(void *pObject, SdpMessage *pSdp, BOOL pIsNewPort)
{
	HS_UINT i;
	ChainUnit *tChain = NULL;

	SdpAttribute *tLocalAttr = NULL;
	SdpAttribute *tRemoteAttr = NULL;
	SdpMediaDescription *tLocalDesc = NULL;
	SdpMediaDescription *tRemoteDesc = NULL;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || pSdp==NULL ) return HS_ERR_NULL_PARAM;

	/* ! NOTE :
			Except 're-INVITE' case, UA stack ignore multiple SDP response. (example 183 and 200, both have SDP response)
			In the 're-INVITE' case, UA stack make null the mAudio and mVideo already.
	*/

	/* get audio media
	*/
	if( pObj->mAudio == NULL )
	{
		if( (tLocalDesc=SdpMessage_FindAudioDescription(&(pObj->mSdp))) != NULL &&
			(tRemoteDesc=SdpMessage_FindAudioDescription(pSdp)) != NULL
		)
		{
			tChain = tLocalDesc->mAttributes.units;
			for( i=0; i<tLocalDesc->mAttributes.size; i++ )
			{
				if( tChain==NULL ) return HS_ERR_CONFLICT;
				if( (tLocalAttr=(SdpAttribute*)(tChain->data))==NULL ) return HS_ERR_CONFLICT;

				tRemoteAttr = SdpMediaDescription_FindAttribute(tRemoteDesc,tLocalAttr->mAttrNumber,strlen(tLocalAttr->mAttrNumber));

				if( tRemoteAttr != NULL )
				{
					if( (pObj->mAudio=(MediaInfo*)HSMalloc(sizeof(MediaInfo)))==NULL )
						return HS_ERR_MALLOC;
					new_MediaInfo(pObj->mAudio);
					pObj->mAudio->mType = SipStringAlloc("audio");
					if( tRemoteAttr->mAttrNumber != NULL )
						pObj->mAudio->mNumber = atoi(tRemoteAttr->mAttrNumber);
					pObj->mAudio->mLocalRtpAddr = SipStringAlloc(pObj->mUa->mStack->mLocalIp);

					if( pIsNewPort==TRUE )
						pObj->mAudio->mLocalRtpPort = IStack_MakeMediaTsapPort(pObj->mUa->mStack);
					else
					{
						if( tLocalDesc->mTsapPort != NULL )
							pObj->mAudio->mLocalRtpPort = (HS_USHORT)atoi(tLocalDesc->mTsapPort);
					}
					pObj->mAudio->mRemoteRtpAddr = SipStringAlloc(pSdp->mConnection.mAddr);

					if( tRemoteDesc->mTsapPort != NULL )
						pObj->mAudio->mRemoteRtpPort = atoi(tRemoteDesc->mTsapPort);
					break;
				}

				tChain = (ChainUnit*)(tChain->next);
			}
		}
	}

	/* get video media
	*/
	if( pObj->mVideo == NULL )
	{
		if( (tLocalDesc=SdpMessage_FindVideoDescription(&(pObj->mSdp))) != NULL &&
			(tRemoteDesc=SdpMessage_FindVideoDescription(pSdp)) != NULL
		)
		{
			tChain = tLocalDesc->mAttributes.units;
			for( i=0; i<tLocalDesc->mAttributes.size; i++ )
			{
				if( tChain==NULL ) return HS_ERR_CONFLICT;
				if( (tLocalAttr=(SdpAttribute*)(tChain->data))==NULL ) return HS_ERR_CONFLICT;

				tRemoteAttr = SdpMediaDescription_FindAttribute(tRemoteDesc,tLocalAttr->mAttrNumber,strlen(tLocalAttr->mAttrNumber));

				if( tRemoteAttr != NULL )
				{
					if( (pObj->mVideo=(MediaInfo*)HSMalloc(sizeof(MediaInfo)))==NULL )
						return HS_ERR_MALLOC;
					new_MediaInfo(pObj->mVideo);
					pObj->mVideo->mType = SipStringAlloc("video");
					if( tRemoteAttr->mAttrNumber != NULL )
						pObj->mVideo->mNumber = atoi(tRemoteAttr->mAttrNumber);
					pObj->mVideo->mLocalRtpAddr = SipStringAlloc(pObj->mUa->mStack->mLocalIp);

					if( pIsNewPort==TRUE )
						pObj->mVideo->mLocalRtpPort = IStack_MakeMediaTsapPort(pObj->mUa->mStack);
					else
					{
						if( tLocalDesc->mTsapPort != NULL )
							pObj->mVideo->mLocalRtpPort = (HS_USHORT)atoi(tLocalDesc->mTsapPort);
					}
					pObj->mVideo->mRemoteRtpAddr = SipStringAlloc(pSdp->mConnection.mAddr);
					if( tRemoteDesc->mTsapPort != NULL )
						pObj->mVideo->mRemoteRtpPort = atoi(tRemoteDesc->mTsapPort);
					break;
				}

				tChain = (ChainUnit*)(tChain->next);
			}
		}
	}

	if( pObj->mAudio==NULL && pObj->mVideo==NULL )
		return HS_ERR_SIP_NO_EXIST;
	return HS_OK;
}


SdpMessage *IDialogue_MakeSdpAnswer(void *pObject)
{
	char tString[512];
	HS_UINT tSessionId = 0;
	SdpMessage *tResult = NULL;
	SipContactUnit *tContactUnit = NULL;
	SdpAttribute *tAttribute = NULL;
	SdpMediaDescription *tMediaDescription = NULL;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL ) return NULL;
	if( pObj->mSessionId==NULL ) return NULL;
	if( pObj->mAudio==NULL && pObj->mVideo==NULL ) return NULL;
	if( (tContactUnit=IUa_GetPrimeContactUnit(pObj->mUa))==NULL ) return NULL;
	if( (tResult=(SdpMessage*)HSMalloc(sizeof(SdpMessage)))==NULL ) return NULL;
	if( new_SdpMessage(tResult) != HS_OK )
	{
		HSFree(tResult);
		return NULL;
	}

	tResult->mVersion = SipStringAlloc("0");
	sprintf(tString,"%s %s %s IN IP4 %s",
		tContactUnit->mUri.mUserName,
		pObj->mSessionId, pObj->mSessionId,
		tContactUnit->mUri.mAddress
	);
	tResult->mOwner = SipStringAlloc(tString);
	tResult->mSessionName = SipStringAlloc("-");
	tResult->mConnection.mAddr = SipStringAlloc(pObj->mUa->mStack->mLocalIp);
	tResult->mConnection.mAddrType = SipStringAlloc("IP4");
	tResult->mConnection.mNetworkType = SipStringAlloc("IN");

	/* make audio description
	*/
	if( pObj->mAudio != NULL )
	{
		if( (tMediaDescription=(SdpMediaDescription*)HSMalloc(sizeof(SdpMediaDescription)))==NULL )
		{
			delete_SdpMessage(tResult);
			HSFree(tResult);
			return NULL;
		}
		new_SdpMediaDescription(tMediaDescription);
		if( (tAttribute=(SdpAttribute*)HSMalloc(sizeof(SdpAttribute)))==NULL )
		{
			delete_SdpMessage(tResult);
			HSFree(tResult);
			delete_SdpMediaDescription(tMediaDescription);
			HSFree(tMediaDescription);
			return NULL;
		}
		new_SdpAttribute(tAttribute);
		SdpAttribute_SetAttrNumber(tAttribute,pObj->mAudio->mNumber);
		tMediaDescription->mMediaType = SipStringAlloc(pObj->mAudio->mType);
		tMediaDescription->mTsapPort = SipUintAlloc((HS_UINT)(pObj->mAudio->mLocalRtpPort));

		NoLockList_AttachData(&(tMediaDescription->mAttributes),tAttribute);
		NoLockList_AttachData(&(tResult->mDescriptions),tMediaDescription);
	}

	/* make video description
	*/
	if( pObj->mVideo != NULL )
	{
		if( (tMediaDescription=(SdpMediaDescription*)HSMalloc(sizeof(SdpMediaDescription)))==NULL )
		{
			delete_SdpMessage(tResult);
			HSFree(tResult);
			return NULL;
		}
		new_SdpMediaDescription(tMediaDescription);
		if( (tAttribute=(SdpAttribute*)HSMalloc(sizeof(SdpAttribute)))==NULL )
		{
			delete_SdpMessage(tResult);
			HSFree(tResult);
			delete_SdpMediaDescription(tMediaDescription);
			HSFree(tMediaDescription);
			return NULL;
		}
		new_SdpAttribute(tAttribute);
		SdpAttribute_SetAttrNumber(tAttribute,pObj->mVideo->mNumber);
		tMediaDescription->mMediaType = SipStringAlloc(pObj->mVideo->mType);
		tMediaDescription->mTsapPort = SipUintAlloc((HS_UINT)(pObj->mVideo->mLocalRtpPort));

		NoLockList_AttachData(&(tMediaDescription->mAttributes),tAttribute);
		NoLockList_AttachData(&(tResult->mDescriptions),tMediaDescription);
	}

	return tResult;
}


HS_RESULT IDialogue_ChangeTarget(void *pObject, IUa *pUa, SipMessage *pMessage)
{
	IDialogue *pObj = (IDialogue*)pObject;
	SipContactUnit *tContactUnit = NULL;

	if( pObj==NULL || pUa==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;
	if( pUa->mOutBoundProxy[0] != '\0' ) return HS_OK;

	if( (tContactUnit=SipRecordRoutes_GetPrimeContactUnit(&(pMessage->mRecordRoutes))) == NULL )
	{
		if( (tContactUnit=SipContacts_GetPrimeContactUnit(&(pMessage->mContacts))) == NULL )
		return HS_OK;
	}

	if( tContactUnit->mUri.mAddress==NULL )
		return HS_ERR_CONFLICT;
	strcpy(pObj->mTarget,tContactUnit->mUri.mAddress);

	if( tContactUnit->mUri.mPort!=NULL )
		pObj->mTargetPort = (HS_USHORT)atoi(tContactUnit->mUri.mPort);
	else
		pObj->mTargetPort = 5060;

	return HS_OK;
}





/* finding functions
*/
ChainUnit *IDialogue_FindTransactionChainById(void *pObject, char *pBranchId, char *pMethod, char *pNum)
{
	HS_UINT i;
	ChainUnit *tChain = NULL;
	ITransaction *tResult = NULL;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || pBranchId==NULL || pMethod==NULL || pNum==NULL ) return NULL;

	tChain = pObj->mTransactions.units;
	for( i=0; i<pObj->mTransactions.size; i++ )
	{
		if( tChain==NULL ) return NULL;
		if( (tResult=(ITransaction*)(tChain->data))==NULL ) return NULL;
		if( !strcmp(tResult->mBranchId,pBranchId) &&
			( !strcmp(tResult->mMethod,pMethod) || !strcmp("ACK",pMethod) || !strcmp("PRACK",pMethod) ) &&
			tResult->mCseq == (HS_UINT)atoi(pNum)
		) return tChain;

		tChain = (ChainUnit*)(tChain->next);
	}

	return NULL;
}


ChainUnit *IDialogue_FindTransactionChainByHandle(void *pObject, HS_TR_HANDLE pHandle)
{
	HS_UINT i;
	ChainUnit *tChain = NULL;
	ITransaction *tResult = NULL;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || pHandle==HS_INVALID_HANDLE ) return NULL;

	tChain = pObj->mTransactions.units;
	for( i=0; i<pObj->mTransactions.size; i++ )
	{
		if( tChain==NULL ) return NULL;
		if( (tResult=(ITransaction*)(tChain->data))==NULL ) return NULL;
		if( tResult->mHandle==pHandle ) return tChain;

		tChain = (ChainUnit*)(tChain->next);
	}

	return NULL;
}


void *IDialogue_FindTransactionById(void *pObject, char *pBranchId, char *pMethod, char *pNum)
{
	ChainUnit *tChain = IDialogue_FindTransactionChainById(pObject,pBranchId,pMethod,pNum);

	if( tChain==NULL ) return NULL;
	return (tChain->data);
}


void *IDialogue_FindTransactionByHandle(void *pObject, HS_TR_HANDLE pHandle)
{
	IDialogue *pObj = (IDialogue*)pObject;
	ChainUnit *tChain = IDialogue_FindTransactionChainByHandle(pObject,pHandle);

	if( tChain==NULL ) return NULL;
	return (tChain->data);
}


void *IDialogue_FindTransactionByMethod(void *pObject, char *pMethod)
{
	HS_UINT i;
	ChainUnit *tChain = NULL;
	ITransaction *tResult = NULL;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || pMethod==NULL ) return NULL;

	tChain = pObj->mTransactions.units;
	for( i=0; i<pObj->mTransactions.size; i++ )
	{
		if( tChain==NULL ) return NULL;
		if( (tResult=(ITransaction*)(tChain->data))==NULL ) return NULL;
		if( !strcmp(tResult->mMethod,pMethod) )
			return tResult;

		tChain = (ChainUnit*)(tChain->next);
	}

	return NULL;
}


void *IDialogue_FindActiveTransactionByMethod(void *pObject, char *pMethod)
{
	HS_UINT i;
	ChainUnit *tChain = NULL;
	ITransaction *tResult = NULL;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || pMethod==NULL ) return NULL;

	tChain = pObj->mTransactions.units;
	for( i=0; i<pObj->mTransactions.size; i++ )
	{
		if( tChain==NULL ) return NULL;
		if( (tResult=(ITransaction*)(tChain->data))==NULL ) return NULL;
		if( !strcmp(tResult->mMethod,pMethod) && tResult->mState<e_TransactionState_Completed )
			return tResult;

		tChain = (ChainUnit*)(tChain->next);
	}

	return NULL;
}


HS_RESULT IDialogue_DeleteTransaction(void *pObject, HS_TR_HANDLE pHandle)
{
	IDialogue *pObj = (IDialogue*)pObject;
	ChainUnit *tChain = IDialogue_FindTransactionChainByHandle(pObject,pHandle);

	if( tChain==NULL ) return HS_ERR_SIP_NO_EXIST;
	return NoLockList_DeleteChain(&(pObj->mTransactions),tChain);

	return HS_OK;
}


void IDialogue_StateToKillMe(void *pObject, IStack *pStack, IUa *pUa, RemoveReason pReason)
{
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL ) return;
	if( pStack==NULL || pUa==NULL )
	{
		pObj->mState = e_DialogueState_KillMe;
		return;
	}
	if( pStack->CallbackCallToRemove == NULL )
	{
		pObj->mState = e_DialogueState_KillMe;
		return;
	}
	if( pStack->CallbackCallToRemove(pUa->mHandle,pObj->mHandle,pReason) == TRUE )
		pObj->mState = e_DialogueState_KillMe;
	else
		pObj->mState = e_DialogueState_Idle;
}


void IDialogue_CallbackOpenMedia(void *pObject, IStack *pStack)
{
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || pStack==NULL ) return;
	if( pStack->CallbackOpenMedia==NULL ) return;

	if( pObj->mAudio != NULL )
	{
		if( pObj->mAudio->mStarted == FALSE )
		{
			pStack->CallbackOpenMedia(pObj->mHandle,pObj->mAudio);
			pObj->mAudio->mStarted = TRUE;
		}
	}
	if( pObj->mVideo != NULL )
	{
		if( pObj->mVideo->mStarted == FALSE )
		{
			pStack->CallbackOpenMedia(pObj->mHandle,pObj->mVideo);
			pObj->mVideo->mStarted = TRUE;
		}
	}

	return;
}


void IDialogue_CallbackCloseMedia(void *pObject, IStack *pStack)
{
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || pStack==NULL ) return;
	if( pStack->CallbackCloseMedia==NULL ) return;

	if( pObj->mAudio != NULL )
	{
		if( pObj->mAudio->mStarted == TRUE )
		{
			pStack->CallbackCloseMedia(pObj->mHandle,pObj->mAudio);
			pObj->mAudio->mStarted = FALSE;
		}
	}
	if( pObj->mVideo != NULL )
	{
		if( pObj->mVideo->mStarted == TRUE )
		{
			pStack->CallbackCloseMedia(pObj->mHandle,pObj->mVideo);
			pObj->mVideo->mStarted = FALSE;
		}
	}

	return;
}


void IDialogue_CallbackInstantMessage(void *pObject,IStack *st,IUa *ua,SipMessage *pMessage)
{
	char *tType = NULL;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || st==NULL || ua==NULL || pMessage==NULL ) return;
	if( st->CallbackInstantMessage==NULL ) return;

	if( pMessage->mContentType != NULL )
		tType = pMessage->mContentType->mInfo;

	st->CallbackInstantMessage(ua->mHandle,pObj->mHandle,tType,pMessage->mBody,pMessage->mBodyLen);
	return;
}


void IDialogue_CallbackOutbandDtmf(void *pObject,IStack *st,IUa *ua,SipMessage *pMessage)
{
	char *tType = NULL;
	IDialogue *pObj = (IDialogue*)pObject;

	if( pObj==NULL || st==NULL || ua==NULL || pMessage==NULL ) return;
	if( st->CallbackOutbandDtmf==NULL ) return;

	if( pMessage->mContentType != NULL )
		tType = pMessage->mContentType->mInfo;

	st->CallbackOutbandDtmf(ua->mHandle,pObj->mHandle,tType,pMessage->mBody,pMessage->mBodyLen);
	return;
}
