'''Database Independent BBS library module.
'''

from __future__ import generators

import os
import shutil
import time
import re
import weakref
from StringIO import StringIO
from urllib import urlretrieve
from urllib import quote
import cgi

import dblib
import config
import users
import exceptionLib
from templateProcessor import templateProcessor, getTemplate
import pyConfigParser, types, glob
import htmlValidate
import linkProcessor
import bbcode
import cache
import userGroup

# Thread type
UNDEFINED, DISCUSSION, QNA, CLOSED, NOTICE = range(5)
gdb = None
groups = None
globalNS = None
articleCache = cache.Cache(config.maxArticleCache)

def connectDB():
    global gdb
    if not gdb:
        gdb = dblib.connectDB()

def disconnectDB():
    global gdb
    if gdb:
        dblib.closeDB(gdb)
        gdb = None

def getGlobalNS():
    global globalNS
    if globalNS:
        getStatistic()
        return globalNS
    class NameSpace(dict):
        __slots__ = 'nTotalPost groupName nGroupPost forumId \
            forumName nForumPost forumId threadId articleId userId \
            action subject body \
            REMOTE_ADDR HTTP_USER_AGENT HTTP_ACCEPT_CHARSET \
            HTTP_ACCEPT_LANGUAGE HTTP_HOST'.split()
        def __init__(self, **kw):
            dict.__init__(self, kw)
        def __getattr__(self, name):
            return self[name]
    globalNS = NameSpace(
        user=None,
        session=None
        )
    globalNS.update(globals()['__builtins__'])
    d = config.__dict__
    for key, value in d.items():
        globalNS[key] = value
    globalNS['templateProcessor'] = templateProcessor
    globalNS['getTemplate'] = getTemplate
    globalNS['getCookedDateFormat'] = getCookedDateFormat
    globalNS['groups'] = getGroups()
    globalNS['strCutter'] = strCutter
    globalNS['config'] = config
    def escape(s, flag=True):
        return cgi.escape(s, flag)
    globalNS['escape'] = escape
    getStatistic()
    return globalNS

def getStatistic():
    globalNS['numberOfUsers'] = users.numberOfUsers()
    globalNS['numberOfActiveUsers'] = users.numberOfUsers(config.memberLevel)
    groups = getGroups()
    sum = 0
    sumOfnTopics = 0
    sumOfnPosts = 0
    latestPost = '0000-00-00 00:00:00'
    latestPoster = 'nobody'
    for category in groups:
        sum += len(category)
        for forum in category:
            sumOfnTopics += forum.nTopics
            if latestPost < forum.lastDate:
                latestPost = forum.lastDate
                latestPoster = forum.lastPoster
    globalNS['numberOfCategories'] = len(groups)
    globalNS['numberOfForums'] = sum
    globalNS['numberOfThreads'] = sumOfnTopics
    globalNS['latestPost'] = latestPost
    globalNS['latestPoster'] = latestPoster

def getGroups():
    global groups
    if groups:
        return groups
    else:
        groups = Groups()
        groups.refresh()
        return groups

def escapeTextForDb(text):
    text = text.replace('\\', '\\\\')
    text = text.replace("'", "\\'")
    return text

def escapeWhiteSpace(text):
    sp = re.compile(' ( +)')
    def spacerepl(match):
        s = match.group()
        return ' '+'&nbsp;'*(len(s)-1)
    text = text.expandtabs()
    text = sp.sub(spacerepl, text)
    text = text.replace('\n', ' <br/>\n')
    return text

def escapeTagMarker(s):
    s = s.replace('<:', '&lt;:')
    return s.replace(':>', ':&gt;')

def escapeTextForm(body):
    body = body.replace('<br/>','\n')
    body = body.replace('&', '&amp;')
    body = escapeTagMarker(body)
    return body

class Page(object):
    def __init__(self, pagesize, startPage=1):
        self.nPages = 0
        self.pagesize = int(pagesize)
        self.startPage = startPage

    def setnPages(self, N=-1):
        if N == -1:
            N = len(self)
        self.nPages = (N + self.pagesize - 1) / self.pagesize
#        self.pageIndex = [no*self.pagesize for no in range(self.nPages)]
        self.pageIndex = [(no-(self.startPage-1))*self.pagesize for no in range(self.nPages)]

    def page(self, no=1):
        pagesize = self.pagesize
        try:
            k = self.pageIndex[no-1]
            for i in range(k, min(k+pagesize, len(self))):
                yield self[i]
        except:
            raise StopIteration

class Tree:
    def __nonzero__(self):
        return True

    def setParent(self, parent):
        if parent != None:
            self.parent = weakref.ref(parent)
        else:
            self.parent = None
    def getParent(self):
        if self.parent == None:
            return None
        return self.parent()
    def makeOneNameSpace(self):
        L = []
        node = self
        while node != None:
            L.append(node.__dict__)
            node = node.getParent()
        d = {}
        while L:
            d.update(L.pop())
        return d

    def toHTML(self, html=None, locals={}):
        if html == None:
            html = self.standard_html
        globalNS = getGlobalNS()
        d = self.makeOneNameSpace()
        d['self'] = self
        d['templateProcessor'] = templateProcessor
        d.update(locals)
        template = getTemplate(os.path.join(config.skinPath, html))
        return templateProcessor(template, globalNS, d)

class Groups(list, Tree):
    '''List of Category'''
    standard_html = 'groups.teul'
    def __init__(self):
        self.parent = None
        self.nTotalPosts = 0
        for category in config.getCategories():
            self.append(Category(self, category))

    def updateDbTables(self):
        config.dbTables = []
        for child in self:
            config.dbTables.append(child.toList())
        self.saveGroupDescFile()

    def saveGroupDescFile(self):
        config.saveGroupDescFile()

    def addCategory(self, categoryName):
        if self.getCategoryIndexByName(categoryName) > -1:
            return False
        for char in ['\\', '<', '>', '?', ':', '"', '|', '/', '*']:
            if char in categoryName:
                return False

        self.append(Category(self, config.CategoryConfig([categoryName])))
        self.updateDbTables()
        return self[-1]

    def deleteCategory(self, categoryName):
        index = self.getCategoryIndexByName(categoryName)
        if index < 0:
            return False
        if len(self[index]):
            return False
        del self[index]
        self.updateDbTables()
        os.remove(config.encode2Hex(categoryName)+'.ini')
        return True

    def moveUpCategory(self, categoryName):
        gIndex = self.getCategoryIndexByName(categoryName)
        if gIndex <= 0:
            return False
        moveCategory = self.pop(gIndex)
        self.insert(gIndex - 1, moveCategory)
        self.updateDbTables()
        return True

    def moveDownCategory(self, categoryName):
        gIndex = self.getCategoryIndexByName(categoryName)
        if gIndex < 0 or gIndex == len(self)-1:
            return False
        moveCategory = self.pop(gIndex)
        self.insert(gIndex + 1, moveCategory)
        self.updateDbTables()
        return True

    def changeCategoryName(self, oldCategoryName, newCategoryName):
        for char in ['\\', '<', '>', '?', ':', '"', '|', '/', '*']:
            if char in newCategoryName:
                return False

        category = self.getCategoryByName(oldCategoryName)
        if not category:
            return False
        else:
            category.changeCategoryName(newCategoryName)
            self.updateDbTables()
            return True


    def getCategoryIndexByName(self, categoryName):
        c = 0
        for child in self:
            if child.categoryName.lower() == categoryName.lower():
                return c
            c += 1
        return -1

    def getCategoryByName(self, categoryName):
        for child in self:
            if child.categoryName == categoryName:
                return child
        return False

    def getForum(self, forumId):
        for category in self:
            forum = category.getForumById(forumId)
            if isinstance(forum, Forum):
                return forum
        return False

    getForumById = getForum

    def getForumByName(self, forumName):
        for category in self:
            forum = category.getForumByName(forumName)
            if isinstance(forum, Forum):
                return forum
        return False

    def getThread(self, forumId, threadId):
        return self.getForum(forumId).getThreadById(threadId)

    def getArticle(self, forumId, threadId, articleId):
        return self.getForum(forumId).getThreadById(threadId).readArticle(articleId)

    def refresh(self):
        for child in self:
            child.refresh()
        self.updateStatistic()

    def updateStatistic(self):
        self.nTotalPosts = reduce(lambda a,b:a+b, [child.nCategoryPost for child in self], 0)


class Category(list, Tree):
    '''List of Forums'''
    standard_html = 'category.teul'
    def __init__(self, parent, category):
        self.setParent(parent)
        self.categoryName = category.getName()
        self.quotedCategoryName = quote(quote(self.categoryName))
        for forum in category.getForumConfigs():
            self.append(Forum(self, *forum))
        self.nCategoryPost = 0

    def changeCategoryName(self, newCategoryName):
        groups = self.getParent()
        catNames = [cat.categoryName.strip() for cat in groups]
        if newCategoryName.strip() in catNames:
            return None
        self.categoryName = newCategoryName
        self.quotedCategoryName = quote(quote(newCategoryName))
        self.getParent().updateDbTables()

    def addForum(self, *args, **kw):  # new forum
        if 'magicNumber' in kw:
            forumId = config.tablePrefix + '_forum_' + str(int(kw['magicNumber']))
        else:
            forumId = config.tablePrefix + '_forum_' + str(int(time.time()))
        ret = self.addForumById(forumId, *args)
        return ret

    def addForumById(self, forumId, *args):
        registeredTableNames = dblib.getTableNames(dblib.getCursor())
        if forumId in registeredTableNames:
            return False

        for forum in self:
            if forumId == forum.forumId:
                return False

        self.append(Forum(self, forumId, *args))
        self.getParent().updateDbTables()
        dblib.makeTables()
        return True

    def deleteForumById(self, forumId):
        k = 0
        for forum in self:
            if forum.forumId == forumId:
                break
            k += 1
        else:
            return False
        dblib.removeTable(forum.forumId)
        dblib.removeTable(forum.forumId+'Thread')
        self.pop(k)
        self.getParent().updateDbTables()
        return True

    def editForum(self, forumId, forumName, forumDesc, lock, moderator,
                  view, read, post, reply, edit, delete, sortBy, order, userGroupList):
        index = 0
        for forum in self:
            if forumId == forum.forumId:
                break
            index += 1
        else:
            return False
        self[index] = Forum(self, forumId, forumName, forumDesc, lock, moderator,
                            view, read, post, reply, edit, delete, sortBy, order, userGroupList)
        self[index].refresh()
        self.getParent().updateDbTables()
        return True

    def moveUpForum(self, forumId):
        fIndex = 0
        groups = self.getParent()
        cIndex = groups.getCategoryIndexByName(self.categoryName)
        for forum in self:
            if forumId == forum.forumId:
                break
            fIndex += 1
        else:
            return False

        if fIndex == 0 and cIndex == 0:
            return False
        elif fIndex == 0 and cIndex > 0:
            moveForum = self.pop(fIndex)
            category = groups[cIndex - 1]
            moveForum.setParent(category)
            category.append(moveForum)
        else:
            moveForum = self.pop(fIndex)
            self.insert(fIndex - 1, moveForum)
        groups.updateDbTables()
        return True

    def moveDownForum(self, forumId):
        nForum = len(self) - 1
        fIndex = 0
        groups = self.getParent()
        nCategory = len(groups) - 1
        cIndex = groups.getCategoryIndexByName(self.categoryName)
        for forum in self:
            if forumId == forum.forumId:
                break
            fIndex += 1
        else:
            return False

        if fIndex == nForum and cIndex == nCategory:
            return False
        elif fIndex == nForum and cIndex < nCategory:
            moveForum = self.pop(fIndex)
            category = groups[cIndex + 1]
            moveForum.setParent(category)
            category.insert(0, moveForum)
        else:
            moveForum = self.pop(fIndex)
            self.insert(fIndex + 1, moveForum)
        groups.updateDbTables()
        return True

    def refresh(self):
        for child in self:
            child.refresh()
        self.updateStatistic()

    def updateStatistic(self):
        self.nCategoryPost = reduce(lambda a,b:a+b, [child.nForumPost for child in self], 0)
        parent = self.getParent()
        if parent:
            parent.updateStatistic()

    def getForumIdListByName(self, forumName):
        L = []
        for forum in self:
            if forumName == forum.forumName:
                L.append(forum.forumId)

        if not L:
            return False

        return L

    def getForumById(self, forumId):
        for forum in self:
            if forum.forumId == forumId:
                return forum
        return False

    def getForumByName(self, forumName):
        for forum in self:
            if forum.forumName == forumName:
                return forum
        return False


    def toList(self):
        L = [self.categoryName]
        for child in self:
            L.append(child.toTuple())
        return L

# Forum
################################################################################

class Forum(list, Tree, Page):
    '''List of Threads'''
    standard_html = 'forum.teul'
    def __init__(self, parent, forumId, forumName='', forumDesc='',
                 lock='unlocked', moderator='',
                 view='all', read='all', post='member', reply='member',
                 edit='private', delete='private', sortBy='mdate', order='desc',
                 userGroupList=None):
        self.setParent(parent)
        Page.__init__(self, config.numberOfTitlesInAPage)
        self.forumId = forumId
        self.forumName = forumName
        self.forumDesc = forumDesc
        self.lock = lock
        self.moderator = moderator.replace(',', ' ')

        self.view = view
        self.read = read
        self.post = post
        self.reply = reply
        self.edit = edit
        self.delete = delete
        self.sortBy = sortBy
        self.order = order
        self.curSortBy = sortBy
        self.curOrder = order
        if type(userGroupList) == type(''):
            self.userGroupList = userGroupList.split()
        else:
            self.userGroupList = userGroupList or []

        self.nTopics = 0
        self.nForumPost = 0
        cursor = dblib.getCursor()
        self.noticeThreadList = []
        self.lastDate = ''
        self.lastPoster = ''

        self.threadIdIndexDict = {}
	self.cache = cache.Cache(config.maxThreadCachePerForum)

    def isUserGroupMember(self, user):
        registeredUserGroupConfig = userGroup.getUserGroup()
        for groupName in self.userGroupList:
            if registeredUserGroupConfig.isMember(groupName, user):
                return True
        return False

    def getNoticeList(self):
        return self.noticeThreadList

    def updateNoticeList(self):
        cursor = dblib.getCursor()
        cursor.execute('SELECT threadId FROM %sThread WHERE type = %s ORDER BY mdate DESC' % (self.forumId, NOTICE))
        threadIdList = dblib.extractNames(cursor.fetchall())
        self.noticeThreadList = [Thread(self, self.forumId, threadId) for threadId in threadIdList]
        return self.noticeThreadList

    def refresh(self, sortBy='', order='', pagesize=None):
        if pagesize == None:
            pagesize = config.numberOfTitlesInAPage
        cursor = dblib.getCursor()
        cursor.execute('SELECT COUNT(*) FROM %sThread WHERE type = %s' % (self.forumId, NOTICE))
	self.nNotices = cursor.fetchone()[0]
	if self.nNotices != len(self.noticeThreadList):
            self.updateNoticeList()
        self.pagesize = int(pagesize)
        if sortBy:
            self.curSortBy = sortBy
        if order:
            self.curOrder = order
        self.threadIdIndexDict.clear()
        self.updateStatistic()
        self.setnPages()
	self.cache.clear()

    def updateStatistic(self):
        cursor = dblib.getCursor()
        cursor.execute('SELECT COUNT(*) FROM %sThread' % self.forumId)
        self.nTopics = cursor.fetchone()[0]
        cursor.execute('SELECT COUNT(*) FROM %s' % self.forumId)
        self.nForumPost = cursor.fetchone()[0]

        self.lastDate = ''
        self.lastPoster = ''
        cursor.execute('SELECT max(mdate) FROM %sThread' % (self.forumId))
        mdate = cursor.fetchone()[0]
        if mdate:
            self.lastDate = mdate
            cursor.execute('SELECT lastPoster FROM %sThread WHERE mdate="%s"' % (self.forumId, self.lastDate))
            self.lastPoster = cursor.fetchone()[0]
        parent = self.getParent()
        if parent:
            parent.updateStatistic()

    def __len__(self):
        return self.nTopics - self.nNotices

    def __getitem__(self, index):
        thread = self.cache.getItem(index)
        if thread:
            return thread
        cursor = dblib.getCursor()
        try:
            sql = 'SELECT threadId FROM %sThread WHERE type != %s ORDER BY %s %s LIMIT %s, %s' % (
                self.forumId, NOTICE, self.curSortBy, self.curOrder, index, config.numberOfTitlesInAPage)
            cursor.execute(sql)
            threadIdList = dblib.extractNames(cursor.fetchall())
        except:
            raise IndexError, index
        k = 0
        for threadId in threadIdList:
            self.threadIdIndexDict[threadId] = index + k
            thread = Thread(self, self.forumId, threadId)
            self.cache.putItem(index+k, thread)
            k += 1
        return self.cache.getItem(index)

    def __iter__(self):
        for k in range(len(self)):
            yield self[k]

    def getThreadById(self, threadId):
        for thread, time in self.cache.values():
            if thread.threadId == threadId:
                return thread
        for thread in self.noticeThreadList:
            if thread.threadId == threadId:
                return thread
        return Thread(self, self.forumId, threadId) # FIXME

    def getNextThread(self, threadId):
        try:
            pos = self.threadIdIndexDict.get(threadId, -1)
            if pos >= 0:
                thread = self[pos+1]
                return thread.threadId, thread
            pos = 0
            for thread in self:
                if thread.threadId == threadId:
                    if thread.type == NOTICE:
                        return False, False
                    while self[pos+1].type == NOTICE:
                        pos += 1
                    break
                pos += 1
            thread = self[pos+1]
            return thread.threadId, thread
        except (IndexError, AttributeError), errMsg:
            return False, False

    def getPreviousThread(self, threadId):
        try:
            pos = self.threadIdIndexDict.get(threadId, -1)
            if pos >= 0:
                thread = self[pos-1]
                return thread.threadId, thread

            pos = 0
            for thread in self:
                if thread.threadId == threadId:
                    if thread.type == NOTICE:
                        return False, False
                    while self[pos-1].type == NOTICE:
                        pos -= 1
                    break
                pos += 1
            thread = self[pos-1]
            return thread.threadId, thread
        except (IndexError, AttributeError), errMsg:
            return 0, False

    def toTuple(self):
        return (self.forumId, self.forumName, self.forumDesc,
                self.lock, self.moderator,
                self.view, self.read, self.post, self.reply,
                self.edit, self.delete, self.sortBy, self.order,
		' '.join(self.userGroupList),
                )


class Message(object):
    def setCurrentDateAndTime(self):
        self.date = time.strftime('%Y-%m-%d %H:%M:%S')

    def setModifiedDateAndTime(self):
        self.mdate = time.strftime('%Y-%m-%d %H:%M:%S')

# Thread
################################################################################
class Thread(list, Message, Tree, Page):
    '''List of Article objects'''
    standard_html = 'thread.teul'
    def __init__(self, parent, forumId, threadId):
        self.setParent(parent)
        Page.__init__(self, config.numberOfArticlesInAPage)
        self.cursor = dblib.getCursor()
        self.cursor.execute('SELECT * FROM %sThread WHERE threadId=%s' % (forumId, threadId))
        if self.cursor.rowcount < 1:
            raise exceptionLib.RecordNotFound, 'Thread RecordNotFound in DB forumId=%s threadId=%s' % (forumId, threadId)
        threadId, type, mdate, readCount, starter, lastPoster, replies, subject = self.cursor.fetchone()
        self.forumId = forumId
        self.threadId = threadId
        self.type = type
        self.mdate = str(mdate)
        self.starter = starter
        self.lastPoster = lastPoster
        self.readCount = readCount
        self.replies = replies
        self.subject = subject

    def refresh(self):
        self.pagesize = int(config.numberOfArticlesInAPage)
        self.cursor.execute('SELECT id FROM %s WHERE threadId=%s' % (self.forumId, self.threadId))
        articleIdList = list(self.cursor.fetchall())
        articleIdList.sort()
        self[:] = []
        for T in articleIdList:
            articleId = T[0]
            self.append(self.readArticle(articleId))
        self.setnPages()

    def incrReadCount(self):
        self.cursor.execute("UPDATE %sThread SET readCount=readCount+1 WHERE threadId=%s" % (self.forumId, self.threadId))
        self.readCount += 1

    def decrReadCount(self):
        self.cursor.execute("UPDATE %sThread SET readCount=readCount-1 WHERE threadId=%s" % (self.forumId, self.threadId))
        self.readCount -= 1

    def getArticleById(self, articleId):
        for child in self:
            if child.id == articleId:
                return child
        return False

    def getUserListToBeNotified(self):
        d = {}
        for article in self:
            if article.notify:
                d[article.uid] = None
        return d.keys()        
    
    def readArticle(self, id):
        self.cursor.execute('SELECT * FROM %s WHERE id=%s' % (self.forumId, id))
        args = self.cursor.fetchone()
        fieldNames = dblib.getFieldNames(self.forumId)
        return Article(self, **dict(zip(fieldNames, args)))

    def changeType(self, type=0):
        self.cursor.execute('UPDATE %sThread SET type=%d WHERE threadId=%s' % (self.forumId, int(type), self.threadId))
        self.type = type

    def toSQL(self):
        self.escapedSubjectForDb = escapeTextForDb(self.subject)
        return "(%(threadId)s, %(type)s, '%(mdate)s', "\
            "%(readCount)s, '%(starter)s', '%(lastPoster)s', %(replies)s, "\
            "'%(escapedSubjectForDb)s')" % self.__dict__
    def __repr__(self):
        L = [repr(article) for article in self]
        return '\n'.join(L)



# Article
################################################################################
class Article(Message, Tree):
    def __init__(self, parent, id='NULL', threadId=-1, offset=0, ip='',
                 date='', mdate='',
                 readCount=0, uid='', upload1='', downCount1=0,
                 htmltag=0, emoticon=0, bbcode=1, notify=0, subject='', body=''):
        self.setParent(parent)
        self.id = self.articleId = id
        self.threadId = threadId
        self.offset = offset
        self.ip = ip and ip or os.environ.get('REMOTE_ADDR', 'localhost')
        if date:
            self.date = date
        else:
            self.date = time.strftime('%Y-%m-%d %H:%M:%S')
        if mdate:
            self.mdate = mdate
        else:
            self.mdate = self.date
        self.readCount = readCount
        self.uid = uid
        self.upload1 = upload1
        self.downCount1 = int(downCount1)
        self.htmltag = int(htmltag)
        self.emoticon = int(emoticon)
        self.bbcode = bbcode
        self.notify = notify
        self.subject = subject
        self.body = body.rstrip()

    def escapeSubjectAndBody(self):
        self.escapedSubjectForDb = escapeTextForDb(self.subject.rstrip())
        self.escapedBodyForDb = escapeTextForDb(self.body.rstrip())

    def escapeBodyHtml(self):
        if self.htmltag:
            self.escapedHtmlBody = bbcode.bbCodeProcessor(self.body)
        elif self.bbcode:
            self.escapedHtmlBody = cgi.escape(self.body)
            self.escapedHtmlBody = escapeWhiteSpace(self.escapedHtmlBody)
            self.escapedHtmlBody = bbcode.bbCodeProcessor(self.escapedHtmlBody)
        else:
            self.escapedHtmlBody = cgi.escape(self.body)
            self.escapedHtmlBody = escapeWhiteSpace(self.escapedHtmlBody)
            self.escapedHtmlBody = linkProcessor.linkProcessor(self.escapedHtmlBody)
        user = users.getUser(self.uid)
        if self.emoticon:
            import emoticons
            tag = "<img src='./%s' border='0' align='absmiddle'>"
            emo = emoticons.Emoticons()
            for key, value in emo.emoticons.items():
                self.escapedHtmlBody = self.escapedHtmlBody.replace(key, (tag % value[1]).replace('\\', '/'))
        self.escapedHtmlBody = escapeTagMarker(self.escapedHtmlBody)

    def toHTML(self, color):
        self.escapeBodyHtml()
        globalNS = getGlobalNS()
        globalNS['articleId'] = self.id
        d = self.makeOneNameSpace()
        d['articleBgColor'] = color
        d['self'] = self
        d['user'] = users.getUser(self.uid)
        d['uploadFileName'] = os.path.split(self.upload1)[1]
        d['uploadFileIsImage'] = os.path.splitext(self.upload1)[1].lower() in ('.jpg', '.jpeg', '.gif', '.png', '.bmp')
        template = getTemplate(os.path.join(config.skinPath, 'article.teul'))
        return templateProcessor(template, globalNS, d)

    def incrDownCount(self, forumId):
        self.downCount1 += 1
        cursor = dblib.getCursor()
        cursor.execute("UPDATE %s SET downCount1=%s WHERE id=%s" %
                 (forumId, self.downCount1, self.id))

    def toSQL(self, forumId):
        self.escapeSubjectAndBody()
        fieldInfo = dblib.getFieldInfo(forumId)
        fieldInfo = [f[:2] for f in fieldInfo]
        L = []
        for fieldName, fieldType in fieldInfo:
            if fieldName == 'subject': fieldName = 'escapedSubjectForDb'
            if fieldName == 'body': fieldName = 'escapedBodyForDb'
            fieldType = fieldType.lower()
            if fieldType.find('char') >= 0 or fieldType.find('datetime') >= 0 or fieldType.find('text') >= 0 :
                L.append("'%%(%s)s'" % fieldName)
            else:
                L.append('%%(%s)s' % fieldName)
        return '(%s)' % (', '.join(L) % self.__dict__)
    
        fieldNames[fieldNames.lower().find('subject')] = 'escapedSubjectForDb'
        fieldNames[fieldNames.lower().find('body')] = 'escapedBodyForDb'
        
        return "(%(id)s, %(threadId)s, %(offset)s, '%(ip)s', "\
            "'%(date)s', '%(mdate)s', "\
            "%(readCount)s, '%(uid)s', '%(upload1)s', '%(downCount1)s', "\
            "%(htmltag)s, %(emoticon)s, %(bbcode)s, %(notify)s, '%(escapedSubjectForDb)s', '%(escapedBodyForDb)s')" % self.__dict__
    def __repr__(self):
        return """id : %(id)s
threadId : %(threadId)s
offset : %(offset)s
ip : '%(ip)s'
date : '%(date)s'
mdate : '%(mdate)s'
readCount : %(readCount)s
uid : '%(uid)s'
upload1 : '%(upload1)s'
htmltag : '%(htmltag)s'
emoticon : '%(emoticon)s'
bbcode : '%(bbcode)s'
notify : '%(notify)s'
subject : '%(subject)s'
body : '%(body)s'
""" % self.__dict__


def readArticle(forumId, id):
    sql = 'SELECT * FROM %s WHERE id=%s' % (forumId, id)
    article = articleCache.getItem(sql)
    if article: return article
    cursor = dblib.getCursor()
    cursor.execute(sql)
    args = cursor.fetchone()
    fieldNames = dblib.getFieldNames(forumId)
    try:
        article = Article(None, **dict(zip(fieldNames, args)))
        articleCache.putItem(sql, article)
        return article
    except TypeError:
        return None

class PageList(list, Page):
    def __init__(self, pagesize, startPage=1):
        self.pagesize = int(pagesize)
        self.startPage = startPage

def searchUserForums(uid):           # FIXME
    cursor = dblib.getCursor()
    groups = getGroups()
    groups.refresh()    # FIXME
    tableNames = []
    for category in groups:
        tableNames.extend([(forum.forumId, forum.forumName) for forum in category])
    forumList = []
    for forumId, forumName in tableNames:
        cursor.execute('''
        SELECT * FROM %s WHERE uid='%s'
        ''' % (forumId, uid))
        if cursor.rowcount > 0:
            forumList.append((forumId, forumName))
        forumList.sort(lambda lhs, rhs: cmp(lhs[1].capitalize(), rhs[1].capitalize()))
    return forumList


def fileUpload(forumId, article, fileStr):
    path, fname = os.path.split(article.upload1)
    dir = '%s-%s' % (forumId, article.threadId)
    dir = os.path.join(config.uploadPath, dir)
    if not os.path.exists(dir):
        os.makedirs(dir)
    filePath = os.path.join(dir, fname).replace(' ', '_')
    open(filePath, 'wb').write(fileStr)
    article.upload1 = filePath.replace('\\', '/')

def writeThread(forumId, article, fileName='', fileStr='', type=DISCUSSION, readCount=0):
    if article.htmltag:
        htmlValidate.validate(article.body)
    if article.bbcode:
        bbcode.bbCodeProcessor(article.body)
    cursor = dblib.getCursor()
    sqlarg = "(NULL, %s, '%s', %s, '%s', '%s', 0, '%s')" % (
        type, article.mdate, readCount, article.uid, article.uid, escapeTextForDb(article.subject))
    cursor.execute("INSERT INTO %sThread VALUES %s" % (forumId, sqlarg))
    newThreadId = cursor.insert_id()
    article.threadId = newThreadId
    if fileName:
        fileName = fileName.replace('\\', '/')
        article.upload1 = os.path.split(fileName)[1]
        fileUpload(forumId, article, fileStr)
    names = dblib.getFieldNames(forumId)
    cursor.execute("INSERT INTO %s (%s) VALUES %s" % (forumId, ', '.join(names), article.toSQL(forumId)))
    user = users.getUser(article.uid)
    user.incrPostCount()
    refreshForum(forumId)
    return article.threadId

def getMaxArticleOffset(cursor, forumId, threadId):
    cursor.execute('SELECT MAX(offset) from %s WHERE threadId=%s' % (forumId, threadId))
    max = cursor.fetchone()[0]
    if not max:
        return 0
    return max

def replyArticle(forumId, article, threadId, fileName='', fileStr=''):
    if article.htmltag:
        htmlValidate.validate(article.body)
    if article.bbcode:
        bbcode.bbCodeProcessor(article.body)
    cursor = dblib.getCursor()
    article.threadId = threadId
    article.offset = getMaxArticleOffset(cursor, forumId, threadId) + 1   # Critical Section?
    if article.date[:4] == '0000':
        article.setCurrentDateAndTime()
    #article.setModifiedDateAndTime()
    if fileName:
        fileName = fileName.replace('\\', '/')
        article.upload1 = os.path.split(fileName)[1]
        fileUpload(forumId, article, fileStr)
    names = dblib.getFieldNames(forumId)
    cursor.execute("INSERT INTO %s (%s) VALUES %s" % (forumId, ', '.join(names), article.toSQL(forumId)))
    newArticleId = cursor.insert_id()
    cursor.execute("UPDATE %sThread SET replies=replies+1, lastPoster='%s', mdate='%s' WHERE threadId=%s" %
                 (forumId, article.uid,
                  article.mdate, threadId))
    thread = getGroups().getThread(forumId, threadId)
    thread.replies += 1
    thread.mdate = article.mdate
    thread.lastPoster = article.uid
    user = users.getUser(article.uid)
    user.incrPostCount()
    refreshForum(forumId)
    return article.offset, newArticleId

def editArticle(forumId, articleId, article, uid, fileName='', fileStr='', remove=False):
    '''update subject and body text'''
    if article.htmltag:
        htmlValidate.validate(article.body)
    if article.bbcode:
        bbcode.bbCodeProcessor(article.body)
    cursor = dblib.getCursor()
    article.setModifiedDateAndTime()
    if remove:
        if article.upload1:
            os.remove(article.upload1)
        article.upload1 = ''
        article.downCount1 = 0
    if fileName:
        if article.upload1:
            os.remove(article.upload1)
        fileName = fileName.replace('\\', '/')
        article.upload1 = os.path.split(fileName)[1]
        fileUpload(forumId, article, fileStr)
        article.downCount1 = 0
    article.escapeSubjectAndBody()
    cursor.execute("UPDATE %s SET mdate='%s', upload1='%s', downCount1=%s, htmltag=%s, emoticon=%s, bbcode=%s, notify=%s,subject='%s', body='%s' WHERE id=%s" %
                 (forumId, article.mdate, article.upload1, article.downCount1,
                  article.htmltag, article.emoticon, article.bbcode, article.notify,
                  article.escapedSubjectForDb, article.escapedBodyForDb, articleId))
    cursor.execute("UPDATE %sThread SET mdate='%s', lastPoster='%s' WHERE threadId=%s" %
                 (forumId, article.mdate, uid, article.threadId))
    if article.offset == 0:
        cursor.execute("UPDATE %sThread SET subject='%s' WHERE threadId=%s" %
                 (forumId, article.escapedSubjectForDb, article.threadId))
    key = 'SELECT * FROM %s WHERE id=%s' % (forumId, articleId)
    thread = getGroups().getThread(forumId, article.threadId)
    thread.mdate = article.mdate
    thread.subject = article.subject
    thread.escapedHtmlSubject = escapeWhiteSpace(cgi.escape(thread.subject))
    thread.lastPoster = uid
    refreshForum(forumId)
    articleCache.deleteItem(key)

def deleteThread(forumId, threadId):
    cursor = dblib.getCursor()
    cursor.execute('SELECT id FROM %s WHERE threadId=%s' % (forumId, threadId))
    idList = []
    for articleId, in cursor.fetchall():
        deleteArticle(forumId, articleId, '')
        idList.append(articleId)
    cursor.execute('DELETE FROM %sThread WHERE threadId=%s' % (forumId, threadId))
    shutil.rmtree(os.path.join(config.uploadPath, '%s-%s' % (forumId, threadId)))
    refreshForum(forumId)
    return idList

def deleteArticle(forumId, articleId, uid):
    cursor = dblib.getCursor()
    cursor.execute('SELECT threadId, offset, upload1 FROM %s WHERE id=%s' % (forumId, articleId))
    threadId, offset, upload1 = cursor.fetchall()[0]
    article = readArticle(forumId, articleId)
    writer = users.getUser(article.uid)
    writer.decrPostCount()
    cursor.execute('DELETE FROM %s WHERE threadId=%s AND offset=%s' % (forumId, threadId, offset))
    cursor.execute("UPDATE %sThread SET replies=replies-1, mdate='%s', lastPoster='%s' WHERE threadId=%s" % (forumId, time.strftime('%Y-%m-%d %H:%M:%S'), uid, threadId))
    if upload1:
        os.remove(upload1)
    thread = getGroups().getThread(forumId, article.threadId)
    thread.replies -= 1
    thread.setModifiedDateAndTime()
    thread.lastPoster = uid
    refreshForum(forumId)
    return True

def clearTable(forumId):
    cursor = dblib.getCursor()
    cursor.execute('DELETE FROM %s' % self.forumId)

def refreshForum(forumId):
    groups = getGroups()
    forum = groups.getForumById(forumId)
    forum.refresh()


# following functions are used by skins
################################################################################
def getCookedDateFormat(date, format=None):
    if not date:
        return None
    if format == None:
        format = config.dateFormat
    import time
    try:
        ymd, hms= date.split()
        date = map(lambda field: int(field), ymd.split('-') + hms.split(':')) + [0] * 3
        return time.strftime(format, date)
    except TypeError:
        return ''
    except:
        return date


def strCutter(text, length=40, postfix="..."):
    if len(text) < length: return text
    pos = length - 1 
    char = text[pos] 
    while ord(char) > 127 and pos >= 0:
        pos -= 1
        char = text[pos]
    return text[:(length-pos) % 2 and length or length - 1] + postfix
