﻿/*
 *  X-IRC Server
 *   Copyright (C) 2004-2009 Paradoxoft Corporation.
 *   Copyright (C) 2009 Stephanos San Io.
 *  
 *  < GNU GPL v3 Terms >
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *   
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

using System;
using System.Net.Sockets;
using System.Collections.Generic;

namespace Server.Provider
{
    class Connexion
    {
        // Management
        public long Id;

        // Transmission
        private Socket _TransmissionSocket;
        private byte[] _ReceiveBuffer;
        private Queue<byte> _ReceiveQueue;

        // Basic Operation
        public int Mode;
        public int ReceivedPacketsCountInSecond;
        public int ReceivedBytesInSecond;
        public DateTime DisconnectionTime;
        public Boolean DisconnectionFlag;
        
        // Chat
        public string Username;
        public string Nickname;
        public string RealName;
        public List<Channel> JointChannelList;

        public Connexion(long Id, Socket TransmissionSocket)
        {
            // Initialise values
            this.Id = Id;
            
            _TransmissionSocket = TransmissionSocket;
            _ReceiveBuffer = new byte[Convert.ToInt32(Program.XDatabase.Get("Provider_ReceiveBufferSize"))];
            _ReceiveQueue = new Queue<byte>();

            Mode = 0;
            ReceivedPacketsCountInSecond = 0;
            ReceivedBytesInSecond = 0;
            DisconnectionTime = DateTime.Now;
            DisconnectionFlag = false;

            JointChannelList = new List<Channel>();
            // Start receiving
            if (_TransmissionSocket != null)
                _TransmissionSocket.BeginReceive(_ReceiveBuffer, 0, _ReceiveBuffer.Length, SocketFlags.None, new AsyncCallback(ReceiveCallbackProcedure), null);
        }

        public void Disconnect()
        {
            try
            {
                // Leave all joint channels
                for (int i = 0; i < JointChannelList.Count; i++)
                    JointChannelList[i].ConnexionLeave(Id);
                // Transmission shutdown
                _TransmissionSocket.Shutdown(SocketShutdown.Both);
                _TransmissionSocket.Close();
            }
            catch { }
            finally
            {
                Program.XConnexionManager._DeleteConnexionFromList(Id);
            }
        }

        public bool Send(byte[] Buffer)
        {
            try
            {
                _TransmissionSocket.BeginSend(Buffer, 0, Buffer.Length, SocketFlags.None, new AsyncCallback(SendCallbackProcedure), null);
                return true;
            }
            catch
            {
                return false;
            }
        }

        private void SendCallbackProcedure(IAsyncResult Result)
        {
            try
            {
                int SentBytes = _TransmissionSocket.EndSend(Result);
            }
            catch
            {
                Disconnect();
            }
        }

        private void ReceiveCallbackProcedure(IAsyncResult Result)
        {
            try
            {
                int ReceivedBytes = _TransmissionSocket.EndReceive(Result);
                ReceivedBytesInSecond += ReceivedBytes;
                if (ReceivedBytes == 0) // Disconnection
                {
                    Disconnect();
                    return;
                }
                for (int i = 0; i < ReceivedBytes; i++)
                {
                    if (_ReceiveBuffer[i] == '\n')
                    {
                        if (Packet.Processor.Process(this, _ReceiveQueue) == false)
                        {
                            Disconnect();
                            return;
                        }
                        ReceivedPacketsCountInSecond++;
                    }
                    else
                        _ReceiveQueue.Enqueue(_ReceiveBuffer[i]);
                }
                _TransmissionSocket.BeginReceive(_ReceiveBuffer, 0, _ReceiveBuffer.Length, SocketFlags.None, new AsyncCallback(ReceiveCallbackProcedure), null);
            }
            catch
            {
                Disconnect();
            }
        }
    }
}