package zetablitz.zeta;

import bsh.Interpreter;
import bsh.EvalError;
import cookxml.cookswt.CookSwt;
import cookxml.cookbsh.CookBSH;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tray;
import org.eclipse.swt.widgets.TrayItem;

/**
 * @author ni2
 * @version 0.1, 08/17/06
 *
 * This code licensed under the terms of the GPL.  See doc/LICENSE.
 */
public class Zeta{
	private static final String TITLE = "Zeta!";
	private static final String ICON_PATH = "resources/icon/icon.png";
	//import.bsh 파일에 미리 import문을 정의해 두면 후에 입력하는 수고를 덜 수 있다.
	private static final String BSH_IMPORT_PATH = "config/import.bsh";
	private static final String MODULES = "Modules";
	private static final String LOAD = "Load";
	private static final String CONSOLE = "Console";
	private static final String EXIT = "Exit";
	
	private static Interpreter bsh;
	private static CookSwt cookSWT;
	private static Display display;
	private static Shell shell;
	private static Image icon;
	private static SimpleLogger logger;
	private static TrayItem trayItem;
	private static Menu trayMenu;
	
	//등록된 셸들을 보이게 할 것인지를 결정한다.
	private static boolean shellsVisible;
	
	private Zeta(){
		initUI();
		initTrayMenu();
		initInterpreter();
	}
	
	//콘솔을 포함한 UI를 초기화한다.
	private void initUI(){
		display = new Display();
		icon = new Image(display, ICON_PATH);
		shell = new Shell(display);
		
		shell.setText(TITLE);
		shell.setImage(icon);
		
		GridLayout layout = new GridLayout(1, false);
		shell.setLayout(layout);
		
		Text prompt = new Text(shell, SWT.SINGLE | SWT.BORDER);
		prompt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		Text logView = new Text(shell, SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI | SWT.BORDER);
		logView.setLayoutData(new GridData(GridData.FILL_VERTICAL | GridData.FILL_HORIZONTAL));
		logger = new SimpleLogger(logView);
		
		PromptListener listener = new PromptListener(prompt);
		prompt.addKeyListener(listener);
		
		shell.setSize(300, 200);
		shell.addListener(SWT.Close, new Listener(){
			public void handleEvent(Event event) {
				shell.setVisible(false);
				event.doit = false;
			}
		});
	}
	
	//트레이에 아이콘과 메뉴를 만든다.
	private void initTrayMenu(){
		Tray tray = display.getSystemTray();
		trayItem = new TrayItem(tray, SWT.NULL);
		trayItem.setToolTipText(TITLE);
		trayItem.setImage(icon);
		
		MenuListener listener = new MenuListener();
		
		trayMenu = new Menu(shell, SWT.POP_UP);
		trayItem.addListener(SWT.MenuDetect, listener);
		trayItem.addListener(SWT.Selection, listener);
		
		MenuItem mi = new MenuItem(trayMenu, SWT.PUSH);
		mi.setText(CONSOLE);
		mi.addListener(SWT.Selection, listener);
		mi = new MenuItem(trayMenu, SWT.PUSH);
		mi.setText(LOAD);
		mi.addListener(SWT.Selection, listener);
		/*
		mi = new MenuItem(trayMenu, SWT.CASCADE);
		mi.setText(MODULES);
			Menu modules = new Menu(shell, SWT.DROP_DOWN);
			mi.setMenu(modules);
			mi = new MenuItem(modules, SWT.PUSH);
			mi.setText(LOAD);
			mi.addListener(SWT.Selection, listener);
			new MenuItem(modules, SWT.SEPARATOR);
		*/
		mi = new MenuItem(trayMenu, SWT.PUSH);
		mi.setText(EXIT);
		mi.addListener(SWT.Selection, listener);
	}
	
	//bsh, cookswt 인터프리터를 만들고 초기화한다.
	private void initInterpreter(){
		bsh = new Interpreter();
		try{
			//bsh를 통해 Shell을 생성할 수 있도록 display를 변수로 등록한다.
			bsh.set("display", display);
			bsh.set("shell", shell);
			//준비된 import문을 입력한다.
			interpretFromBSH(BSH_IMPORT_PATH);
		}catch(EvalError e){
			logger.printStackTrace(e);
		}
		
		cookSWT = new CookSwt();
		CookBSH.setupTagLibrary (cookSWT.getSwtTagLibrary());
	}
	
	//파일로부터 bsh 스크립트를 읽어 실행한다.
	private static void interpretFromBSH(String fileName){
		try{
			Object result = bsh.source(fileName);
			if(result != null) logger.println(result.toString());
			
		}catch(Exception e){
			logger.printStackTrace(e);
		}
	}
	
	//파일로부터 cookxml 스크립트를 읽어 실행한다.
	//Zeta의 cookswt 스크립트는 shell에서 시작하는 것을 기본으로 한다.
	//demo/HelloWorld.xml 참고.
	private static void interpretFromXML(String fileName){
		try{
			((Shell)cookSWT.xmlDecode(fileName)).open();
			
		}catch(Exception e){
			logger.printStackTrace(e);
		}
	}
	
	//bsh 인터프리터로 입력받은 명령어를 실행한다.
	private static void interpret(String s){
		try{
			Object result = bsh.eval(s);
			if(result != null) logger.println(result.toString());
			
		}catch(EvalError e){
			logger.printStackTrace(e);
		}
	}
	
	//프로그램을 종료한다.
	public static final void exit(){
		trayItem.setVisible(false);
		System.exit(0);
	}
	
	//bsh 혹은 coolxml 스크립트를 불러온다.
	public static void openScript(){
		FileDialog fd = new FileDialog(shell);
		fd.setFilterExtensions(new String[]{"*.bsh;*.xml"});
		String s = fd.open();
		if(s != null){
			if(s.endsWith("bsh"))
				Zeta.interpretFromBSH(s);
			else if(s.endsWith("xml"))
				Zeta.interpretFromXML(s);
		}
	}
	
	//콘솔을 보이게 하거나 감춘다.
	public static void consoleShowHide(){
		if(shell.isVisible())
			shell.setVisible(false);
		else
			shell.open();
	}
	
	//콘솔을 제외한 모든 셸을 보이거나 감춘다.
	public static void shellsShowHide(){
		Shell[] shells = display.getShells ();
		for (int i = 0; i < shells.length; ++i)
		{
			//콘솔은 영향을 받지 않음
			if(shells[i] != shell){
				if(shellsVisible){
					if (!shells[i].isDisposed ()) shells[i].open ();
				}else{
					shells[i].setVisible(false);
				}
			}
		}
		shellsVisible = shellsVisible ? false : true;
	}
	
	class MenuListener implements Listener{
		public void handleEvent(Event event) {
			if(event.widget instanceof MenuItem){
				String cmd = ((MenuItem) event.widget).getText();
				if(cmd == Zeta.CONSOLE){
					Zeta.consoleShowHide();
				}else if(cmd == Zeta.LOAD){
					Zeta.openScript();
				}else if(cmd == Zeta.EXIT){
					Zeta.exit();
				}
			}else if(event.widget instanceof TrayItem){
				if(event.type == SWT.MenuDetect)
					//팝업 메뉴를 연다.
					Zeta.trayMenu.setVisible(true);
				if(event.type == SWT.Selection)
					Zeta.shellsShowHide();
			}
		}
	}
	
	class PromptListener implements KeyListener{
		Text prompt;
		
		public PromptListener(Text prompt){
			this.prompt = prompt;
		}
		
		public void keyPressed(KeyEvent e){
			//Enter 가 입력되면 명령어를 실행.
			if(e.keyCode == SWT.CR){
				String s = prompt.getText();
				if(!s.equals("")){
					Zeta.interpret(s);
					prompt.setText("");
				}
			}
		}
		public void keyReleased(KeyEvent e){}
	}
	
	public static void main(String args[]){
		Zeta zb = new Zeta();
		
		while(!zb.shell.isDisposed()){
			if(!zb.display.readAndDispatch())
				zb.display.sleep();
		}
		zb.display.dispose();
	}
}

class SimpleLogger{
	public Text viewer;
	
	public SimpleLogger(Text viewer){
		viewer.setEditable(false);
		this.viewer = viewer;
	}
	
	//예외를 받아 viewer에 출력한다.
	public void printStackTrace(Exception e){
		StringBuilder sb = new StringBuilder();
		sb.append(e.toString());
		StackTraceElement stes[] = e.getStackTrace();
		for(StackTraceElement ste : stes){
			sb.append("\tat ");
			sb.append(ste.toString());
			sb.append("\n");
		}
		sb.append("\n\n");
		print(sb.toString());
	}
	
	//주어진 문자열을 viewer에 출력한다.
	public void print(String s){
		viewer.append(s);
	}
	
	//newline 문자를 출력한다.
	public void println(){
		viewer.append("\n");
	}
	
	//주어진 문자열을 viewer에 출력하고 newlinet 문자를 덧붙인다.
	public void println(String s){
		print(s);
		println();
	}
}
