package zetablitz.zeta;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tray;
import org.eclipse.swt.widgets.TrayItem;

/**
 * @author ni2
 * @version 0.2, 09/02/06
 * This code licensed under the terms of the GPL.  See doc/LICENSE.
 *
 * 메인 클래스.
 */
public class Zeta{
	private static final String TITLE = "Zeta!";
	private static final String ICON_PATH = "resources/icon/icon.png";
	private static final String MODULES = "Modules";
	private static final String LOAD = "Load";
	private static final String CONSOLE = "Console";
	private static final String EXIT = "Exit";
	
	//세션 정보가 담긴 파일
	public static final String SESSION_PATH = "config/session.properties";
	//세션 정보 파일 안에서, 배열 값의 구분자
	public static final String CONFIG_KEYS_SEPARATOR = "\t";
	//import.bsh 파일에 미리 import문을 정의해 두면 후에 입력하는 수고를 덜 수 있다.
	public static final String BSH_IMPORT_PATH = "config/import.bsh";
	
	private static final ZetaInterpreter interpreter = ZetaInterpreter.instance();
	private static final ZetaSessionManager session = ZetaSessionManager.instance();
	private static Display display;
	private static Shell shell;
	private static Image icon;
	private static TrayItem trayItem;
	private static Menu trayMenu;
	
	public static SimpleLogger logger;
	
	//등록된 셸들을 보이게 할 것인지를 결정한다.
	private static boolean shellsVisible = false;
	
	private Zeta(){
		initUI();
		initTrayMenu();
		loadScripts(session.modules.toArray(new String[0]));
	}
	
	//콘솔을 포함한 UI를 초기화한다.
	private void initUI(){
		display = new Display();
		shell = new Shell(display);
		shell.setText(TITLE);
		
		icon = new Image(display, ICON_PATH);
		shell.setImage(icon);
		
		shell.setLayout(new GridLayout(1, false));
		
		Text prompt = new Text(shell, SWT.SINGLE | SWT.BORDER);
		prompt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		Text logView = new Text(shell, SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI | SWT.BORDER);
		logView.setLayoutData(new GridData(GridData.FILL_VERTICAL | GridData.FILL_HORIZONTAL));
		logger = new SimpleLogger(logView);
		
		PromptListener listener = new PromptListener(prompt);
		prompt.addKeyListener(listener);
		
		shell.setSize(500, 300);
		shell.addListener(SWT.Close, new Listener(){
			public void handleEvent(Event event) {
				shell.setVisible(false);
				event.doit = false;
			}
		});
	}
	
	//트레이에 아이콘과 메뉴를 만든다.
	private void initTrayMenu(){
		Tray tray = display.getSystemTray();
		trayItem = new TrayItem(tray, SWT.NULL);
		trayItem.setToolTipText(TITLE);
		trayItem.setImage(icon);
		
		trayMenu = new Menu(shell, SWT.POP_UP);
		MenuListener listener = new MenuListener();
		trayItem.addListener(SWT.MenuDetect, listener);
		trayItem.addListener(SWT.Selection, listener);
		
		MenuItem mi = new MenuItem(trayMenu, SWT.PUSH);
		mi.setText(CONSOLE);
		mi.addListener(SWT.Selection, listener);
		mi = new MenuItem(trayMenu, SWT.PUSH);
		mi.setText(LOAD);
		mi.addListener(SWT.Selection, listener);
		/*
		mi = new MenuItem(trayMenu, SWT.CASCADE);
		mi.setText(MODULES);
			Menu modules = new Menu(shell, SWT.DROP_DOWN);
			mi.setMenu(modules);
			mi = new MenuItem(modules, SWT.PUSH);
			mi.setText(LOAD);
			mi.addListener(SWT.Selection, listener);
			new MenuItem(modules, SWT.SEPARATOR);
		*/
		mi = new MenuItem(trayMenu, SWT.PUSH);
		mi.setText(EXIT);
		mi.addListener(SWT.Selection, listener);
	}
	
	//프로그램을 종료한다.
	public static final void exit(){
		session.save();
		trayItem.setVisible(false);
		System.exit(0);
	}
	
	//생성된 SWT 위젯을 가져와 모듈 관리용 이벤트 리스너를 등록한 후 연다.
	//리턴하지 않는 일회성 스크립트는 세션에서 관리할 모듈로 등록하지 않는다.
	public static void openWidget(final ZetaModule module){
		if(module.obj instanceof Shell){
			Shell s = (Shell) module.obj;
			s.addListener(SWT.Close, new Listener(){
				public void handleEvent(Event event) {
					session.removeModule(module.fileName);
				}
			});
			s.open();
			session.addModule(module.fileName);
		}
	}
	
	//FileDialog를 열어 실행할 스크립트를 불러온다.
	public static void openScript(){
		FileDialog fd = new FileDialog(shell);
		fd.setFilterExtensions(new String[]{"*.bsh;*.xml"});
		String s = fd.open();
		
		if(s != null)
			interpreter.interpret(s);
		
	}
	
	//파일명들로부터 스크립트를 불러온다.
	public void loadScripts(String[] modules){
		for(String m : modules)
			interpreter.interpret(m);
		
	}
	
	//콘솔을 보이게 하거나 감춘다.
	public static void consoleShowHide(){
		if(shell.isVisible())
			shell.setVisible(false);
		else
			shell.open();
	}
	
	//콘솔을 제외한 모든 셸을 보이거나 감춘다.
	public static void shellsShowHide(){
		Shell[] shells = display.getShells ();
		for (int i = 0; i < shells.length; ++i)
		{
			//콘솔은 영향을 받지 않음
			if(shells[i] != shell){
				if(shellsVisible){
					if (!shells[i].isDisposed ()) shells[i].open ();
				}else{
					shells[i].setVisible(false);
				}
			}
		}
		shellsVisible = shellsVisible ? false : true;
	}
	
	class MenuListener implements Listener{
		public void handleEvent(Event event) {
			if(event.widget instanceof MenuItem){
				String cmd = ((MenuItem) event.widget).getText();
				if(cmd == Zeta.CONSOLE)
					Zeta.consoleShowHide();
				else if(cmd == Zeta.LOAD)
					Zeta.openScript();
				else if(cmd == Zeta.EXIT)
					Zeta.exit();
				
			}else if(event.widget instanceof TrayItem){
				if(event.type == SWT.MenuDetect)
					//팝업 메뉴를 연다.
					Zeta.trayMenu.setVisible(true);
				else if(event.type == SWT.Selection)
					Zeta.shellsShowHide();
			}
		}
	}
	
	class PromptListener implements KeyListener{
		Text prompt;
		
		public PromptListener(Text prompt){
			this.prompt = prompt;
		}
		
		public void keyPressed(KeyEvent e){
			//Enter 가 입력되면 명령어를 실행.
			if(e.keyCode == SWT.CR){
				String s = prompt.getText();
				if(!s.equals("")){
					logger.println(">>>" + s);
					interpreter.bshInterpret(s);
					prompt.setText("");
				}
			}
		}
		public void keyReleased(KeyEvent e){}
	}
	
	public static void main(String args[]){
		//서브 스크립트 파일들의 인코딩 문제를 해결.
		System.setProperty("file.encoding", "utf8");
		
		Zeta zb = new Zeta();
		
		while(!zb.shell.isDisposed()){
			if(!zb.display.readAndDispatch())
				zb.display.sleep();
		}
		zb.display.dispose();
	}
}

//주어진 Text 위젯에 입력받은 로그를 출력하는 역할을 한다.
class SimpleLogger{
	public Text viewer;
	
	public SimpleLogger(Text viewer){
		viewer.setEditable(false);
		this.viewer = viewer;
	}
	
	//예외를 받아 viewer에 출력한다.
	public void printStackTrace(Exception e){
		StringBuilder sb = new StringBuilder();
		sb.append(e.toString());
		StackTraceElement stes[] = e.getStackTrace();
		for(StackTraceElement ste : stes){
			sb.append("\tat ");
			sb.append(ste.toString());
			sb.append("\n");
		}
		sb.append("\n\n");
		print(sb.toString());
	}
	
	//주어진 문자열을 viewer에 출력한다.
	public void print(String s){
		viewer.append(s);
	}
	
	//newline 문자를 출력한다.
	public void println(){
		viewer.append("\n");
	}
	
	//주어진 문자열을 viewer에 출력하고 newlinet 문자를 덧붙인다.
	public void println(String s){
		print(s);
		println();
	}
}
